define('atlassian-nps-plugin/js/nps/survey-dialog', [
    'atlassian-nps-plugin/js/nps/util',
    'atlassian-nps-plugin/js/nps/product',
    'atlassian-nps-plugin/js/amd-shims/ajs',
    'atlassian-nps-plugin/js/amd-shims/templates',
    'jquery'
],
/**
 * Handles the display and behaviour of the NPS survey.
 * Cloud - extra 'page' in survey asking for user to optionally provide name and email
 * Server - does not ask user for name/email
 * @param Util helper functions
 * @param Product adapter for a specific product such as JIRA or Confluence
 * @param AJS
 * @param Templates survey modal dialog templates
 * @param $ jQuery
 * @tainted AJS.bind
 * @tainted AJS.dialog2
 */
function (
    Util,
    Product,
    AJS,
    Templates,
    $
) {
    "use strict";

    var SurveyDialog = {};
    var eventAttrs = {};

    SurveyDialog.previewDialog = function () {
        var $survey = generateTemplate();
        AJS.dialog2($survey).show();

        // Rank selection
        $survey.on('click', "#nps-rank-buttons > button", function (event) {
            var $buttonClicked = $(event.target);
            $survey.find("#nps-rank-buttons > button").removeClass("nps-rank-selected");

            $buttonClicked.addClass("nps-rank-selected");
            $buttonClicked.blur();
        });

        // "Cancel" button
        var $cancelButton = $survey.find("#nps-cancel-button");
        $cancelButton.on("click", function () {
            AJS.dialog2("#nps-survey-dialog").remove();
        });
    };

    SurveyDialog.showDialog = function () {
        var $survey = generateTemplate();
        AJS.dialog2($survey).show();
        $(".aui-button").blur();

        // Rank selection
        $survey.on('click', "#nps-rank-buttons > button", function (event) {
            var $buttonClicked = $(event.target);
            $survey.find("#nps-rank-buttons > button").removeClass("nps-rank-selected");

            $buttonClicked.addClass("nps-rank-selected");
            $buttonClicked.blur();

            SurveyDialog.rankSelected = parseInt($buttonClicked.text(), 10);

            _updateSendButton();
        });

        // Role selection
        var $roleListControl = $survey.find("#nps-role");
        $roleListControl.on("change", function () {
            SurveyDialog.roleSelected = this.value;
            _updateSendButton();
        });

        // "Send survey" button
        var $sendSurveyButton = $survey.find("#nps-send-button");
        $sendSurveyButton.on("click", function () {
            eventAttrs.rank = SurveyDialog.rankSelected;
            eventAttrs.role = SurveyDialog.roleSelected;
            var comment = $("#nps-comment").val();
            if (comment) {
                eventAttrs.comment = comment;
            }

            if (Product.isServerMode()) {
                AJS.dialog2("#nps-survey-dialog").hide();
                AJS.trigger("/nps/surveyCompleted", eventAttrs);
            } else {
                _navigateToContactPage();
            }
        });

        // "Cancel" button
        var $cancelButton = $survey.find("#nps-cancel-button");
        $cancelButton.on("click", function () {
            AJS.dialog2("#nps-survey-dialog").hide();
            AJS.trigger("/nps/cancelSurvey", "click");
        });
    };

    function generateTemplate() {
        return $(Templates.survey({
            productName: Product.getProductName(),
            roles: _getRoles()
        }));
    }

    function _getRoles() {
        var prefix = [
            {
                value: "",
                text: AJS.I18n.getText("nps.survey.question.role.choose"),
                disabled: true,
                selected: true
            }
        ];

        var roles = [
            {
                value: "software-engineer",
                text: AJS.I18n.getText("nps.survey.inline.dialog.roles.software.engineer")
            },
            {
                value: "product-manager",
                text: AJS.I18n.getText("nps.survey.inline.dialog.roles.product.manager")
            },
            {
                value: "qa",
                text: AJS.I18n.getText("nps.survey.inline.dialog.roles.qa")
            },
            {
                value: "design",
                text: AJS.I18n.getText("nps.survey.inline.dialog.roles.design")
            },
            {
                value: "management",
                text: AJS.I18n.getText("nps.survey.inline.dialog.roles.management")
            },
            {
                value: "sys-admin",
                text: AJS.I18n.getText("nps.survey.inline.dialog.roles.sys.admin")
            }
        ];

        var suffix = [
            {
                value: "other",
                text: AJS.I18n.getText("nps.survey.inline.dialog.roles.other")
            }
        ];

        return prefix.concat(Util.kfyShuffle(roles)).concat(suffix);
    }

    function _navigateToContactPage() {
        AJS.dialog2("#nps-survey-dialog").hide();
        var $contact = $(Templates.contact({
            productName: Product.getProductName()
        }));
        AJS.dialog2($contact).show();

        // "Done" button
        $("#nps-done-button").on("click", function () {
            var name = $('#nps-contact-name').val();
            if (name) {
                eventAttrs.name = name.trim();
            }
            var email = $('#nps-contact-email').val();
            if (email) {
                eventAttrs.email = email.trim();
            }

            AJS.dialog2("#nps-contact-dialog").hide();
            AJS.trigger("/nps/surveyCompleted", eventAttrs);
        });

        $("#nps-contact-name").focus();
    }

    function _updateSendButton() {
        var canSend = !!SurveyDialog.roleSelected && !isNaN(SurveyDialog.rankSelected);
        var sendButton = $("#nps-send-button");
        sendButton.attr("aria-disabled", canSend ? "false" : "true");
        if (canSend) {
            sendButton.removeAttr("disabled");
        } else {
            sendButton.attr("disabled", "");
        }
    }

    return SurveyDialog;
});
