define('atlassian-nps-plugin/js/nps/session-manager', [
    'atlassian-nps-plugin/js/nps/client-storage'
],
/**
 * Monitors a users session. To avoid interrupting the user's work, the NPS waits for a lull in user
 * activity and then presents the survey prompt. The users session restarts every 60 minutes without a
 * page load. Any page load more than 10 minutes after a session start and more than 5 minutes after
 * using the editor is considered a good time to prompt the user. The session state is kept in localStorage
 * @param ClientStorage client-side storage mechanism name spaced by user for retrieving scheduling data
 * @returns {object} Session manager implementation
 */
function (
    ClientStorage
) {
    "use strict";

    var millisecondsInMinute = 1000 * 60;   // 1000 milliseconds per second * 60 seconds
    var minimumIdleMinutes = 5 * millisecondsInMinute;
    var minutesToSessionTimeout = 60 * millisecondsInMinute;
    var minimumSessionMinutes = 10 * millisecondsInMinute;

    /**
     * If sessionStart does not exist yet, or if 60 minutes have elapsed since the last page load,
     * reset the sessionStart time
     */
    function updateSessionStart() {
        if (!ClientStorage.containsKey("sessionStart")) {
            ClientStorage.set("sessionStart", Date.now());
        } else {
            if ( (Date.now() - ClientStorage.getNumber("lastActive") ) > minutesToSessionTimeout) {
                ClientStorage.set("sessionStart", Date.now());
            }
        }
    }

    /**
     * idleStart does not exist on first load and is removed anytime the user invokes the
     * Confluence editor
     */
    function updateIdleStart() {
        if (!ClientStorage.containsKey("idleStart")) {
            ClientStorage.set("idleStart", Date.now());
        }
    }

    return {
        update: function () {
            updateIdleStart();
            updateSessionStart();
            ClientStorage.set("lastActive", Date.now());
        },

        isSurveyTime: function () {
            return this.isInSession() && this.isIdle();
        },

        isIdle: function () {
            var timeSinceIdleStart = Date.now() - ClientStorage.getNumber("idleStart");
            return timeSinceIdleStart > minimumIdleMinutes;
        },

        isInSession: function () {
            var timeSinceSessionStart = Date.now() - ClientStorage.getNumber("sessionStart");
            return timeSinceSessionStart > minimumSessionMinutes;
        },

        reset: function () {
            ClientStorage.remove("lastActive");
            ClientStorage.remove("sessionStart");
            ClientStorage.remove("idleStart");
        }
    };
});
