define('atlassian-nps-plugin/js/atlassian-nps-plugin', [
    'atlassian-nps-plugin/js/amd-shims/ajs',
    'atlassian-nps-plugin/js/amd-shims/wrm',
    'atlassian-nps-plugin/js/nps/client-storage',
    'atlassian-nps-plugin/js/nps/product',
    'atlassian-nps-plugin/js/nps/session-manager',
    'atlassian-nps-plugin/js/nps/submission',
    'atlassian-nps-plugin/js/nps/util',
    'jquery'
],
/**
 * Main module for the Net Promoter Score plugin. It determines if it is time to show a NPS survey and executes
 * the code needed to show the survey if it is time
 * @param AJS
 * @param WRM Web Resource Manager for async requiring of web resources
 * @param ClientStorage client-side storage mechanism name spaced by user for retrieving scheduling data
 * @param Product adapter for a specific product such as JIRA or Confluence
 * @param Session calculates and manages the fine-grained display conditions
 * @param Submission handles the firing of analytics events
 * @param $ jQuery
 */
function(
    AJS,
    WRM,
    ClientStorage,
    Product,
    Session,
    Submission,
    Util,
    $
) {
    "use strict";

    var ConfigManager;
    var THIRTY_DAYS = 1000 * 60 * 60 * 24 * 30; // min time since last survey date before a new one is shown

    function init(configManager) {
        ConfigManager = configManager;
        var displayCount = ClientStorage.getNumber("displayCount");
        Product.setBindings();
        Session.update();

        if (ClientStorage.get("optedOut") === "true") { // user has opted out
            // Migrate this value to the server so that on subsequent page loads, the NPS
            // surveys web resources don't get loaded (NpsNotOptedOutUrlReadingCondition fails)
            ConfigManager.setConfig("optedOut", true);
        } else if (displayCount !== 0) { // survey is currently being shown
            if (displayCount < 3) {
                ConfigManager.fetchConfig().then(furtherChecks);
            } else {
                ignoreNotification();
            }
        } else if (Date.now() - ClientStorage.get("lastSurveyDate") >= THIRTY_DAYS) { // might be survey time!
            if (ClientStorage.getProductValue("nextSurveyDate") === null) { // no value in localstorage, check server
                ConfigManager.fetchConfig().then(furtherChecks);
            } else if (ClientStorage.getProductValue("nextSurveyDate") - Date.now() < 0) { // past the next survey date, confirm with server
                ConfigManager.fetchConfig().then(furtherChecks);
            }
        }
    }

    function furtherChecks() {
        checkDisplayConditions().then(showNotification);
    }

    function checkDisplayConditions() {
        var timeout = 30000; // 30 second timeout
        var dfd = $.Deferred();

        if (AJS.DarkFeatures && AJS.DarkFeatures.isEnabled("nps.survey.inline.dialog")) {
            timeout = 5000;
        }

        if (localStorage && localStorage.getItem('nps-testing') === "true") {
            timeout = 0; // for WD testing, don't use a timeout
        }

        setTimeout(function() {
            var noShowingDialog = !Util.hasShowingDialog();
            if (noShowingDialog && Session.isSurveyTime() && ConfigManager.getCachedData().timeToNextSurvey != null && ConfigManager.getCachedData().timeToNextSurvey <= 0) {
                dfd.resolve();
            }
        }, timeout);

        return dfd.promise();
    }

    function showNotification() {
        if (Product.allowDisplayNotification() === true) {
            WRM.require('wr!com.atlassian.plugins.atlassian-nps-plugin:nps-specific-resources-async', function () {
                require(['atlassian-nps-plugin/js/nps/main'], function (npsSurvey) {
                    npsSurvey.launch(ConfigManager);
                    ClientStorage.increment("displayCount");
                    Submission.showNotification({
                        displayCount: ClientStorage.getNumber("displayCount"),
                        dismissedCount: parseInt(ConfigManager.getCachedData().dismissedCount, 10)
                    });
                    ClientStorage.set("lastSurveyDate", Date.now());
                });
            });
        }
    }

    function ignoreNotification() {
        WRM.require('wr!com.atlassian.plugins.atlassian-nps-plugin:nps-submission-resources', function () {
            require(['atlassian-nps-plugin/js/nps/submission'], function (Submission) {
                ClientStorage.remove("displayCount");
                Submission.ignoreSurvey();
                ConfigManager.scheduleNextSurveyDate();
                Session.reset();
            });
        });
    }

    return {
        init: init
    };
});
