(function ($, JBAM, Calendar) {

    var events = require('jira/util/events');

    if (typeof JBAM.CONFIG === "undefined") {
        JBAM.CONFIG = {
            $planSelect: $(),
            $buildTypes: $(),
            buildResultsContainerId: "bamboo-build-results",
            stagesContainerId: "bamboo-plan-stages",
            variablesContainerId: "bamboo-override-variables",
            initialised: false,
            updateTimeout: null,
            init: function () {
                if (JBAM.CONFIG.initialised) {
                    return;
                }

                $(document).bind("dialogContentReady", function (e) {
                    // Set up dialog contents when it opens
                    var buildResultsContainerId = JBAM.CONFIG.buildResultsContainerId,
                        stagesContainerId = JBAM.CONFIG.stagesContainerId,
                        variablesContainerId = JBAM.CONFIG.variablesContainerId;

                    JBAM.CONFIG.$planSelect = $("#bamboo-plan").change(JBAM.CONFIG.onPlanChange);

                    JBAM.CONFIG.BuildResults.init({
                        container: {
                            id: buildResultsContainerId,
                            insertAfter: JBAM.CONFIG.$planSelect.closest(".field-group")
                        }
                    });
                    JBAM.CONFIG.Stages.init({
                        container: {
                            id: stagesContainerId,
                            insertAfter: "#" + buildResultsContainerId
                        }
                    });
                    JBAM.CONFIG.Variables.init({
                        container: {
                            id: variablesContainerId,
                            insertAfter: "#" + stagesContainerId
                        }
                    });

                    JBAM.CONFIG.$buildTypes = $("input:radio[name='buildType']").change(JBAM.CONFIG.onBuildTypeChange).filter(":checked").change().end();
                });
                JBAM.CONFIG.initialised = true;
            }
        };

        /**
         * @class Spinner
         * @namespace JBAM.CONFIG
         * @constructor
         * @param container {String}, {DOMElement} or {jQuery Object} to append the spinner to
         * @param beforeShow {Function} to execute before spinner is shown
         */
        JBAM.CONFIG.Spinner = function (container, beforeShow) {
            this.beforeShow = beforeShow;
            this.add = function () {
                return $(bamboo.components.icon({type: "loading"})).appendTo(container);
            };
        };
        JBAM.CONFIG.Spinner.prototype.show = function () {
            var $spinner = this.$spinner;

            this.beforeShow();
            if (!$spinner || ($spinner && (!$spinner.length || !$spinner.parent().length))) { // checks if spinner exists and is in DOM
                this.$spinner = this.add();
            } else {
                $spinner.show();
            }
            return this;
        };
        JBAM.CONFIG.Spinner.prototype.hide = function () {
            if (this.$spinner) {
                this.$spinner.hide();
            }
            return this;
        };
    }

    JBAM.CONFIG.onPlanChange = function (e) {
        var isExistingBuild = $("#existing-build").is(":checked"),
            $select = $(this),
            $planSelectAndBuildTypes = $select.add(JBAM.CONFIG.$buildTypes).attr("disabled", true),
            $selectedPlan = $select.find("option:selected"),
            $releaseButton = $("#release"),
            selectedBuildResult = (isExistingBuild ? $selectedPlan.attr("data-selected-build-result") : null),
            selectedStages = (isExistingBuild ? null : $selectedPlan.attr("data-selected-stages")),
            selectedVariables = (isExistingBuild ? null : $selectedPlan.attr("data-selected-variables")),
            requestUrl, requestData, successCallback;

        // If we don't have any plans, return.
        if (!$select.val()) {
            return;
        }

        $releaseButton.attr("disabled", "disabled");

        if (selectedVariables) {
            selectedVariables = $.parseJSON(selectedVariables);
        }

        JBAM.CONFIG.BuildResults.updateSelectedBuildResult(selectedBuildResult);
        JBAM.CONFIG.Stages.updateSelectedStages(selectedStages);
        JBAM.CONFIG.Variables.updateSelectedVariables(selectedVariables);
        JBAM.CONFIG.Variables.hide();

        if (isExistingBuild) {
            JBAM.CONFIG.Stages.hide();
            JBAM.CONFIG.BuildResults.showSpinner();

            requestUrl = JBAM.restURL + "history/" + JBAM.jiraProjectKey + "/" + $select.val() + "/5";
            requestData = {continuable: true, buildstate: "successful"};
            successCallback = function (data) {
                if (data.results.result && data.results.result.length) {
                    $releaseButton.removeAttr("disabled");

                    JBAM.CONFIG.BuildResults.show(data.results.result);
                } else {
                    JBAM.CONFIG.BuildResults.showNoResultsMsg();
                }
                $planSelectAndBuildTypes.attr("disabled", false);
            };
        } else {
            $releaseButton.removeAttr("disabled");

            JBAM.CONFIG.BuildResults.hide();

            JBAM.CONFIG.Stages.showSpinner();

            requestUrl = JBAM.restURL + "plan/" + JBAM.jiraProjectKey + "/" + $select.val();
            successCallback = function (data) {
                if (data.stages.stage && data.stages.stage.length) {
                    JBAM.CONFIG.Stages.show(data.stages.stage);
                } else {
                    JBAM.CONFIG.Stages.showNoStagesMsg();
                }
                if (data.variableContext.variable && data.variableContext.variable.length) {
                    JBAM.CONFIG.Variables.show(data.variableContext.variable);
                }
                $planSelectAndBuildTypes.attr("disabled", false);
            };
        }

        $.ajax({
            url: requestUrl,
            dataType: "json",
            data: requestData,
            beforeSend: function () {
                $select.attr("disabled", "true");
            },
            success: successCallback,
            error: function (xhr, textStatus) {
                var json, title, body, xhrResponse,
                    $newDialogContent = $('<div class="form-body"></div><div class="form-footer"><a class="cancel">Cancel</a></div>');

                $newDialogContent.find(".cancel").click(function (e) {
                    if (JBAM.CONFIG.FormDialog && JBAM.CONFIG.FormDialog.hide) {
                        JBAM.CONFIG.FormDialog.hide();
                    }
                });

                try {
                    xhrResponse = xhr.responseText || xhr.response;
                    if (xhrResponse) {
                        json = $.parseJSON(xhrResponse);
                        title = json.message;
                        body = (json.messageBody ? '<p>' + json.messageBody + '</p>' : null);
                        if (json.oauthCallback) {
                            $newDialogContent.filter(".content-footer").prepend('<a href="' + json.oauthCallback + '&amp;redirectUrl=' + encodeURIComponent(JBAM.baseURL) + '">Login &amp; Approve</a> | ');
                        }
                    } else {
                        throw "noJSONResponse";
                    }
                }
                catch (e) {
                    title = "An error occurred while trying to retrieve the build information.";
                }

                AJS.messages.warning($newDialogContent.filter(".form-body"), {
                    title: title,
                    body: body,
                    closeable: false
                });

                $select.closest("form").replaceWith($newDialogContent);

            },
            ajaxComplete: function () {
                $select.attr("disabled", "false");
            }
        });

    };

    JBAM.CONFIG.onBuildTypeChange = function (e) {
        var $radio = $(this),
            $buildTypeFieldset = $radio.closest("fieldset.group"),
            $releaseBuildFields = $buildTypeFieldset.nextAll(".field-group,.group"),
            $buildResultsFieldset = $("#" + JBAM.CONFIG.buildResultsContainerId);

        $("#release").removeAttr("disabled");

        switch ($radio.val()) {
            case "no-build":
                $releaseBuildFields.hide();
                $buildTypeFieldset.css({ // mimick AUI's form.aui fieldset:last-child so we don't get huge amounts of unnecessary whitespace
                    "margin-bottom": 0,
                    "padding-bottom": 0
                });
                break;
            case "new-build":
                $releaseBuildFields.show();
                $buildResultsFieldset.hide();
                $buildTypeFieldset.css({
                    "margin-bottom": "",
                    "padding-bottom": ""
                });
                JBAM.CONFIG.$planSelect.change();
                break;
            case "existing-build":
                $releaseBuildFields.show();
                $buildTypeFieldset.css({
                    "margin-bottom": "",
                    "padding-bottom": ""
                });
                JBAM.CONFIG.$planSelect.change();
                break;
        }
    };

    JBAM.CONFIG.Stages = (function () {
        var defaults = {
                container: {
                    id: null,
                    insertAfter: null,
                    i18n: {
                        legend: "Stages"
                    }
                },
                selectedStages: null
            },
            options,
            $container,
            containerInitialContent,
            createContainer = function () {
                $container = $("<fieldset/>", {
                    id: options.container.id,
                    "class": "group",
                    html: containerInitialContent
                })
                    .insertAfter(options.container.insertAfter)
                    .delegate("input:checkbox", "click", checkboxClick);
            },
            createStagesList = function (stages) {
                var hasManualStage = false,
                    $preselectedStage,
                    i, ii;

                for (i = 0, ii = stages.length; i < ii; i++) {
                    var stage = stages[i],
                        alreadyRun = !!(stage.state && stage.state != "Unknown"),
                        checked = !!(stage.manual),
                        disabled = !!(!stage.manual || alreadyRun),
                        $stage,
                        $input,
                        inputId = "stage-" + i,
                        $label;

                    if (!hasManualStage && stage.manual && !alreadyRun) {
                        hasManualStage = true;
                    }

                    $input = $('<input type="checkbox" />').attr({
                        name: "selectedStages",
                        id: inputId,
                        "class": "checkbox",
                        checked: checked,
                        disabled: disabled
                    }).val(stage.name);

                    $label = $('<label />', {
                        text: stage.name,
                        "for": inputId
                    });

                    $stage = $("<div/>", {
                        "class": "checkbox" + (stage.manual ? (alreadyRun ? " already-run" : " manual") : "") + (!hasManualStage ? " selected" : "")
                    })
                        .append($input)
                        .append($label)
                        .appendTo($container);

                    if (!alreadyRun && checked) {
                        $preselectedStage = $stage;
                    }
                }
                if ($preselectedStage) {
                    $preselectedStage.find("input:checkbox").trigger("click", true);
                }
            },
            removeStagesList = function () {
                $container.find("div.checkbox,p").remove();
            },
            checkboxClick = function (e, preventDefault) {
                if (preventDefault) {
                    e.preventDefault();
                }
                var $checkbox = $(this),
                    checked = $checkbox.is(":checked"),
                    prevManualFound = false;

                $checkbox.closest("div.checkbox").toggleClass("selected", checked).prevAll("div.checkbox").each(function (i) {
                    var $checkboxContainer = $(this),
                        $checkbox = $checkboxContainer.children("input:checkbox"),
                        isManual = $checkboxContainer.hasClass("manual");

                    if (checked) {
                        $checkboxContainer.addClass("selected");
                        if (isManual) {
                            $checkbox.attr("checked", "checked").attr("disabled", "disabled");
                        }
                    } else if (isManual && !prevManualFound) {
                        $checkbox.removeAttr("disabled");
                        prevManualFound = true;
                    }
                }).end().nextAll("div.checkbox").each(function (i) {
                    var $checkboxContainer = $(this);

                    if ($checkboxContainer.hasClass("manual")) {
                        return false; // breaks the loop
                    } else {
                        $checkboxContainer.toggleClass("selected", checked);
                    }
                });
            },
            spinner;
        return {
            init: function (opts) {
                options = $.extend(true, defaults, opts);
                containerInitialContent = '<legend><span>' + options.container.i18n.legend + '</span></legend>';
                createContainer();
                spinner = new JBAM.CONFIG.Spinner($container, removeStagesList);
            },
            show: function (stages) {
                $container.html(containerInitialContent).show();
                createStagesList(stages);
            },
            hide: function () {
                $container.hide().html(containerInitialContent);
            },
            showNoStagesMsg: function () {
                spinner.hide();
                $container.append($("<p/>", {text: "No stages found."}));
            },
            showSpinner: function () {
                spinner.show();
            },
            updateSelectedStages: function (selectedStages) {
                options.selectedStages = selectedStages;
            }
        }
    })();

    JBAM.CONFIG.Variables = (function () {
        var defaults = {
                container: {
                    id: null,
                    insertAfter: null,
                    i18n: {
                        legend: "Build Variables"
                    }
                },
                selectedVariables: null
            },
            options,
            $container,
            $variableSelect,
            containerInitialContent,
            createContainer = function () {
                $container = $("<fieldset/>", {
                    id: options.container.id,
                    "class": "group",
                    html: containerInitialContent
                })
                    .insertAfter(options.container.insertAfter)
                    .delegate("#override-another", "click", addVariable)
                    .delegate(".bamboo-icon-variable-delete", "click", removeVariable)
                    .delegate("select", "change", keyChange);
            },
            createVariableSelect = function (variables) {
                var currentVariableType,
                    $currentOptGroup,
                    i, ii;

                $variableSelect = $('<select/>', {"class": "select"});

                for (i = 0, ii = variables.length; i < ii; i++) {
                    var variable = variables[i];

                    if (currentVariableType != variable.variableType) {
                        currentVariableType = variable.variableType;
                        $currentOptGroup = $('<optgroup/>', {label: currentVariableType}).appendTo($variableSelect)
                    }
                    $('<option/>', {
                        text: variable.key,
                        data: {"server-value": variable.value}
                    }).appendTo($currentOptGroup);
                }
            },
            addVariable = function () {
                var $variableDD = $variableSelect.clone(true),
                    $fieldGroup,
                    $selectedVariable,
                    $input,
                    inputValue = "";

                if (arguments.length == 2) {
                    $selectedVariable = $variableDD.find("option[text='" + arguments[0] + "']");
                    inputValue = arguments[1];
                } else {
                    $selectedVariable = $variableDD.find("option:first");
                }

                $selectedVariable.attr("selected", "selected");

                $input = $('<input type="text" />').attr({
                    name: 'variable_' + $selectedVariable.val(),
                    placeholder: $selectedVariable.data("server-value"),
                    "class": "text medium-field"
                }).val(inputValue);

                $fieldGroup = $('<div/>', {
                    "class": "field-group"
                })
                    .append($variableDD)
                    .append($input)
                    .append(bamboo.components.icon({type: "variable-delete"}))
                    .insertBefore($container.children("a"));
            },
            removeVariable = function (e) {
                $(this).closest(".field-group").remove();
            },
            keyChange = function (e) {
                var $variableDD = $(this),
                    $selectedVariable = $variableDD.find("option:selected");

                $variableDD.nextAll(".text").attr({
                    name: "variable_" + $selectedVariable.val(),
                    "placeholder": $selectedVariable.data("server-value")
                });
            };
        return {
            init: function (opts) {
                options = $.extend(true, defaults, opts);
                containerInitialContent = '<legend><span>' + options.container.i18n.legend + '</span></legend><a id="override-another" class="aui-button aui-button-link">Override variable<span class="icon icon-edit-sml"></span></a>';
                createContainer();
            },
            show: function (variables) {
                createVariableSelect(variables);
                if (options.selectedVariables && options.selectedVariables.variables) {
                    $.each(options.selectedVariables.variables, addVariable); // adds a variable override for every preselected variable
                }
                $container.show();
            },
            hide: function () {
                $container.hide().html(containerInitialContent);
            },
            updateSelectedVariables: function (selectedVariables) {
                options.selectedVariables = selectedVariables;
            }
        }
    })();

    JBAM.CONFIG.BuildResults = (function () {
        var defaults = {
                container: {
                    id: null,
                    insertAfter: null,
                    i18n: {
                        legend: "Existing build"
                    }
                },
                selectedBuildResult: null
            },
            options,
            $container,
            containerInitialContent,
            createContainer = function () {
                $container = $("<fieldset/>", {
                    id: options.container.id,
                    "class": "group",
                    html: containerInitialContent
                })
                    .insertAfter(options.container.insertAfter)
                    .delegate("input:radio", "change", radioChange)
                    .delegate("a", "click", function (e) {
                        e.preventDefault();
                        window.open(this.href);
                    });
            },
            addResult = function (buildResult) {
                var $field,
                    $input,
                    inputId = "build-result-" + buildResult.number,
                    $label,
                    status = (buildResult.lifeCycleState == "Finished" ? (buildResult.continuable ? "SuccessfulPartial" : buildResult.state) : buildResult.lifeCycleState),
                    buildResultChecked = (buildResult.key == options.selectedBuildResult),
                    buildStages = (buildResult.stages.stage && buildResult.stages.stage.length ? buildResult.stages.stage : []);

                $input = $('<input type="radio" />').attr({
                        name: "buildResult",
                        id: inputId,
                        "class": "radio",
                        checked: buildResultChecked
                    })
                    .val(buildResult.key)
                    .data("stages", buildStages);

                $label = $('<label />', {
                    html: '<a href="' + JBAM.bambooBaseURL + '/browse/' + buildResult.key + '" class="' + status + '">#' + buildResult.number + '</a><span class="trigger-reason">' + JBAM.replaceWithBambooIcon(buildResult.buildReason) + '</span>',
                    "for": inputId
                })
                    .prepend(bamboo.components.icon({type: status}));

                $field = $('<div/>', {
                    "class": "radio"
                })
                    .append($input)
                    .append($label)
                    .appendTo($container);
                if (buildResultChecked) {
                    showStages(buildStages);
                }
            },
            showStages = function (stages) {
                if (stages && stages.length) {
                    JBAM.CONFIG.Stages.show(stages);
                } else {
                    JBAM.CONFIG.Stages.showNoStagesMsg();
                }
            },
            removeBuildResultsList = function () {
                $container.find("div.radio,p").remove();
            },
            radioChange = function (e) {
                var stages = $(this).data("stages");

                showStages(stages);
            },
            spinner;
        return {
            init: function (opts) {
                options = $.extend(true, defaults, opts);
                containerInitialContent = '<legend><span>' + options.container.i18n.legend + '</span></legend>';
                createContainer();
                spinner = new JBAM.CONFIG.Spinner($container, removeBuildResultsList);
            },
            show: function (buildResults) {
                $container.html(containerInitialContent).show();

                if (buildResults.length > 0 && !options.selectedBuildResult) {
                    options.selectedBuildResult = buildResults[0].key;
                }

                for (var i = 0, ii = buildResults.length; i < ii; i++) {
                    addResult(buildResults[i]);
                }
            },
            hide: function () {
                $container.hide().html(containerInitialContent);
            },
            showNoResultsMsg: function () {
                spinner.hide();
                $container.append($("<p/>", {text: "No builds found with unbuilt manual stages."})).show();
            },
            showSpinner: function () {
                spinner.show();
            },
            updateSelectedBuildResult: function (selectedBuildResult) {
                options.selectedBuildResult = selectedBuildResult;
            }
        }
    })();

    JBAM.flagNotification = function (flagType, title, message) {
        require(['aui/flag'], function (flag) {
            flag({
                type: flagType,
                title: title,
                close: 'manual',
                body: message
            });
        });
    };

    JBAM.getNewUrlObject = function (url, paramsToExclude) {
        var urlParts = URI.parse(url);
        var params = URI.parseQuery(urlParts.query);

        if (paramsToExclude && paramsToExclude.length > 0) {
            for (var i = 0; i < paramsToExclude.length; i++) {
                delete params[paramsToExclude[i]];

                // delete the parameter from the fragment / hash
                if (urlParts.fragment) {
                    var fragmentUri = new URI("?" + urlParts.fragment);
                    fragmentUri.removeQuery(paramsToExclude[i]);
                    urlParts.fragment = fragmentUri.toString().substring(1);
                }
            }
        }

        urlParts.query = URI.buildQuery(params);
        return urlParts;
    };

    JBAM.excludeParamsFromUrl = function (url, paramsToExclude) {
        var urlParts = JBAM.getNewUrlObject(url, paramsToExclude);
        return URI.build(urlParts);
    };

    JBAM.getReloadUrl = function () {
        return JBAM.excludeParamsFromUrl(window.location.href, ['showReleaseDialog']);
    };

    // update window.location.href with url without the parameters 'paramsToExclude' in HTML5 browsers
    JBAM.replaceState = function (url, paramsToExclude) {
        if (window.history.replaceState) {
            var newUrlObj = JBAM.getNewUrlObject(url, paramsToExclude)
            var newUrl = URI.build(newUrlObj);
            window.history.replaceState(newUrlObj, '', newUrl);
        }
    };

    JBAM.CheckReleaseBuildStatus = function (buildResultKey, $duration) {
        $.ajax({
            url: JBAM.restURL + "status/" + JBAM.jiraVersionId + "/" + buildResultKey,
            dataType: "json",
            success: function (json) {
                var result = json.bamboo,
                    status = (result.lifeCycleState == "Finished" ? (result.continuable ? "SuccessfulPartial" : result.state) : result.lifeCycleState),
                    $buildDetails;

                if ((result.lifeCycleState != "InProgress" && result.lifeCycleState != "Queued") || (json.jira && json.jira.forceRefresh)) {
                    AJS.reloadViaWindowLocation(JBAM.getReloadUrl());
                    return;

                } else if (result.lifeCycleState == "InProgress" && result.progress) {
                    if (!$duration) {
                        $buildDetails = $("#build-details");
                        $duration = $(bamboo.components.buildDetail({
                            key: "Duration",
                            keyClass: "duration",
                            value: result.progress.prettyBuildTime + ' &ndash; <span>' + result.progress.prettyTimeRemaining + '</span>'
                        })).appendTo($buildDetails).find("dd");
                        $buildDetails.find(".build + dd > span.bamboo-icon").replaceWith(bamboo.components.icon({type: status}));
                    } else {
                        $duration.html(result.progress.prettyBuildTime + ' &ndash; <span>' + result.progress.prettyTimeRemaining + '</span>');
                    }
                    clearTimeout(JBAM.CONFIG.updateTimeout);
                    JBAM.CONFIG.updateTimeout = setTimeout(function () {
                        JBAM.CheckReleaseBuildStatus(buildResultKey, $duration);
                    }, 5000);
                }
            },
            error: function (xhr, textStatus) {
                var json, title, body, xhrResponse,
                    $buildDetails = $("#build-details").empty();

                try {
                    xhrResponse = xhr.responseText || xhr.response;
                    if (xhrResponse) {
                        json = $.parseJSON(xhrResponse);
                        title = json.message;
                        body = (json.messageBody ? '<p>' + json.messageBody + '</p>' : "");
                        if (json.oauthCallback) {
                            body += '<p><a href="' + json.oauthCallback + '&amp;redirectUrl=' + encodeURIComponent(JBAM.baseURL) + '">Login &amp; Approve</a></p>';
                        }
                    } else {
                        throw "noJSONResponse";
                    }
                }
                catch (e) {
                    title = "An error occurred while trying to retrieve the build information.";
                }

                JBAM.flagNotification('warning', title, body);
            }
        });
    };

    JBAM.ShowBuildReleaseDetails = function (buildResultKey) {
        var $buildDetails = $("#build-details"),
            $buildDetailsLoadingIndicator = $(bamboo.components.icon({type: "loading"}));

        if ($buildDetails.length) {
            $buildDetailsLoadingIndicator.appendTo($buildDetails);
            $.ajax({
                url: JBAM.restURL + "status/" + JBAM.jiraVersionId + "/" + buildResultKey,
                dataType: "json",
                success: function (json) {
                    var result = json.bamboo,
                        status = (result.lifeCycleState == "Finished" ? (result.continuable ? "SuccessfulPartial" : result.state) : result.lifeCycleState),
                        $duration;

                    if (json.jira && json.jira.forceRefresh) {
                        AJS.reloadViaWindowLocation(JBAM.getReloadUrl());
                        return;
                    }
                    $buildDetailsLoadingIndicator.remove();

                    // Build status icon, number and name
                    $buildDetails.append(bamboo.components.buildDetail({
                        key: "Build",
                        keyClass: "build",
                        value: bamboo.components.icon({type: status}) + ' <a class="build-link" href="' + JBAM.bambooBaseURL + '/browse/' + buildResultKey + '" title="' + result.planName + '"><span class="' + status + '">#' + result.number + '</span> ' + result.planName + '</a>'
                    }));

                    if (result.lifeCycleState == "InProgress" || result.lifeCycleState == "Queued") {
                        // Duration
                        if (result.lifeCycleState == "InProgress" && result.progress) {
                            $duration = $(bamboo.components.buildDetail({
                                key: "Duration",
                                keyClass: "duration",
                                value: result.progress.prettyBuildTime + ' &ndash; <span>' + result.progress.prettyTimeRemaining + '</span>'
                            })).appendTo($buildDetails).find("dd");
                        }
                        clearTimeout(JBAM.CONFIG.updateTimeout);
                        JBAM.CONFIG.updateTimeout = setTimeout(function () {
                            JBAM.CheckReleaseBuildStatus(buildResultKey, $duration);
                        }, 5000);
                    } else if (result.lifeCycleState == "Finished") {

                        // Completed date/time
                        if (result.buildCompletedTime) {
                            $buildDetails.append(bamboo.components.buildDetail({
                                key: "Completed",
                                keyClass: "completed",
                                value: '<time datetime="' + result.buildCompletedTime + '">' + result.buildRelativeTime + '</time>'
                            }));
                        }
                    } else if (result.lifeCycleState == "NotBuilt") {
                        // Completed date/time
                        if (result.buildCompletedTime) {
                            $buildDetails.append(bamboo.components.buildDetail({
                                key: "Completed",
                                keyClass: "completed",
                                value: '<time datetime="' + result.buildCompletedTime + '">' + result.buildRelativeTime + '</time>'
                            }));
                        }
                    }

                    // Artifacts
                    if (result.artifacts.artifact && result.artifacts.artifact.length) {
                        $buildDetails.append(bamboo.components.buildDetail({
                            key: "Artifacts",
                            keyClass: "artifacts",
                            value: bamboo.components.artifacts({
                                artifacts: result.artifacts.artifact
                            })
                        }));
                    }
                },
                error: function (xhr, textStatus) {
                    var json, title, body, xhrResponse;

                    $buildDetailsLoadingIndicator.remove();

                    try {
                        xhrResponse = xhr.responseText || xhr.response;
                        if (xhrResponse) {
                            json = $.parseJSON(xhrResponse);
                            title = json.message;
                            body = (json.messageBody ? '<p>' + json.messageBody + '</p>' : "");
                            if (json.oauthCallback) {
                                body += '<p><a href="' + json.oauthCallback + '&amp;redirectUrl=' + encodeURIComponent(JBAM.baseURL) + '">Login &amp; Approve</a></p>';
                            }
                        } else {
                            throw "noJSONResponse";
                        }
                    }
                    catch (e) {
                        title = "An error occurred while trying to retrieve the build information.";
                    }

                    JBAM.flagNotification('warning', title, body);
                }
            });
        }
    };

    JBAM.replaceWithBambooIcon = function (original) {
        return original.replace(/(\b)icon(\b)/g, "$1bamboo-icon$2");
    };

    JBAM.clearReleaseErrors = function () {
        var clearErrorsUrl = JBAM.restURL + "errors/" + JBAM.jiraVersionId;

        $.ajax({
            url: clearErrorsUrl,
            type: 'DELETE'
        }).done(function (data) {
            // Nothing to handle in case of success
            // we are ignoring errors as well
        })
    };

    JBAM.publishAnalyticsForReleaseClicked = function (versionId, overdue, daysRemaining) {
        AJS.trigger('analyticsEvent', {
            name: "jira.project.versions.version-detail.release.clicked",
            data: {
                versionId: versionId,
                overdue: overdue,
                daysRemaining: daysRemaining
            }
        });
    };

    JBAM.publishAnalyticsForReleaseDialogClicked = function (versionId, released) {
        AJS.trigger('analyticsEvent', {
            name: "jira.project.versions.version-detail.release.dialog.clicked",
            data: {
                versionId: versionId,
                released: released
            }
        });
    };

    JBAM.init = function (context) {
        JBAM.CONFIG.init();
        var $releaseButton = $("#runRelease", context);

        if ($releaseButton.length) {
            JBAM.CONFIG.FormDialog = new JIRA.FormDialog({
                trigger: $releaseButton,
                widthClass: "large",
                autoClose: true
            });

            // init the calendar js
            events.bind("dialogContentReady", function ($popup, dialog, id) {
                var currentReleaseDate = AJS.$('#release-form-date-picker input[title="currentReleaseDate"]').val();
                var formattedReleaseDate = AJS.$('#release-form-date-picker input[title="formattedReleaseDate"]').val();
                var firstDay = AJS.$('#release-form-date-picker input[title="firstDay"]').val();
                var currentMillis = AJS.$('#release-form-date-picker input[title="currentMillis"]').val();
                var useISO8601 = AJS.$('#release-form-date-picker input[title="useISO8601"]').val();

                var dateFormat = AJS.$('#release-form-date-picker input[title="dateFormat"]').val();
                var inputField = AJS.$("#release-form-release-date-field")[0];
                Calendar.setup({
                    singleClick: true,
                    align: "Bl",
                    firstDay: firstDay,
                    button: AJS.$("#release-form-release-date-trigger")[0],
                    inputField: inputField,
                    currentMillis: currentMillis,
                    useISO8601WeekNumbers: useISO8601,
                    ifFormat: dateFormat
                });
                if (typeof formattedReleaseDate === 'string' && formattedReleaseDate.length > 0) {
                    inputField.value = formattedReleaseDate;
                }
            });

            // Set the redirect URL upon release success. In IE9, JBAM.init() will be executed twice because JIRA.Events.NEW_CONTENT_ADDED
            // will be raised (unexpectedly) in the page reload so this condition guards against that scenario.
            if ($releaseButton.attr('href').indexOf('redirectUrlOnSuccess') === -1) {
                $releaseButton.attr('href', $releaseButton.attr('href') + '&redirectUrlOnSuccess=' + URI.encode(JBAM.getReloadUrl()));
            }

            if (/showReleaseDialog=true/.test(window.location.href)) {
                // remove 'showReleaseDialog' from browser url in HTML5 browsers
                JBAM.replaceState(window.location.href, ['showReleaseDialog']);
                $releaseButton.click();
            }
        }
    };

    //init on DOM ready.
    $(function () {
        JBAM.init(AJS.$(document));
    });

    //Also init when the tab contents are added via AJAX!
    JIRA.bind(JIRA.Events.NEW_CONTENT_ADDED, function (e, context, reason) {
        if (reason === JIRA.CONTENT_ADDED_REASON.tabUpdated) {
            JBAM.init(context);
        }
    });
}(AJS.$, window.JBAM = (window.JBAM || {}), Calendar));
