AJS.test.require([
    "jira.webresources:legacy-namespacing",
    "com.atlassian.gadgets.dashboard.refimpl.ui:required-libraries",
    "com.atlassian.gadgets.dashboard:dashboard-test"
], function(){
    module("Directory loader test", {
        setup: function() {
            this.modelRetrieveStub = sinon.stub(JIRA.Dashboard.Entities.DirectoryItems.prototype, "retrieve");
            this.modelIsReadyStub = sinon.stub(JIRA.Dashboard.Entities.DirectoryItems.prototype, "isReady");
            this.modelRetrievalStartedStub = sinon.stub(JIRA.Dashboard.Entities.DirectoryItems.prototype, "retrievalStarted");
            this.modelRetrievalInProgressStub = sinon.stub(JIRA.Dashboard.Entities.DirectoryItems.prototype, "retrievalInProgress");
            this.modelRetrievalFailedStub = sinon.stub(JIRA.Dashboard.Entities.DirectoryItems.prototype, "retrievalFailed");
            this.dialogControllerShowDirectoryStub = sinon.stub(JIRA.Dashboard.Controllers.DialogController.prototype, "showDirectory");
            this.dialogControllerRenderDirectoryStub = sinon.stub(JIRA.Dashboard.Controllers.DialogController.prototype, "renderDirectory");
            this.dialogControllerShowLoadingPageStub = sinon.stub(JIRA.Dashboard.Controllers.DialogController.prototype, "showDirectoryLoadingPage");
            this.dialogControllerShowLoadFailurePageStub = sinon.stub(JIRA.Dashboard.Controllers.DialogController.prototype, "showDirectoryLoadFailurePage");
            this.dialogControllerRenderLoadFailurePageStub = sinon.stub(JIRA.Dashboard.Controllers.DialogController.prototype, "renderDirectoryLoadFailurePage");
            this.dialogControllerGetDirectoryControllerStub = sinon.stub(JIRA.Dashboard.Controllers.DialogController.prototype, "getDirectoryController");

            this.dialogControllerGetDirectoryControllerStub.returns(_.extend({
                enableDirectoryAddButtons: function() {},
                disableDirectoryAddButtons: function() {}
            }, Backbone.Events));

            this.$trigger = AJS.$('<a class="add-gadget-link"></a>');
            AJS.$('#qunit-fixture').append(this.$trigger);

            this.controller = new JIRA.Dashboard.Controllers.DirectoryLoader('.add-gadget-link');
        },

        clickTrigger: function() {
            this.$trigger.click();
        },

        teardown: function() {
            this.modelRetrieveStub.restore();
            this.modelIsReadyStub.restore();
            this.modelRetrievalStartedStub.restore();
            this.modelRetrievalInProgressStub.restore();
            this.modelRetrievalFailedStub.restore();
            this.dialogControllerShowDirectoryStub.restore();
            this.dialogControllerRenderDirectoryStub.restore();
            this.dialogControllerShowLoadingPageStub.restore();
            this.dialogControllerShowLoadFailurePageStub.restore();
            this.dialogControllerRenderLoadFailurePageStub.restore();
            this.dialogControllerGetDirectoryControllerStub.restore();

            this.controller.stopListeningToTriggers();
        }
    });

    test("Should show directory when click event is received and items are ready", function(){
        this.modelIsReadyStub.returns(true);

        this.clickTrigger();

        ok(this.dialogControllerShowDirectoryStub.calledOnce);
        ok(this.dialogControllerShowDirectoryStub.calledOn(this.controller.dialogController));
        ok(!this.modelRetrieveStub.called);
    });

    test("Should start retrieval when click event is received and items are not ready and retrieval has not started", function(){
        this.modelIsReadyStub.returns(false);
        this.modelRetrievalStartedStub.returns(false);

        this.clickTrigger();

        ok(this.dialogControllerShowLoadingPageStub.calledOnce);
        ok(this.dialogControllerShowLoadingPageStub.calledOn(this.controller.dialogController));
        ok(!this.dialogControllerShowDirectoryStub.called);
        ok(this.modelRetrieveStub.calledOnce);
        ok(this.modelRetrieveStub.calledOn(this.controller.directoryItems));
    });

    test("Should show directory loading page when click event is received and items are not ready and retrieval has started", function(){
        this.modelIsReadyStub.returns(false);
        this.modelRetrievalStartedStub.returns(true);
        this.modelRetrievalFailedStub.returns(false);

        this.clickTrigger();

        ok(this.dialogControllerShowLoadingPageStub.calledOnce);
        ok(this.dialogControllerShowLoadingPageStub.calledOn(this.controller.dialogController));
        ok(!this.dialogControllerShowDirectoryStub.called);
        ok(!this.modelRetrieveStub.called);
    });

    test("Should show directory load failure page when click event is received and retrieval has failed", function(){
        this.modelIsReadyStub.returns(false);
        this.modelRetrievalStartedStub.returns(true);
        this.modelRetrievalFailedStub.returns(true);

        this.clickTrigger();

        ok(!this.dialogControllerShowLoadingPageStub.called);
        ok(!this.dialogControllerShowDirectoryStub.called);
        ok(!this.modelRetrieveStub.called);
        ok(this.dialogControllerShowLoadFailurePageStub.calledOnce);
        ok(this.dialogControllerShowLoadFailurePageStub.calledOn(this.controller.dialogController));
    });

    test("Should render directory when the first batch of items have returned", function(){
        this.modelIsReadyStub.returns(false);
        this.modelRetrievalStartedStub.returns(false);
        this.clickTrigger();

        this.controller.directoryItems.trigger("retrieve", [1, 3]);

        ok(this.dialogControllerRenderDirectoryStub.calledOnce);
        ok(this.dialogControllerRenderDirectoryStub.calledOn(this.controller.dialogController));
    });

    test("Should not render directory when the first batch of items have returned and no items are returned", function(){
        this.modelIsReadyStub.returns(false);
        this.modelRetrievalStartedStub.returns(false);
        this.modelRetrievalInProgressStub.returns(true);
        this.clickTrigger();

        this.controller.directoryItems.trigger("retrieve", []);

        ok(!this.dialogControllerRenderDirectoryStub.called);
    });

    test("Should render directory when all requests have completed and the most recent one returned no items", function(){
        this.modelIsReadyStub.returns(false);
        this.modelRetrievalStartedStub.returns(false);
        this.modelRetrievalInProgressStub.returns(false);
        this.clickTrigger();

        this.controller.directoryItems.trigger("retrieve", []);

        ok(this.dialogControllerRenderDirectoryStub.calledOnce);
    });

    test("Should stop listening to retrieve event after the first batch of items have returned", function(){
        this.modelIsReadyStub.returns(false);
        this.modelRetrievalStartedStub.returns(false);
        this.clickTrigger();

        this.controller.directoryItems.trigger("retrieve", [1, 3]);
        ok(this.dialogControllerRenderDirectoryStub.calledOnce);
        ok(this.dialogControllerRenderDirectoryStub.calledOn(this.controller.dialogController));

        this.controller.directoryItems.trigger("retrieve", [1,3]);
        ok(this.dialogControllerRenderDirectoryStub.calledOnce);
    });

    test("Should stop listening to retrieve failure event after the first batch of items have returned", function(){
        this.modelIsReadyStub.returns(false);
        this.modelRetrievalStartedStub.returns(false);
        this.modelRetrievalInProgressStub.returns(false);
        this.clickTrigger();

        this.controller.directoryItems.trigger("retrieve", [1,3]);
        this.controller.directoryItems.trigger("retrieveFailure");

        ok(!this.dialogControllerShowLoadFailurePageStub.called);
    });

    test("Should render directory load failure page when both requests fail", function(){
        this.modelIsReadyStub.returns(false);
        this.modelRetrievalStartedStub.returns(false);
        this.modelRetrievalInProgressStub.returns(false);
        this.clickTrigger();

        this.controller.directoryItems.trigger("retrieveFailure");

        ok(this.dialogControllerRenderLoadFailurePageStub.calledOnce);
        ok(this.dialogControllerRenderLoadFailurePageStub.calledOn(this.controller.dialogController));
    });

    test("Should do nothing when one request fail but another is still in progress", function(){
        this.modelIsReadyStub.returns(false);
        this.modelRetrievalStartedStub.returns(false);
        this.modelRetrievalInProgressStub.returns(true);
        this.clickTrigger();

        this.controller.directoryItems.trigger("retrieveFailure");

        ok(!this.dialogControllerShowLoadFailurePageStub.called);
    });

    test("Should stop listening to directory items events when both requests have failed", function(){
        this.modelIsReadyStub.returns(false);
        this.modelRetrievalStartedStub.returns(false);
        this.modelRetrievalInProgressStub.returns(false);
        this.clickTrigger();

        this.controller.directoryItems.trigger("retrieveFailure");
        this.controller.directoryItems.trigger("retrieve", [1, 3]);

        ok(!this.dialogControllerShowDirectoryStub.called);
    });

    test("Should propagate dashboard item add event", function(){
        var dashboardItemAddCallbackSpy = sinon.spy();
        this.controller.on("dashboardItemAdd", dashboardItemAddCallbackSpy);

        this.controller.directoryController.trigger("dashboardItemAdd", "some-url");

        ok(dashboardItemAddCallbackSpy.calledOnce);
        ok(dashboardItemAddCallbackSpy.calledWithExactly("some-url"));
    });

    test("Should delegate disabling directory add buttons to directory controller", function(){
        var spy = sinon.spy(this.controller.directoryController, "disableDirectoryAddButtons");

        this.controller.disableDirectoryAddButtons();

        ok(spy.calledOnce);
        spy.restore();
    });

    test("Should delegate enabling directory add buttons to directory controller", function(){
        var spy = sinon.spy(this.controller.directoryController, "enableDirectoryAddButtons");

        this.controller.enableDirectoryAddButtons();

        ok(spy.calledOnce);
        spy.restore();
    });

    test("Should reset directory items and invoke load again when triggered with directoryReloadRequest", function(){
        var directoryItemsResetStub = sinon.stub(this.controller.directoryItems, "reset");
        var loaderLoadStub = sinon.stub(this.controller, "load");

        this.controller.dialogController.trigger("directoryReloadRequest");

        ok(directoryItemsResetStub.calledOnce);
        ok(loaderLoadStub.calledOnce);

        directoryItemsResetStub.restore();
        loaderLoadStub.restore();
    });
});
