AJS.namespace("JIRA.Dashboard.Entities");

JIRA.Dashboard.Entities.DirectoryItem = Backbone.Model.extend({});

JIRA.Dashboard.Entities.DirectoryItems = Backbone.Collection.extend({
    model: JIRA.Dashboard.Entities.DirectoryItem,

    initialize: function(models, options) {
        this.localItemsEndpointUrl = AJS.contextPath() + "/rest/config/1.0/directoryitems/local.json";
        this.externalItemsEndpointUrl = AJS.contextPath() + "/rest/config/1.0/directoryitems/external.json";
        this.endpointUrls = [this.localItemsEndpointUrl, this.externalItemsEndpointUrl];

        this.retriever = options.retriever;
        this.listenTo(this.retriever, {
            "retrieving": this.beforeRetrieve,
            "retrieve": this.processRetrievedItems,
            "retrieveFailure": this.handleRetrievalError,
            "batchRetrieving": function() {
                this.trigger("batchRetrieving");
            },
            "batchRetrieveComplete": function() {
                this.trigger("batchRetrieveComplete");
            }
        });

        this.categorisedItems = {};
    },

    comparator: "titleLabel",

    retrievalStarted: function() {
        return this.retriever.retrievalStarted();
    },

    retrievalInProgress: function() {
        return this.retriever.retrievalInProgress();
    },

    retrievalCompleted: function() {
        return this.retriever.retrievalCompleted();
    },

    retrievalFailed: function() {
        return this.retriever.retrievalFailed();
    },

    isReady: function() {
        return this.retriever.isReady();
    },

    hasSomeFailure: function() {
        return this.retriever.hasSomeFailure();
    },

    retrieve: function() {
        this.retriever.retrieve(this.endpointUrls);
    },

    beforeRetrieve: function() {
        this.trigger("retrieving");
    },

    processRetrievedItems: function(data) {
        this.set(this.transformResponseData(data), {remove: false});
        this.updateCategorisedItems();
        this.trigger("retrieve", data);
    },

    handleRetrievalError: function(failedEndpointUrl) {
        this.trigger("retrieveFailure", failedEndpointUrl);
    },

    updateCategorisedItems: function() {
        this.categorisedItems = {};
        this.categorisedItems[JIRA.Dashboard.Directory.Constants.getDefaultCategoryName()] = [];

        _.each(this.models, function(model){
            this.categorisedItems[JIRA.Dashboard.Directory.Constants.getDefaultCategoryName()].push(model);

            _.each(model.get("categories"), function(category){
                this.categorisedItems[category] = this.categorisedItems[category] || [];
                this.categorisedItems[category].push(model);
            }, this);
        }, this);
    },

    getCategorisedItems: function() {
        return this.categorisedItems;
    },

    reset: function() {
        this.set([]);
        this.categorisedItems = {};
        this.retriever.reset();
        this.trigger("reset");
    },

    retryFailedEndpoints: function() {
        this.retriever.retryFailedEndpoints();
    },

    transformResponseData: function(data) {
        return _.map(data, function(item) {
            item.isLocal = false;
            item.specUriDisplay = item.specUri;
            item.description = item.description || AJS.I18n.getText("description.not.available");
            item.authorLabel = item.authorLabel || AJS.I18n.getText("author.unknown");
            item.authorLineItem = AJS.format(AJS.I18n.getText("gadgets.directory.author.label"), item.authorLabel);

            if(!/^http/.test(item.specUri)) {
                item.specUriDisplay = AG.param.get("dashboardDirectoryBaseUrl") + item.specUri;
                item.isLocal = true;
            } else if (_.has("completeModuleKey")) {
                item.isLocal = true;
            }

            return item;
        });
    }
});