/**
 * Gadget implementation.
 *
 * @module dashboard
 * @class Gadget
 * @constructor
 * @namespace AG
 */

(function () {

    function FauxList(listContainer) {

        var that = this, ENTER_KEY = 13;

        this.container = AJS.$(listContainer);
        this.values = AJS.$("input.list", this.container).hide().val().split("|");
        this.originalValues = this.values.join("|").split("|"); // creating a separate array - ugly but it works.

        // Create fake form elements and add to container
        this.container.append("<input class=\"add-to-list text med js\" type=\"text\" value=\"\" />" + "<button class=\"submit-to-list js\">" + AJS.I18n.getText('add') + "</button>");

        // Add list items to list individually
        this.container.children(".submit-to-list").click(function (e) {
            that.addToList();
            e.preventDefault();
        });

        AJS.$("input.add-to-list", this.container).keydown(function (e) {
            if (e.keyCode == ENTER_KEY) {
                that.addToList();
                e.preventDefault();
            }
        });

        AJS.$(".description", this.container).appendTo(this.container);

        this.create(this.values);
    }

    FauxList.prototype = {

        reset: function () {
            this.create(this.originalValues);
        },

        create: function (values) {

            var ul, listItem, that = this;

            // If the list is empty, skip all of this.
            if (values) {

                // Strip empty values
                AJS.$.each(values, function () {
                    if (this == "") {
                        values.splice(values.indexOf(this), 1);
                    }
                });

                // Create UL to hold visible list
                ul = AJS.$("<ul class=\"faux-list\"></ul>");

                // Create list for any list items
                // Else, remove if none
                if (values.length > 0) {
                    for (var i = 0, ii = values.length; i < ii; i++) {

                        // Make LI element with delete functionality
                        listItem = AJS.$("<li class=\"listvalue-" + i + "\"><span>" + values[i] + "</span><a class=\"remove\" href=\"#remove-from-list\" title=\"" + AJS.I18n.getText("remove.from.list") + "\">x</a></li>");

                        // Remove list items from list individually
                        AJS.$("a", listItem).click(function (e) {

                            // Grab the value to be removed from the array
                            var removeValue = AJS.$(this).parent("li").attr("class").split("-")[1];

                            // Find the index of the value and remove it
                            values.splice(removeValue, 1);

                            if (values.length > 0) {
                                that.create(values);
                            } else {
                                AJS.$("input.list", this.container).val("");
                            }

                            // Remove LI from visible list
                            listItem.remove();
                            e.preventDefault();
                        });

                        // Add new list items to ul.faux-list
                        ul.append(listItem);
                    }

                    // If no values existed, add ul.faux-list before the old input
                    // Else, replace old .faux-list with new version
                    if (AJS.$("ul.faux-list", this.container).length == 0) {
                        AJS.$("input.list", this.container).before(ul);
                    } else {
                        AJS.$("ul.faux-list", this.container).replaceWith(ul);
                    }
                } else {
                    AJS.$("ul.faux-list", this.container).remove();
                }

                // Rewrite the old input value with current array
                AJS.$("input.list", this.container).val(values.join("|"));

                AG.DashboardManager.getLayout().refresh();

            }
        },

        addToList: function () {
            // Stripping pipes from our pipe-delimited input
            var newValue = AJS.$("input.add-to-list", this.container).val().replace(/\|+/g, '%7C');

            //Grab value from new input and add to array
            this.values.push(newValue);

            // Clear new input
            AJS.$("input.add-to-list", this.container).val("");

            // Remake the list
            this.create(this.values);
        }
    };

    var Gadget = {

        /**
         * Draws gadget, preference form and furniture
         *
         * @method draw
         */
        draw: function (json) {

            var that = this;

            function addRenderingData(args) {

                function generateMenuItems() {
                    var menu = [];

                    function isEditable() {
                        return !!(isWritable() && args.hasNonHiddenUserPrefs);
                    }

                    function isWritable() {
                        return args.layout.writable;
                    }

                    function generateColorList() {
                        var colorList = [];
                        AJS.$.each(AG.Gadget.COLORS, function (colorIndex) {
                            var color = AG.Gadget.getColorAttrName(this);
                            colorList.push({
                                styleClass: color, link: {
                                    href: "#", text: colorNames[colorIndex]
                                }
                            });
                        });
                        return {
                            styleClass: "item-link gadget-colors", items: colorList
                        };
                    }

                    if (isEditable()) {
                        menu.push({
                            styleClass: "dropdown-item", link: {
                                styleClass: "item-link edit",
                                href: "#gadget-" + args.id + "-edit",
                                text: AJS.I18n.getText('edit')
                            }
                        });
                    }

                    menu.push({
                        styleClass: "dropdown-item", link: {
                            styleClass: "item-link " + (args.minimized ? "maximization" : "minimization"),
                            href: "#",
                            text: args.minimized ? AJS.I18n.getText('expand') : AJS.I18n.getText('minimize')
                        }
                    });

                    if (isWritable()) {
                        menu.push(generateColorList());
                        menu.push({
                            styleClass: "dropdown-item", link: {
                                styleClass: "item-link delete", href: "#", text: AJS.I18n.getText('delete')
                            }
                        });
                    }

                    return menu;
                }

                return AJS.$.extend({
                    menu: {
                        trigger: {
                            text: "Gadget menu", href: "#"
                        }, list: {
                            items: generateMenuItems()
                        }
                    }
                }, args);
            }

            function createElement() {
                json.minimized = that.minimized;
                json.view = that.view;
                var renderingData = addRenderingData(json);
                that.$ = AJS.$(Gadgets.Templates.Dashboard.gadget(renderingData));
            }

            // setHeightFromCookie must be called before rendering the gadget
            function setHeightFromCookie() {
                if (json) {
                    var height = AG.Cookie.read("gadget-" + json.id + "-fh", null);
                    if (!isNaN(parseInt(height))) {
                        json = AJS.$.extend(json, {height: height});
                    }
                }
            }

            function setElementShortcuts() {
                that.$.layoutRep = AJS.$("<li class='gadget' id='rep-" + json.id + "' />").height(that.$.height());
                that.$.layoutRep.get(0).getGadgetInstance = that.$.get(0).getGadgetInstance = function () {
                    return that.getPublicInstance();
                };
            }

            function applyDropdownControls() {

                var ACTIVE_CLASS = "dropdown-active", ddParent = AJS.$(".aui-dd-parent", that.$);

                ddParent.mousedown(function (e) {
                    e.stopPropagation();
                });

                that.$.dropdown = ddParent.dropDown("standard", {
                    selectionHandler: function (e) {
                        e.preventDefault();
                    }, item: "> li"
                })[0];

                that.$.dropdown.addCallback("show", function () {
                    that.$.addClass(ACTIVE_CLASS);
                    AG.DashboardManager.showShims();
                });

                that.$.dropdown.addCallback("hide", function () {
                    that.$.removeClass(ACTIVE_CLASS);
                    // if we are not dragging
                    AG.DashboardManager.hideShims();
                });

                that.$.hover(function () {}, function () {
                    if (that.$.dropdown.$.is(":visible")) {
                        that.$.dropdown.hide();
                    }
                });
            }

            function applyGadgetHoverControls() {

                var HOVER_CLASS = "gadget-hover";

                that.$.hover(function () {
                    AJS.$(".gadget-container", that.$).addClass(HOVER_CLASS);
                }, function () {
                    AJS.$(".gadget-container", that.$).removeClass(HOVER_CLASS);
                });
            }

            function applyColorControls() {
                AJS.$(".gadget-colors a", that.$).click(function (e) {
                    that.setColor(this.parentNode.className);
                    e.preventDefault();
                });
            }

            function applyMinimizeControls() {

                var menuElem = AJS.$("a.minimization, a.maximization", that.$), titleElem = AJS.$(".dashboard-item-title", that.$),

                        maxMinToggle = function (e) {

                            if (that.minimized) {
                                that.maximize();
                            } else {
                                that.minimize();
                            }

                            AJS.$(this).one(e.type, function (e) {
                                if (that.minimized) {
                                    that.maximize();
                                } else {
                                    that.minimize();
                                }
                                AJS.$(this).one(e.type, maxMinToggle);
                            });
                        };

                titleElem.one("dblclick", maxMinToggle);
                menuElem.one("click", maxMinToggle);

                /* AJS.$.browser is deprecated, for the preferred feature detection. However feature detection cannot detect safari. */
                if (AJS.$.browser.safari) {
                    /* stops double click from selecting title text */
                    titleElem.get(0).onselectstart = function () {
                        return false;
                    };
                }

            }

            function applyFocusControls() {

                var column;

                AJS.$(".dashboard-item-header", that.$).mousedown(function (e) {
                    // Don't make draggable in Canvas mode unless there are multiple Dashboards.
                    if (!that.$.hasClass("maximized") || AG.DashboardManager.getDashboard().hasClass("v-tabs")) {

                        if (!column) {
                            column = that.$.layoutRep.parent();
                        }

                        // hide dropdown if it is visible
                        that.$.dropdown.hide();

                        that.$.focusTimeout = setTimeout(function () {
                            delete that.$.focusTimeout;
                        }, 150);

                        // Our sortable control uses the layout rep to manage sorting. For this to work we modify the
                        // event and trigger it on the layout rep. This way it is as if the click occured on it.
                        // See AG.Sortable for wiring of layout rep to sortable control.
                        e.target = that.$.layoutRep[0];
                        that.$.layoutRep.trigger(e);
                    }

                });
                that.$.mouseup(function () {
                    if (that.$.focusTimeout) {
                        clearTimeout(that.$.focusTimeout);
                    } else if (that.$.layoutRep.is(":visible")) {
                        that.$.stop(true, true);
                    }
                });
            }

            function applyUserPrefControls() {

                var fauxLists = [], prefForm = AJS.$(".userpref-form", that.$), prefsChanged = false, savePrefForm = function (success) {
                    var formArray = prefForm.serializeArray(), submittedState = {};
                    AJS.$.each(formArray, function () {
                        submittedState[this.name] = this.value || '';
                    });

                    AJS.$(":checkbox:not(:checked)", prefForm).each(function () {
                        submittedState[this.name] = false;
                    });

                    AJS.$.ajax({
                        url: prefForm.attr("action"),
                        type: "PUT",
                        contentType: "application/json",
                        data: JSON.stringify(submittedState),
                        dataType: "text",
                        success: success
                    });
                };

                if (that.isEditable()) {
                    // Creates a neater form element for adding multiple values
                    AJS.$(".list-container", that.$).each(function () {
                        fauxLists.push(new FauxList(this));
                    });

                    AJS.$(".edit", that.$).click(function (e) {
                        if (!prefForm.is(":visible")) {
                            if (that.minimized) {
                                that.maximize();
                            }
                            prefForm.show();
                            AJS.$(".field-group :input:first", prefForm).focus();
                            that.getLayoutManager().refresh();
                        } else {
                            prefForm.hide();
                            that.getLayoutManager().refresh();
                        }
                        e.preventDefault();
                    });

                    prefForm.submit(function (e) {
                        savePrefForm(function () {
                            window.location.reload();
                        });

                        e.preventDefault();
                    });

                    AJS.$(":reset", prefForm).click(function () {
                        AJS.$.each(fauxLists, function () {
                            this.reset();
                        });
                        prefForm.hide();
                        that.getLayoutManager().refresh();
                    });
                }

                /*
                 * Event handler for a customed event "setUserPref".  This event
                 * is triggered inside gadgets-dashboard.js -> setUserPref().
                 * It acts like a buffer for all the setPref() calls.
                 * The results are flushed to the server using ajax later.
                 * arguments e is something like [name1, value1, name2, value2]
                 */
                AJS.$("iframe.gadget-iframe", that.$).bind("setUserPref", function (e) {

                    // update the preference form with the given name value pairs
                    for (var i = 1, j = arguments.length; i < j; i += 2) {
                        if (pref_name = arguments[i]) {
                            var pref_value = arguments[i + 1];
                            var input = AJS.$(":input[name='up_" + pref_name + "']", prefForm);
                            // checkbox needs a default value of false
                            if (input.attr("type") == "checkbox") {
                                input.attr("checked", pref_value);
                            } else {
                                input.val(pref_value);
                            }
                        }
                    }

                    // flag for whether a flush is scheduled
                    if (!prefsChanged) {
                        setTimeout(function () {
                            savePrefForm(function () {});
                            prefsChanged = false;
                        }, 100);
                        prefsChanged = true;
                    }
                });

            }

            function applyDeleteControls() {
                AJS.$("a.delete", that.$).bind("click", function (e) {
                    that.destroy();
                    e.preventDefault();
                });
            }

            function applyViewToggleControls() {

                AJS.$("a.maximize", that.$).click(function (e) {
                    if (!AG.Gadget.isCanvasView(that.id)) {
                        that.getPublicInstance().setView("canvas");
                        e.preventDefault();
                    } else {
                        that.getPublicInstance().setView("default");
                        e.preventDefault();
                    }
                });
            }

            function applyErrorMessage() {
                AJS.$("iframe.gadget-iframe", that.$).load(function () {
                    AJS.$("#error-gadget-message", AJS.$(this).contents()).html(that.errorMessage);
                });
            }

            setHeightFromCookie();
            createElement();
            setElementShortcuts();
            applyDropdownControls();
            applyGadgetHoverControls();
            applyMinimizeControls();

            if (this.isMaximizable) {
                applyViewToggleControls();
            }

            if (!this.loaded) {
                applyErrorMessage();
            }

            if (this.getLayoutManager().isWritable()) {
                applyDeleteControls();
                applyFocusControls();
                applyColorControls();
                applyUserPrefControls();
            }
        },

        isEditable: function () {
            //dashboard items provide a 'configurable' property if they can be edited.
            if (this.configurable) {
                return this.getLayoutManager().isWritable();
            } else {
                return !!(this.getLayoutManager().isWritable() && this.hasNonHiddenUserPrefs);
            }
        },

        setLayoutManager: function (layoutManager) {
            if (layoutManager) {
                this.layoutManager = layoutManager;
            } else {
                this.layoutManager = AG.DashboardManager.getLayout();
            }
        },

        /**
         *
         * Provides access to gadget's Layout manager. When the gadget
         * needs to know how it's owner is. Such as in the method destroy
         * where we need to also remove it from the layout.
         *
         * @method getLayoutManager
         */
        getLayoutManager: function () {
            if (!this.layoutManager) {
                this.setLayoutManager();
            }
            return this.layoutManager;
        },

        /**
         * Sets gadget chrome colour. Gets chrome and applies associated class and
         * sends preference back to server to persist.
         *
         * @method setColor
         */
        setColor: function (color) {

            var that = this;

            that.$.removeClass(that.color).addClass(color);
            that.color = color;
            AJS.$.ajax({
                type: "PUT",
                url: that.colorUrl,
                contentType: "application/json",
                data: JSON.stringify({color: color}),
                error: function (request) {
                    if (request.status == 403 || request.status == 401) {
                        alert(AJS.I18n.getText("dashboard.error.dashboard.permissions"));
                    } else {
                        alert(AJS.I18n.getText("dashboard.error.could.not.save"));
                    }
                }
            });
        },

        updatePosition: function () {

            var gadgetCSSToUpdate, layoutCSSToUpdate, that = this;

            function isGadgetBeingDragged() {
                return that.$.hasClass("dragging");
            }

            function getCurrentGadgetCSS() {
                var LAYOUT_REP_OFFSET, dashboard = AG.DashboardManager.getDashboard().contents, DASHBOARD_OFFSET = dashboard.offset();

                if (!getCurrentGadgetCSS.cache) {
                    LAYOUT_REP_OFFSET = that.$.layoutRep.offset();
                    getCurrentGadgetCSS.cache = {
                        left: (LAYOUT_REP_OFFSET.left - DASHBOARD_OFFSET.left) / dashboard.width() * 100 + "%",
                        top: LAYOUT_REP_OFFSET.top - DASHBOARD_OFFSET.top,
                        width: that.$.layoutRep.width() / dashboard.width() * 100 + "%"
                    };
                }
                return getCurrentGadgetCSS.cache;
            }

            function getCurrentLayoutRepCSS() {
                if (!getCurrentLayoutRepCSS.cache) {
                    getCurrentLayoutRepCSS.cache = {
                        height: that.$.height()
                    };
                }
                return getCurrentLayoutRepCSS.cache;
            }

            function filterModifiedCSS(lastRecordedCSS, currentCSS) {
                if (lastRecordedCSS) {
                    AJS.$.each(lastRecordedCSS, function (property) {
                        if (this === currentCSS[property]) {
                            delete currentCSS[property];
                        }
                    });
                }
                return currentCSS;
            }

            if (!isGadgetBeingDragged()) {
                layoutCSSToUpdate = filterModifiedCSS(this.$.layoutRep.lastRecordedCSS, getCurrentLayoutRepCSS());
                this.$.layoutRep.css(layoutCSSToUpdate);
                this.$.layoutRep.lastRecordedCSS = layoutCSSToUpdate;

                gadgetCSSToUpdate = filterModifiedCSS(this.$.lastRecordedCSS, getCurrentGadgetCSS());
                this.$.css(gadgetCSSToUpdate);
                this.$.lastRecordedCSS = gadgetCSSToUpdate;

                if (this.$.hasClass("hidden")) {
                    this.$.removeClass("hidden");
                }
            }

        },

        /**
         * Minimises gadget. Hides everything but title bar.
         *
         * @method maximize
         */
        maximize: function () {
            var MIN_CLASS = "minimization",
                MAX_CLASS = "maximization",
                MIN_TEXT = AJS.I18n.getText('minimize'),
                menuElem = AJS.$("a.minimization, a.maximization", this.$);

            menuElem.removeClass(MAX_CLASS).addClass(MIN_CLASS).text(MIN_TEXT);

            // need to reset height to auto because sortable control sets an explicit pixel height
            this.$.css({height: "auto"});
            AJS.$(".dashboard-item-content", this.$).removeClass(MIN_CLASS);
            // updates positioning of gadgets & their layout references
            this.getLayoutManager().refresh();

            /* erase cookie */
            AG.Cookie.erase(this.COOKIE_MINIMIZE);

            if (this.minimized) {
                this.moduleAPI.trigger("expand");
            }
            this.minimized = false;
        },

        minimize: function () {
            var MIN_CLASS = "minimization",
                MAX_CLASS = "maximization",
                MAX_TEXT = AJS.I18n.getText('expand'),
                menuElem = AJS.$("a.minimization, a.maximization", this.$);

            menuElem.removeClass(MIN_CLASS).addClass(MAX_CLASS).text(MAX_TEXT);

            // need to reset height to auto because sortable control sets an explicit pixel height
            this.$.css({height: "auto"});
            AJS.$(".dashboard-item-content", this.$).addClass(MIN_CLASS);
            this.getLayoutManager().refresh();
            AG.Cookie.save(this.COOKIE_MINIMIZE, "true");
            this.minimized = true;
            this.moduleAPI.trigger("minimize");
        },

        remove: function () {
            var that = this;
            that.$.layoutRep.remove();
            that.$.remove();

            this.moduleAPI.trigger("remove");
            that.getLayoutManager().removeGadget(that.getPublicInstance());
            that.getLayoutManager().refresh();
            // remove from memory
            AJS.$.each(this, function (name, property) {
                property = null;
            });
            AG.Cookie.erase("gadget-" + that.id + "-fh");
        },

        /**
         * Moves gadget from the dashboard it's currently on to a new dashboard specified
         * by the target resource URL
         */
        move: function (targetResourceUrl) {
            this.remove();

            AJS.$(AJS.$.ajax({
                type: "PUT",
                data: {
                    id: this.id,
                    title: this.title,
                    titleUrl: this.titleUrl,
                    gadgetSpecUrl: this.gadgetSpecUrl,
                    height: this.$.height(),
                    width: this.$.width(),
                    color: this.color,
                    isMaximizable: this.isMaximizable,
                    userPrefs: this.userPrefs,
                    renderedGadgetUrl: this.renderedGadgetUrl,
                    colorUrl: this.colorUrl,
                    gadgetUrl: this.gadgetUrl,
                    hasNonHiddenUserPrefs: this.hasNonHiddenUserPrefs,
                    column: this.column,
                    loaded: this.loaded
                },
                contentType: "application/json",
                url: targetResourceUrl + "/gadget/" + this.id,
            })).throbber({target: AJS.$("#dash-throbber")});
        },

        /**
         * Removes gadget from dashboard and deletes object references
         *
         * @method destroy
         */
        destroy: function () {

            var that = this;

            if (confirm(AJS.I18n.getText("are.you.sure", that.title))) {

                AJS.$("#dash-throbber").addClass("loading");

                AJS.$.ajax({
                    type: "DELETE", url: this.gadgetUrl,
                    complete: function () {
                        AJS.$("#dash-throbber").removeClass("loading");
                    },
                    success: function () {
                        that.$.fadeOut(function () {
                            that.remove();
                            if (that.view === "canvas") {
                                that.getLayoutManager().restoreDefaultView();
                            }
                        });
                    },
                    error: function (request) {
                        if (request.status == 403 || request.status == 401) {
                            alert(AJS.I18n.getText("dashboard.error.dashboard.permissions"));
                        } else {
                            alert(AJS.I18n.getText("dashboard.error.could.not.save"));
                        }
                    }
                });
            }
        },

        /**
         * Sets the layout of the gadget to either canvas or dashboard. Does
         * so by delegating layout actions to LayoutManager.
         *
         * @method setView
         * @param {String} view - Accepts either "canvas" or "dashboard"
         */
        setView: function (view) {
            var MAXIMIZED_CLASS = "maximized", uri = {}, that = this, anchor = this.title.replace(/\s/g, "-") + "/" + this.id, layoutManager = this.getLayoutManager(), rpctoken;

            function toDefaultViewHandler() {
                that.getPublicInstance().setView("default");
            }

            if (this.view === view) {
                return;
            }

            if (view === "canvas" || view === "default") {

                // use rendered url to get latest user prefs and a fresh security token
                if (this.gadgetUrl) {
                    AJS.$.ajax({
                        async: false, type: "GET", url: this.gadgetUrl, dataType: "json", success: function (rep) {
                            // create the rpctoken that will be used in rpc calls
                            rpctoken = Math.round(Math.random() * 10000000);
                            uri = AJS.parseUri(rep.renderedGadgetUrl + "#rpctoken=" + rpctoken);
                        }
                    });
                }

                // setup the iframe to send/receive rpc calls
                gadgets.rpc.setAuthToken(AJS.$("iframe.gadget-iframe", this.$).attr("id"), rpctoken);

                if (view === "canvas") {

                    AJS.$(".operations li", AG.DashboardManager.getDashboard()).toggleClass("hidden");

                    AJS.$.extend(uri.queryKey, {view: "canvas"});
                    AJS.$("iframe.gadget-iframe", this.$).attr("src", uri.toString());

                    layoutManager.getContainer().addClass(MAXIMIZED_CLASS);
                    AJS.$(".gadget-container", this.$).addClass(MAXIMIZED_CLASS);
                    this.$.addClass(MAXIMIZED_CLASS);
                    AJS.$(".aui-icon", this.$).attr("title", AJS.I18n.getText("dashboard.disabled.canvas.mode"));
                    this.$.layoutRep.addClass(MAXIMIZED_CLASS);
                    this.$.layoutRep.parent().addClass(MAXIMIZED_CLASS);

                    // Not really sure about this, would prefer to add a class to the body tag. Problem is I can't as I
                    // do not want every gadget to be hidden, in the case of multiple tabs this would cause problems.
                    AJS.$.each(this.getLayoutManager().getGadgets(), function () {
                        if (that.getPublicInstance() !== this) {
                            this.getElement().hide();
                        }
                    });

                    AJS.$.extend(uri.queryKey, {view: "canvas"});
                    AJS.$("iframe.gadget-iframe", this.$).attr("src", uri.toString());

                    this.maximize();
                    // add bookmarking capabilities
                    window.location.href = window.location.href.replace(/#.*/, "") + "#" + anchor;
                    AJS.$(".minimize", AG.DashboardManager.getDashboard()).click(toDefaultViewHandler);
                    this.view = "canvas";

                    this.moduleAPI.trigger("maximize");
                } else {

                    AJS.$(".gadget-container", this.$).removeClass(MAXIMIZED_CLASS);
                    this.$.removeClass(MAXIMIZED_CLASS);
                    AJS.$(".aui-icon", this.$).attr("title", AJS.I18n.getText("maximize"));
                    this.$.layoutRep.removeClass(MAXIMIZED_CLASS);
                    this.$.layoutRep.parent().removeClass(MAXIMIZED_CLASS);

                    this.getLayoutManager().restoreDefaultView();

                    AJS.$.extend(uri.queryKey, {view: "default"});
                    AJS.$("iframe.gadget-iframe", this.$).attr("src", uri.toString());
                    this.getLayoutManager().refresh();
                    window.location.href = window.location.href.replace(anchor, "");
                    AJS.$("a.minimize", AG.DashboardManager.getDashboard()).unbind("click", toDefaultViewHandler);
                    this.view = "default";

                    this.moduleAPI.trigger("restore");
                }

            } else if (AJS.debug) {
                console.warn("AG.Gadget.setView: Ignored! not a valid view. Was supplied '" + view + "' but expected " + "either 'default' or 'canvas'");
            }
        },

        /**
         * Displays edit preferences form
         *
         * @method editPrefs
         */
        editPrefs: function () {},

        getPublicInstance: function () {
            var gadget = this;
            var itemContent = gadget.$.find(".gadget-inline");
            var freezeHeight = function() {
                itemContent.css('height', itemContent.height() + 'px');
            };

            if (!this.publicInterface) {
                this.publicInterface = {
                    updatePosition: function () {
                        return gadget.updatePosition.apply(gadget, arguments);
                    },
                    getLayoutManager: function () {
                        return gadget.getLayoutManager.apply(gadget, arguments);
                    },
                    setLayoutManager: function () {
                        return gadget.setLayoutManager.apply(gadget, arguments);
                    },
                    getElement: function () {
                        return gadget.$;
                    },
                    move: function (targetUrl) {
                        return gadget.move(targetUrl);
                    },
                    remove: function () {
                        return gadget.remove.apply(gadget, arguments);
                    },
                    getId: function () {
                        return gadget.id;
                    },
                    showShim: function () {
                        return gadget.showShim.apply(gadget, arguments);
                    },
                    hideShim: function () {
                        return gadget.hideShim.apply(gadget, arguments);
                    },
                    minimize: function () {
                        return gadget.minimize.apply(gadget, arguments);
                    },
                    maximize: function () {
                        return gadget.minimize.apply(gadget, arguments);
                    },
                    getSecurityToken: function () {
                        return gadget.securityToken;
                    },
                    setSecurityToken: function (securityToken) {
                        gadget.securityToken = securityToken;
                    },
                    onDragStart: function () {
                        gadget.moduleAPI.trigger("dragStart");
                    },
                    onDragStop: function () {
                        gadget.moduleAPI.trigger("dragStop");
                    },
                    onAfterRender: function () {
                        gadget.moduleAPI.trigger("afterRender");
                    },
                    freezeHeight: freezeHeight,
                    resize: function() {
                        itemContent.css('height', '');
                        gadget.moduleAPI.trigger("force-layout-refresh");
                        freezeHeight();
                    }
                };

                if (this.isMaximizable) {
                    this.publicInterface.setView = function () {
                        return gadget.setView.apply(gadget, arguments);
                    };
                }
            }

            return this.publicInterface;
        },

        init: function (options) {
            this.COOKIE_MINIMIZE = options.id + ":minimized";
            this.minimized = AG.Cookie.read(this.COOKIE_MINIMIZE) === "true";
            this.title = options.title;
            this.color = options.color;
            this.colorUrl = options.colorUrl;
            this.gadgetUrl = options.gadgetUrl;
            this.id = options.id;
            this.hasNonHiddenUserPrefs = options.hasNonHiddenUserPrefs;
            this.isMaximizable = options.isMaximizable;
            this.titleUrl = options.titleUrl;
            this.gadgetSpecUrl = options.gadgetSpecUrl;
            this.userPrefs = options.userPrefs;
            this.renderedGadgetUrl = options.renderedGadgetUrl;
            this.column = options.column;
            this.loaded = options.loaded;
            this.errorMessage = options.errorMessage;
            this.inlineHtml = options.inlineHtml;
            this.configurable = options.configurable;
            this.amdModule = options.amdModule;
            this.webResourceKey = options.webResourceKey;
            this.context = options.context;
            this.securityToken = AJS.parseUri(options.renderedGadgetUrl).queryKey["st"];
            this.draw(options);

            this.moduleAPI = new AG.InlineGadgetAPI(this);

            if (this.amdModule) {
                if (this.webResourceKey) {
                    WRM.require(['wr!' + this.webResourceKey], function() {
                        this.initializeAmdModule(options);
                    }.bind(this));
                }
                else {
                    this.initializeAmdModule(options);
                }
            }
        },
        initializeAmdModule: function initializeAmdModule(options) {
            var gadget = this;
            var gadgetElement = this.$.find("#gadget-" + this.id);
            var Module = AG.Util.safeInit(function () {
                return require(gadget.amdModule);
            }, gadgetElement, gadget, "Can not import amd Module " + this.amdModule);

            if (Module) {
                var userPreferences = AG.Util.extractUserPreferences(options.userPrefs.fields);
                var moduleApi = this.moduleAPI;
                var moduleInstance = AG.Util.safeInit(function () {
                    return new Module(moduleApi);
                }, gadgetElement, gadget, "Error initializing");

                var safeRender = function (renderingFunction) {
                    AG.Util.safeRender(renderingFunction, gadgetElement, gadget);
                };

                if (this.isEditable()) {
                    var dropdown = this.getPublicInstance().getElement().dropdown;
                    var configureLink = AJS.$('<li class="dropdown-item configure"><a class="item-link no_target" href="#">' + AJS.I18n.getText("gadget.common.configure") + '</a></li>');
                    configureLink.on("click", function () {
                        safeRender(function () {
                            moduleInstance.renderEdit(gadgetElement, userPreferences);
                            gadget.moduleAPI.trigger("afterRender");
                        });
                    });
                    configureLink.appendTo(dropdown.$);
                    dropdown.reset();
                }

                if (this.isEditable() && !userPreferences.isConfigured) {
                    safeRender(function () {
                        moduleInstance.renderEdit(gadgetElement, userPreferences);
                    });
                } else {
                    safeRender(function () {
                        moduleInstance.render(gadgetElement, userPreferences);
                    });
                }
                //afterRender is triggered by the layout manager here after the gadget is inserted
                //into the DOM, so no need to call it explicitly.

                this.moduleAPI.on("preferencesSaved", function (prefs) {
                    userPreferences = AG.Util.extractUserPreferences(prefs);
                    safeRender(function () {
                        moduleInstance.render(gadgetElement, userPreferences);
                        gadget.moduleAPI.trigger("afterRender");
                    });
                });
                this.moduleAPI.on("close-edit", function () {
                    safeRender(function () {
                        moduleInstance.render(gadgetElement, userPreferences);
                        gadget.moduleAPI.trigger("afterRender");
                    });
                });
                this.moduleAPI.on("force-layout-refresh", _.bind(function () {
                    this.getLayoutManager().refresh();
                }, this));
            }
        }
    };

    AG.Gadget = function (options) {
        // Using prototype as there could be many gadgets on the page. This is most memory efficient.
        var gadget = AJS.clone(Gadget);

        gadget.init(options);

        return gadget.getPublicInstance();
    };

    AG.Gadget.COLORS = [1, 2, 3, 4, 5, 6, 7, 8];

    AG.Gadget.getColorAttrName = function (color) {
        return "color" + color;
    };

    var colorNames = [
        AJS.I18n.getText('color.1'),
        AJS.I18n.getText('color.2'),
        AJS.I18n.getText('color.3'),
        AJS.I18n.getText('color.4'),
        AJS.I18n.getText('color.5'),
        AJS.I18n.getText('color.6'),
        AJS.I18n.getText('color.7'),
        AJS.I18n.getText('color.8')
    ];

    AG.Gadget.isCanvasView = function (gadgetId) {
        var uri = AJS.parseUri(window.location.href);
        return new RegExp(gadgetId).test(uri.anchor);
    };

    AG.Gadget.getNullGadgetRepresentation = function (errorGadget) {
        return AJS.$.extend(errorGadget, {
            title: AJS.I18n.getText("gadget.error.title"),
            renderedGadgetUrl: AG.param.get("errorGadgetUrl"),
            color: errorGadget.color || "color7"
        });
    };

    AG.Util = {
        extractUserPreferences: function (userPreferences) {
            var preferences = {};
            _.each(userPreferences, function (preference) {
                var parsedValue;
                if (preference.value === "true" || preference.value === "false") {
                    parsedValue = preference.value === "true";
                } else {
                    parsedValue = preference.value;
                }
                preferences[preference.name] = parsedValue;
            });
            return preferences;
        },
        safeRender: function (renderFunction, gadgetElement, gadget) {
            this.safeInvokeFunction(renderFunction, gadgetElement, gadget, AJS.I18n.getText("gadget.error.rendering"), "Error rendering")
        },
        safeInit: function(initFunction, gadgetElement, gadget, errorMsg) {
            return this.safeInvokeFunction(initFunction, gadgetElement, gadget, AJS.I18n.getText("gadget.error.initializing"), errorMsg)
        },
        safeInvokeFunction: function (functionToInvoke, gadgetElement, gadget, userErrorMsg, consoleErrorMsg) {
            try {
                return functionToInvoke();
            } catch (err) {
                var errorContainer = AJS.$('<div class="item-error"/>').append(aui.message.error({
                    content: "<p>" + userErrorMsg + "</p>"
                }));
                AJS.$(gadgetElement).append(errorContainer);
                console.error(consoleErrorMsg + " gadget with id '" + gadget.id + "': " + err.message, err);
                gadget.moduleAPI.hideLoadingBar();
            }
        }
    };

})();
