var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

/* global AJS, _, require, QUnit, sinon, module, test, asyncTest, expect, equal, ok */
AJS.test.require('com.atlassian.jira.jira-onboarding-assets-plugin:sequence-controller-component', function () {
    var jQuery = require("jquery");
    var wrmContextPath = require("wrm/context-path");
    var _ = require("underscore");
    'use strict';

    var Promise = require("bluebird/Promise");

    var SequenceController;

    var DEFAULT_FLOW_KEY = 'flowKey';

    var mockAnalytics = {
        pushEvent: function pushEvent() {}
    };
    var MOCK_ANALYTICS = function MOCK_ANALYTICS() {
        return mockAnalytics;
    };

    MOCK_ANALYTICS.EVENTS = {
        STARTED_SEQUENCE: 'started',
        COMPLETED: 'completed',
        FAILED: 'failed'
    };

    module('jira/onboarding/sequence-controller - construction', {
        setup: function setup() {
            this.context = AJS.test.mockableModuleContext();
            this.context.mock('jira/onboarding/analytics', MOCK_ANALYTICS);
            SequenceController = this.context.require('jira/onboarding/sequence-controller');
        }
    });

    test('Needs a non-null container value', 1, function () {
        try {
            new SequenceController();
        } catch (e) {
            ok(e.message.indexOf('Should have passed a container') > -1, 'should complain about not having an element');
        }
    });

    test('Needs a DOM element for a container', 1, function () {
        try {
            new SequenceController({
                container: {}
            });
        } catch (e) {
            ok(e.message.indexOf('DOM') > -1, 'should complain about not having an element');
        }
    });

    test('Needs a flow function', 2, function () {
        try {
            new SequenceController({
                container: document.createElement('div'),
                flow: {}
            });
        } catch (e) {
            ok(e.message.indexOf('flow') > -1, 'should be something about a flow');
            ok(e.message.indexOf('function') > -1, 'should complain about not having a function');
        }
    });

    test('Can be initialised successfully', function () {
        var controller = new SequenceController({
            container: document.createElement('div'),
            flow: function flow() {
                return {
                    start: 'a',
                    sequences: {
                        'a': {
                            instance: new function sequence() {}()
                        }
                    }
                };
            }
        });
        equal(_typeof(controller.start), 'function');
    });

    module('jira/onboarding/sequence-controller', {
        setup: function setup() {
            this.server = sinon.fakeServer.create();
            this.element = jQuery('#qunit-fixture');

            this.context = AJS.test.mockableModuleContext();
            this.context.mock('jira/onboarding/analytics', MOCK_ANALYTICS);
            SequenceController = this.context.require('jira/onboarding/sequence-controller');
        },

        teardown: function teardown() {
            this.server.restore();
        }
    });

    test('If no current sequence then it begins at the default start sequence', function () {
        expect(1);
        var flow = function flow() {
            return {
                start: 'b',
                sequences: {
                    'a': {
                        instance: {
                            init: function init() {
                                ok(false);
                                return Promise.resolve();
                            }
                        }
                    },
                    'b': {
                        instance: {
                            init: function init() {
                                ok(true);
                                return Promise.resolve();
                            }
                        }
                    }
                }
            };
        };

        var controller = new SequenceController({
            container: this.element,
            flow: flow
        });
        controller.start();
    });

    test('If a server stored sequence is defined it starts on that one', function () {
        expect(1);
        var flow = function flow() {
            return {
                start: 'a',
                sequences: {
                    'a': {
                        instance: {
                            init: function init() {
                                ok(false);
                                return Promise.resolve();
                            }
                        }
                    },
                    'b': {
                        instance: {
                            init: function init() {
                                ok(true);
                                return Promise.resolve();
                            }
                        }
                    }
                }
            };
        };

        var controller = new SequenceController({
            container: this.element,
            flow: flow
        });
        controller.start('b');
    });

    asyncTest('If a string is supplied to the resolve/reject key that is the sequence that is completed next', function () {
        expect(4);
        var flow = function flow() {
            return {
                start: 'a',
                sequences: {
                    'a': {
                        instance: {
                            init: function init() {
                                ok(true);
                                return Promise.resolve();
                            }
                        },
                        resolve: 'b'
                    },
                    'b': {
                        instance: {
                            init: function init() {
                                ok(true);
                                return Promise.resolve();
                            }
                        }
                    }
                }
            };
        };

        var controller = new SequenceController({
            container: this.element,
            flow: flow,
            flowKey: DEFAULT_FLOW_KEY
        });
        controller.start();

        var server = this.server;

        _.defer(function () {
            equal(server.requests.length, 1, 'Should have a request that it is running a new sequence');
            equal(server.requests[0].url, wrmContextPath() + '/rest/onboarding/1.0/flow/flowKey/sequence/current/b', 'Should indicate starting second');
            QUnit.start();
        });
    });

    asyncTest('If a function is supplied to the resolve/reject key the result is the sequence to next be completed', function () {
        expect(4);
        var flow = function flow() {
            return {
                start: 'a',
                sequences: {
                    'a': {
                        instance: {
                            init: function init() {
                                ok(true);
                                return Promise.resolve();
                            }
                        },
                        resolve: function resolve() {
                            return 'b';
                        }
                    },
                    'b': {
                        instance: {
                            init: function init() {
                                ok(true);
                                return Promise.resolve();
                            }
                        }
                    }
                }
            };
        };

        var controller = new SequenceController({
            container: this.element,
            flow: flow,
            flowKey: DEFAULT_FLOW_KEY
        });
        controller.start();

        var server = this.server;

        _.defer(function () {
            equal(server.requests.length, 1, 'Should have a request that it is running a new sequence');
            equal(server.requests[0].url, wrmContextPath() + '/rest/onboarding/1.0/flow/flowKey/sequence/current/b', 'Should indicate starting third');
            QUnit.start();
        });
    });

    asyncTest('If a resolution results in complete being called a post request is sent', function () {
        expect(2);
        var flow = function flow(complete) {
            return {
                start: 'a',
                sequences: {
                    'a': {
                        instance: {
                            init: function init() {
                                return Promise.resolve();
                            }
                        },
                        resolve: function resolve() {
                            complete();
                        }
                    }
                }
            };
        };

        var controller = new SequenceController({
            container: this.element,
            flow: flow,
            flowKey: DEFAULT_FLOW_KEY
        });
        controller.start();

        var server = this.server;

        _.defer(function () {
            equal(server.requests.length, 1, 'Should have a request that it is running a new sequence');
            equal(server.requests[0].url, wrmContextPath() + '/rest/onboarding/1.0/flow/flowKey/complete', 'Should indicate completed flow');
            QUnit.start();
        });
    });

    asyncTest('Test that data is passed from resolution to resolve function if supplied', function () {
        expect(1);
        var flow = function flow() {
            return {
                start: 'a',
                sequences: {
                    'a': {
                        instance: {
                            init: function init() {
                                return Promise.resolve(6);
                            }
                        },
                        resolve: function resolve(reason) {
                            equal(reason, 6);
                            QUnit.start();
                        }
                    }
                }
            };
        };

        var controller = new SequenceController({
            container: this.element,
            flow: flow
        });
        controller.start();
    });

    asyncTest('Test that data is passed from rejection to reject function if supplied', function () {
        expect(1);
        var flow = function flow() {
            return {
                start: 'a',
                sequences: {
                    'a': {
                        instance: {
                            init: function init() {
                                return Promise.reject(6);
                            }
                        },
                        reject: function reject(reason) {
                            equal(reason, 6);
                            QUnit.start();
                        }
                    }
                }
            };
        };

        var controller = new SequenceController({
            container: this.element,
            flow: flow
        });
        controller.start();
    });

    asyncTest('sequences are passed an element and analytics object', function () {
        expect(3);
        var sequenceA = {
            init: sinon.spy(function () {
                return Promise.resolve();
            })
        };
        function runAssertions() {
            equal(sequenceA.init.callCount, 1);
            ok(sequenceA.init.args[0][0] instanceof HTMLElement, 'first argument should be an element');
            equal(sequenceA.init.args[0][1], mockAnalytics, 'second argument should be the analytics object');
            QUnit.start();
        }
        var flow = function flow() {
            return {
                start: 'a',
                sequences: {
                    'a': {
                        instance: sequenceA,
                        resolve: runAssertions
                    }
                }
            };
        };

        var controller = new SequenceController({
            container: this.element,
            flow: flow
        });
        controller.start();
    });

    asyncTest('events triggered (and handled) in one sequence should not be triggered in subsequent ones', function () {
        expect(3);
        function sequenceWithAView(name) {
            return {
                clickHandler: sinon.spy(),
                init: function init(element) {
                    element.addEventListener('click', this.clickHandler);
                    element.click();
                    return Promise.resolve();
                }
            };
        }
        var sequenceA = sequenceWithAView('a');
        var sequenceB = sequenceWithAView('b');
        var sequenceC = sequenceWithAView('c');
        function runAssertions() {
            equal(sequenceA.clickHandler.callCount, 1);
            equal(sequenceB.clickHandler.callCount, 1);
            equal(sequenceC.clickHandler.callCount, 1);
            QUnit.start();
        }
        var flow = function flow() {
            return {
                start: 'a',
                sequences: {
                    'a': {
                        instance: sequenceA,
                        resolve: 'b'
                    },
                    'b': {
                        instance: sequenceB,
                        resolve: 'c'
                    },
                    'c': {
                        instance: sequenceC,
                        resolve: runAssertions
                    }
                }
            };
        };

        var controller = new SequenceController({
            container: this.element,
            flow: flow
        });
        controller.start();
    });

    module('jira/onboarding/sequence-controller - resuming after expired session', {
        setup: function setup() {
            this.server = sinon.fakeServer.create();
            this.windowSpy = sinon.spy();
            this.element = jQuery('#qunit-fixture');
            this.localStore = {
                setItem: sinon.spy(),
                getItem: sinon.stub(),
                removeItem: sinon.stub()
            };

            this.context = AJS.test.mockableModuleContext();
            this.context.mock('jira/data/local-storage', this.localStore);
            this.context.mock('jira/util/browser', { reloadViaWindowLocation: this.windowSpy });
            this.context.mock('jira/onboarding/analytics', MOCK_ANALYTICS);
            SequenceController = this.context.require('jira/onboarding/sequence-controller');
        },

        teardown: function teardown() {
            this.server.restore();
        }
    });

    asyncTest('If the API returns a 401 response, the next sequence key should be cached in localStorage', function () {
        expect(5);
        var flow = function flow() {
            return {
                start: 'a',
                sequences: {
                    'a': {
                        instance: {
                            init: function init() {
                                ok(true, 'First sequence started while logged in');
                                return Promise.resolve();
                            }
                        },
                        resolve: 'b'
                    },
                    'b': {
                        instance: {
                            init: function init() {
                                ok(true, 'Second sequence started after session expired');
                                return Promise.resolve();
                            }
                        }
                    }
                }
            };
        };

        var controller = new SequenceController({
            container: this.element,
            flow: flow,
            flowKey: DEFAULT_FLOW_KEY
        });

        this.server.respondWith([401, { 'Content-Type': 'text/html' }, "Unauthorised"]);
        var windowSpy = this.windowSpy;
        var storageSpy = this.localStore;
        var server401 = this.server;

        controller.start();

        _.defer(function () {
            server401.respond();
            ok(storageSpy.removeItem.calledWith('onboarding.sequence.started'), 'Should clear localStorage when completing sequence');
            ok(storageSpy.setItem.calledWith('onboarding.sequence.started', 'b'), 'Should cache new sequence to localStorage');
            ok(windowSpy.calledOnce, 'Should redirect to login screen');
            QUnit.start();
        });
    });

    test('If a sequenceKey is cached in localStorage, it starts with that one', function () {
        expect(1);
        var flow = function flow() {
            return {
                start: 'a',
                sequences: {
                    'a': {
                        instance: {
                            init: function init() {
                                ok(false, 'First sequence should be skipped');
                                return Promise.resolve();
                            }
                        }
                    },
                    'b': {
                        instance: {
                            init: function init() {
                                ok(true, 'Second sequence correctly resumed');
                                return Promise.resolve();
                            }
                        }
                    }
                }
            };
        };

        var controller = new SequenceController({
            container: this.element,
            flow: flow
        });

        this.localStore.getItem.withArgs('onboarding.sequence.started').returns('b');
        controller.start();
    });
});