AJS.test.require(["com.pyxis.greenhopper.jira:gh-test-common", "com.pyxis.greenhopper.jira:gh-rapid"], function () {

    var $ = require('jquery');
    var WorkController = require('jira-agile/rapid/ui/work/work-controller');

    module("Ajax tests", {
        setup: function setup() {
            GH.Test.setUpFakeServer(this);
            WorkController.isVisible = true;
            WorkController.rapidViewData = {
                id: 1
            };
        },
        teardown: function teardown() {
            GH.Test.restoreServer(this);
        },
        getQunitFixture: function getQunitFixture() {
            if (!this.qunitFixture) {
                this.qunitFixture = AJS.$("#qunit-fixture");
            }

            return this.qunitFixture;
        }
    });

    test("resolveParentTask: Successful first and second Ajax request shows the dialog", sinon.test(function () {
        this.stub(GH.Dialogs.CompleteParentDialog, "showDialog");
        this.stub(GH.GridDataController, "getModel").returns({ data: { columns: [{ statusIds: [1] }] } });

        GH.Test.respondToGetWith200(this, '/xboard/issue/subtasksInFinalColumn.json', { subtasksInFinalColumn: true });
        GH.Test.respondToGetWith200(this, '/xboard/issue/transitions.json', { transitions: [] });

        WorkController.resolveParentTask({ id: 1 });

        this.server.respond();
        this.server.respond();

        ok(GH.Dialogs.CompleteParentDialog.showDialog.called, "Called success method");
    }));

    test("resolveParentTask: Failed first request hides the loading pool", sinon.test(function () {
        this.stub(GH.Dialogs.CompleteParentDialog, "showDialog");
        this.stub(GH.WorkView, "hideLoadingPool");

        GH.Test.respondToGetWith500(this, '/xboard/issue/subtasksInFinalColumn.json', {});

        WorkController.resolveParentTask({ id: 1 });

        this.server.respond();

        ok(!GH.Dialogs.CompleteParentDialog.showDialog.called, "should not call the success method");
        ok(GH.WorkView.hideLoadingPool.called, "should hide the loading pool");
    }));

    test("resolveParentTask: Failed second request does nothing (no specific error handling)", sinon.test(function () {
        this.stub(GH.Dialogs.CompleteParentDialog, "showDialog");
        this.stub(GH.WorkView, "hideLoadingPool");

        GH.Test.respondToGetWith200(this, '/xboard/issue/subtasksInFinalColumn.json', { subtasksInFinalColumn: true });
        GH.Test.respondToGetWith500(this, '/xboard/issue/transitions.json', {});

        WorkController.resolveParentTask({ id: 1 });

        this.server.respond();

        ok(!GH.Dialogs.CompleteParentDialog.showDialog.called, "should not call the success method");
        ok(!GH.WorkView.hideLoadingPool.called, "should not hide the loading pool");
    }));

    test("loadIssue: Successful Ajax request re-renders the issue", sinon.test(function () {
        this.stub(GH.SwimlaneView, "rerenderIssue");
        this.stub(GH.GridDataController, "getModel").returns({ updateIssueData: function updateIssueData() {} });

        GH.Test.respondToGetWith200(this, '/xboard/work/issue.json', {});

        WorkController.loadIssue();

        this.server.respond();

        ok(GH.SwimlaneView.rerenderIssue.called, "Called success method");
    }));

    test("loadIssue: Failed request does nothing (no specific error handling)", sinon.test(function () {
        this.stub(GH.SwimlaneView, "rerenderIssue");

        GH.Test.respondToGetWith500(this, '/xboard/work/issue.json', {});

        WorkController.loadIssue();

        this.server.respond();

        ok(!GH.SwimlaneView.rerenderIssue.called, "should not call success method");
    }));

    test("toggleEpicsShowOnWorkboard: Toggle updates board", sinon.test(function () {
        this.stub(WorkController, "updateUI");
        this.stub(GH.ViewActions, "toggleEpicsShowOnRapidBoard");

        WorkController.toggleEpicsShowOnWorkboard();

        ok(WorkController.updateUI.called, "should call update board");
    }));

    test("renderNoMappedColumnsMsg: The method should stop when the empty state message exist", sinon.test(function () {
        // add a pseudo-div with class name "ghx-no-active-sprint-message" and test
        this.getQunitFixture().append("<div class='ghx-no-active-sprint-message'></div>");

        this.stub(GH.RapidBoard.State, "getRapidViewId");

        WorkController.renderNoMappedColumnsMsg();

        ok(!GH.RapidBoard.State.getRapidViewId.called, "Should not called getRapidViewId");
    }));

    test("renderNoMappedColumnsMsg: The method should stop when the container doesn't exist", sinon.test(function () {

        this.stub(GH.RapidBoard.State, "getRapidViewId");

        WorkController.renderNoMappedColumnsMsg();

        ok(!GH.RapidBoard.State.getRapidViewId.called, "Should not getRapidViewId");
    }));

    test("renderNoMappedColumnsMsg: The second row should NOT has hidden class when canEdit is true", sinon.test(function () {
        // add a pseudo-container with id "ghx-work"
        this.getQunitFixture().append("<div id='ghx-work'></div>");
        var deferred = new AJS.$.Deferred();

        this.stub(GH.RapidBoard.State, "getRapidViewId");
        this.stub(GH.RapidViewConfig, "fetchConfiguration").returns(deferred.promise());

        WorkController.renderNoMappedColumnsMsg();

        deferred.resolve({ canEdit: true });

        ok(GH.RapidBoard.State.getRapidViewId.called, "Called success method");
        ok(!WorkController.activeSprintEmptyMsg.secondLineView.hasClass("hidden"), "The link should be visible");
    }));

    test("renderNoMappedColumnsMsg: The second row should has hidden class when canEdit is false", sinon.test(function () {
        // add a pseudo-container with id "ghx-work"
        this.getQunitFixture().append("<div id='ghx-work'></div>");
        var deferred = new AJS.$.Deferred();

        this.stub(GH.RapidBoard.State, "getRapidViewId");
        this.stub(GH.RapidViewConfig, "fetchConfiguration").returns(deferred.promise());

        WorkController.renderNoMappedColumnsMsg();

        deferred.resolve({ canEdit: false });

        ok(GH.RapidBoard.State.getRapidViewId.called, "Called success method");
        ok(WorkController.activeSprintEmptyMsg.secondLineView.hasClass("hidden"), "The link should NOT be visible");
    }));

    module("loadRapidViewConfig tests", {
        setup: function setup() {
            this.sandbox = sinon.sandbox.create();

            this.jiraAnalytics = require('jira/analytics');
            this.sandbox.stub(this.jiraAnalytics, "send");

            this.kanplanFeatureService = require('jira-agile/rapid/ui/kanplan/kan-plan-feature-service');
            this.sandbox.stub(this.kanplanFeatureService, "shouldShowKanbanBacklog").returns("shouldShowKanbanBacklog");
            this.sandbox.stub(this.kanplanFeatureService, "shouldShowEpicsPanel").returns(true);

            this.issuesMap = {
                1: "one",
                2: 'two'
            };

            this.dataModel = {
                getIssues: function () {
                    return this.issuesMap;
                }.bind(this),
                data: {
                    columns: [{ stats: { total: 123 } }]
                },
                epicData: { epicsCount: 234 }
            };
            this.sandbox.stub(GH.GridDataController, "getModel").returns(this.dataModel);

            this.sandbox.stub(GH.RapidBoard.State, "getRapidViewId").returns("rapidViewId");
        },
        teardown: function teardown() {
            this.sandbox.restore();
        },

        setBoardTypeToKanban: function setBoardTypeToKanban() {
            this.sandbox.stub(GH.RapidBoard.State, "getBoardType").returns("kanban");
            this.sandbox.stub(GH.RapidBoard.State, "isKanbanBoard").returns(true);
        },

        setBoardTypeToScrum: function setBoardTypeToScrum() {
            this.sandbox.stub(GH.RapidBoard.State, "getBoardType").returns("scrum");
            this.sandbox.stub(GH.RapidBoard.State, "isKanbanBoard").returns(false);
        }
    });

    test("loadRapidViewConfig should send analytics events for kanban boards", function () {
        var deferred = $.Deferred();
        var initializeWorkModeDeferred = $.Deferred();

        this.stub(GH.RapidViewConfig, "fetchConfiguration").returns(deferred.promise());
        this.stub(WorkController, "initializeWorkMode").returns(initializeWorkModeDeferred.promise());
        this.setBoardTypeToKanban();

        WorkController.loadRapidViewConfig();

        ok(this.jiraAnalytics.send.notCalled, "Analytics should not be sent yet");

        deferred.resolve();

        ok(this.jiraAnalytics.send.notCalled, "Analytics should not be sent yet");

        initializeWorkModeDeferred.resolve(this.dataModel);

        ok(this.jiraAnalytics.send.calledOnce, "Method should be called when all promises are successfully resolved");
        var args = this.jiraAnalytics.send.getCall(0).args[0];

        deepEqual(args, {
            name: 'jira-software.kanban.work.view',
            properties: {
                boardId: "rapidViewId",
                issuesCount: 2,
                isBacklogEnabled: "shouldShowKanbanBacklog",
                firstColumnIssueCount: 123,
                epicsPanelEnabled: true,
                epicsCount: 234
            }
        }, "Properties should be equal");
    });

    test("loadRapidViewConfig should send analytics events for scrum boards", function () {
        var deferred = $.Deferred();
        var initializeWorkModeDeferred = $.Deferred();

        this.stub(GH.RapidViewConfig, "fetchConfiguration").returns(deferred.promise());
        this.stub(WorkController, "initializeWorkMode").returns(initializeWorkModeDeferred.promise());
        this.setBoardTypeToScrum();
        this.issuesMap[3] = "three";

        WorkController.loadRapidViewConfig();

        ok(this.jiraAnalytics.send.notCalled, "Analytics should not be sent yet");

        deferred.resolve();

        ok(this.jiraAnalytics.send.notCalled, "Analytics should not be sent yet");

        initializeWorkModeDeferred.resolve(this.dataModel);

        ok(this.jiraAnalytics.send.calledOnce, "Method should be called when all promises are successfully resolved");
        var args = this.jiraAnalytics.send.getCall(0).args[0];

        deepEqual(args, {
            name: 'jira-software.scrum.work.view',
            properties: {
                boardId: "rapidViewId",
                issuesCount: 3,
                epicsCount: 234
            }
        }, "Properties should be equal");
        ok(this.kanplanFeatureService.shouldShowKanbanBacklog.notCalled, "This board is scrum so kanplan feature service should not be called");
    });

    test("loadRapidViewConfig should not send analytics when first promise reject", function () {
        var deferred = $.Deferred();
        var initializeWorkModeDeferred = $.Deferred();

        this.stub(GH.RapidViewConfig, "fetchConfiguration").returns(deferred.promise());
        this.stub(WorkController, "initializeWorkMode").returns(initializeWorkModeDeferred.promise());

        WorkController.loadRapidViewConfig();

        ok(this.jiraAnalytics.send.notCalled, "Analytics should not be send");

        deferred.reject();

        ok(this.jiraAnalytics.send.notCalled, "Analytics should not be send because promise was rejected");

        initializeWorkModeDeferred.resolve();

        ok(this.jiraAnalytics.send.notCalled, "Analytics should not be send because promise was rejected");
    });

    test("loadRapidViewConfig should not send analytics when second promise reject", function () {
        var deferred = $.Deferred();
        var initializeWorkModeDeferred = $.Deferred();

        this.stub(GH.RapidViewConfig, "fetchConfiguration").returns(deferred.promise());
        this.stub(WorkController, "initializeWorkMode").returns(initializeWorkModeDeferred.promise());

        WorkController.loadRapidViewConfig();

        ok(this.jiraAnalytics.send.notCalled, "Analytics should not be send");

        deferred.resolve();

        ok(this.jiraAnalytics.send.notCalled, "Analytics should not be send");

        initializeWorkModeDeferred.reject();

        ok(this.jiraAnalytics.send.notCalled, "Analytics should not be send because promise was rejected");
    });

    test("loadRapidViewConfig should not do anything if data model is not available", function () {
        var deferred = $.Deferred();
        var initializeWorkModeDeferred = $.Deferred();

        this.stub(GH.RapidViewConfig, "fetchConfiguration").returns(deferred.promise());
        this.stub(WorkController, "initializeWorkMode").returns(initializeWorkModeDeferred.promise());
        this.setBoardTypeToKanban();
        GH.GridDataController.getModel.returns(undefined);

        WorkController.loadRapidViewConfig();

        ok(this.jiraAnalytics.send.notCalled, "Analytics should not be sent");

        deferred.resolve();

        ok(this.jiraAnalytics.send.notCalled, "Analytics should not be sent");

        initializeWorkModeDeferred.resolve();

        ok(this.jiraAnalytics.send.notCalled, "Analytics should not be sent because model does not exists");
    });
});