AJS.test.require(["com.pyxis.greenhopper.jira:gh-test-common", "com.pyxis.greenhopper.jira:gh-rapid"], function() {
    const GridDataModel = require('jira-agile/rapid/ui/work/grid-data-model')

    function setupGridDataModelTest() {

        GH.GridDataModelTest = {};

        GH.GridDataModelTest.error = 'bad result';

        GH.GridDataModelTest.equal = function(actual, expected) {
            equal(actual, expected, GH.GridDataModelTest.error);
        };

        GH.GridDataModelTest.createModel = function() {
            /* 1  4
             * 2  -
             * -  5
             * 3  6
             */
            var columns = [
                {
                    id: 1
                },
                {
                    id: 2
                }
            ];
            var columnIssueMapping = {
                1: [ 'F-1', 'F-2', 'F-3' ],
                2: [ 'F-4', 'F-5', 'F-6' ]
            };
            var swimlanes = [
                {
                    id: 3
                },
                {
                    id: 4
                }
            ];
            var swimlaneIssueMapping = {
                3: [ 'F-1', 'F-2', 'F-4' ],
                4: [ 'F-3', 'F-5', 'F-6' ]
            };
            var issues = {
                'F-1': { key: 'F-1'},
                'F-2': { key: 'F-2'},
                'F-3': { key: 'F-3'},
                'F-4': { key: 'F-4'},
                'F-5': { key: 'F-5'},
                'F-6': { key: 'F-6'}
            };
            var order = [ 'F-1', 'F-2', 'F-3', 'F-4', 'F-5', 'F-6' ];
            var variables = {};
            return new GridDataModel(columns, columnIssueMapping, swimlanes, swimlaneIssueMapping, issues, order, variables);
        };
    }

    module('GridDataModel', {
        setup : function() {
            this.timer = sinon.useFakeTimers(0);
            setupGridDataModelTest();
        },
        teardown : function() {
            this.timer.restore();
        }
    });

    test('getCells', function() {
        var model = GH.GridDataModelTest.createModel();

        var cells = model.getCells();
        equal(cells[3][1].length, 2, 'Cell length not matching up');
        equal(cells[3][2].length, 1, 'Cell length not matching up');
        equal(cells[4][1].length, 1, 'Cell length not matching up');
        equal(cells[4][2].length, 2, 'Cell ength not matching up');
    });


    test('getFirstIssueKey', function() {
        var model = GH.GridDataModelTest.createModel();

        GH.RapidBoard.State.setSwimlaneCollapsed(3, false);
        GH.RapidBoard.State.setSwimlaneCollapsed(4, false);

        // without collapsed swimlane
        equal(model.getFirstIssueKey(), 'F-1', 'First not correct');

        // with collapsed swimlane
        GH.RapidBoard.State.setSwimlaneCollapsed(3, true);
        equal(model.getFirstIssueKey(), 'F-3', 'First not correct');
    });

    test('getLastIssueKey', function() {
        var model = GH.GridDataModelTest.createModel();

        GH.RapidBoard.State.setSwimlaneCollapsed(3, false);
        GH.RapidBoard.State.setSwimlaneCollapsed(4, false);

        // without collapsed swimlane
        equal(model.getLastIssueKey(), 'F-3', 'Last not correct');

        // with collapsed swimlane
        GH.RapidBoard.State.setSwimlaneCollapsed(4, true);
        equal(model.getLastIssueKey(), 'F-2', 'Last not correct');
    });

    test('getNextIssueKey', function() {
        var model = GH.GridDataModelTest.createModel();

        GH.RapidBoard.State.setSwimlaneCollapsed(3, false);
        GH.RapidBoard.State.setSwimlaneCollapsed(4, false);

        // without collapsed swimlane
        equal(model.getNextIssueKey('F-1'), 'F-2', 'Next not correct');
        equal(model.getNextIssueKey('F-2'), 'F-3', 'Next not correct');
        equal(model.getNextIssueKey('F-3'),false, 'Next not correct');

        // with collapsed swimlane
        GH.RapidBoard.State.setSwimlaneCollapsed(4, true);
        equal(model.getNextIssueKey('F-1'), 'F-2', 'Next not correct');
        equal(model.getNextIssueKey('F-2'), false, 'Next not correct');

        // wrong input
        equal(model.getNextIssueKey(), false, 'Next not correct');
    });

    test('getPreviousIssueKey', function() {
        var model = GH.GridDataModelTest.createModel();

        GH.RapidBoard.State.setSwimlaneCollapsed(3, false);
        GH.RapidBoard.State.setSwimlaneCollapsed(4, false);

        // without collapsed swimlane
        equal(model.getPreviousIssueKey('F-3'), 'F-2', 'Previous not correct');
        equal(model.getPreviousIssueKey('F-2'), 'F-1', 'Previous not correct');
        equal(model.getPreviousIssueKey('F1'), false, 'Previous not correct');

        // with collapsed swimlane
        GH.RapidBoard.State.setSwimlaneCollapsed(3, true);
        equal(model.getPreviousIssueKey('F-3'), false, 'Previous not correct');
        // TODO: this has changed with the introduction of parent swimlanes. Currently F-1 is returned
        //equal(model.getPreviousIssueKey('F-2'), false, 'Previous not correct');
        equal(model.getPreviousIssueKey('F-1'), false, 'Previous not correct');

        // wrong input
        equal(model.getPreviousIssueKey(), false, 'Previous not correct');
    });

    test('_getIssuePositionAndDataByKey', function() {
        var model = GH.GridDataModelTest.createModel();

        deepEqual(model._getIssuePositionAndDataByKey('F-6'), { swimlaneId:4, columnId: 2, issueIndex: 1, issue: { key: 'F-6' }}, 'Wrong result');
    });

    test('updateIssueData', function() {
        var model = GH.GridDataModelTest.createModel();
        var time = 1000*60*60*24*2;
        var data = { key: 'F-6', timeInColumn: { durationPreviously: time, enteredStatus: new Date().getTime() } };
        model.updateIssueData(data);
        deepEqual(model.getIssueDataByKey('F-6'), {key: 'F-6', timeInColumn: { durationPreviously: time, enteredStatus: new Date().getTime(), days : 2 } }, 'Wrong result');
    });

    test('getIssueKeyForColumnAtIndex', function() {
        var model = GH.GridDataModelTest.createModel();
        // fetches the closest issue in that cell
        equal(model.getIssueKeyForColumnAtIndex(3, 2, -1), 'F-4', 'Wrong result');
        equal(model.getIssueKeyForColumnAtIndex(3, 2, 0), 'F-4', 'Wrong result');
        equal(model.getIssueKeyForColumnAtIndex(3, 2, 1), 'F-4', 'Wrong result');
    });

    test('getIssueKeyInPreviousColumn', function() {
        var model = GH.GridDataModelTest.createModel();
        // fetches the closest issue in that cell
        equal(model.getIssueKeyInPreviousColumn('F-4', 0), 'F-1', 'Wrong result');
        equal(model.getIssueKeyInPreviousColumn('F-4', 1), 'F-2', 'Wrong result');
        equal(model.getIssueKeyInPreviousColumn('F-1', 1), false, 'Wrong result');
    });

    test('getIssueKeyInNextColumn', function() {
        var model = GH.GridDataModelTest.createModel();
        // fetches the closest issue in that cell
        equal(model.getIssueKeyInNextColumn('F-3', 0), 'F-5', 'Wrong result');
        equal(model.getIssueKeyInNextColumn('F-3', 1), 'F-6', 'Wrong result');
        equal(model.getIssueKeyInNextColumn('F-5', 1), false, 'Wrong result');
    });

    test('issueCounts', function() {
        var model = GH.GridDataModelTest.createModel();
        // fetches the closest issue in that cell
        equal(model.getSwimlaneIssueCount(3), 3, 'Wrong result');
        equal(model.getSwimlaneIssueCount(4), 3, 'Wrong result');
        equal(model.getColumnIssueCount(1), 3, 'Wrong result');
        equal(model.getColumnIssueCount(2), 3, 'Wrong result');
    });

    test('compareToModel', function() {
        var modelOne = GH.GridDataModelTest.createModel();
        var modelTwo = new GridDataModel(modelOne.data.columns, modelOne.data.columnIssueMapping, modelOne.data.swimlanes, modelOne.data.swimlaneIssueMapping, modelOne.data.issues, modelOne.data.order, modelOne.data.variables);

        // should be equal
        equal(modelOne.isModelDifferent(modelTwo), false, 'Models are equal');

        // test order change
        var changedOrder = _.flatten([modelOne.data.order[1]], [modelOne.data.order[0]], modelOne.data.order.slice(2));
        modelTwo = new GridDataModel(modelOne.data.columns, modelOne.data.columnIssueMapping, modelOne.data.swimlanes, modelOne.data.swimlaneIssueMapping, modelOne.data.issues, changedOrder, modelOne.data.variables);
        equal(modelOne.isModelDifferent(modelTwo), true, 'Models differ as order differs');

        // test issue change
        var changedIssues = _.clone(modelOne.data.issues);
        var firstKey = _.first(_.keys(changedIssues));
        var firstClone = _.clone(changedIssues[firstKey]);
        firstClone.summary = 'Test';
        changedIssues[firstKey] = firstClone;
        modelTwo = new GridDataModel(modelOne.data.columns, modelOne.data.columnIssueMapping, modelOne.data.swimlanes, modelOne.data.swimlaneIssueMapping, changedIssues, modelOne.data.order, modelOne.data.variables);
        equal(modelOne.isModelDifferent(modelTwo), true, 'Models differ as one issue is different');

        // change the swimlane issue mapping
        var changedSwimlaneIssueMapping = _.clone(modelOne.data.swimlaneIssueMapping);
        changedSwimlaneIssueMapping = {
            3: [ 'F-1', 'F-2', 'F-4', 'F-6' ],
            4: [ 'F-3', 'F-5' ]
        };
        modelTwo = new GridDataModel(modelOne.data.columns, modelOne.data.columnIssueMapping, modelOne.data.swimlanes, changedSwimlaneIssueMapping, modelOne.data.issues, modelOne.data.order, modelOne.data.variables);
        equal(modelOne.isModelDifferent(modelTwo), true, 'Models differ as one issue is in a different swimlane');

        // change the column issue mapping
        var changedColumnIssueMapping = _.clone(modelOne.data.columnIssueMapping);
        changedColumnIssueMapping = {
            1: [ 'F-1', 'F-2', 'F-3', 'F-6' ],
            2: [ 'F-4', 'F-5' ]
        };
        modelTwo = new GridDataModel(modelOne.data.columns, changedColumnIssueMapping, modelOne.data.swimlanes, modelOne.data.swimlaneIssueMapping, modelOne.data.issues, modelOne.data.order, modelOne.data.variables);
        equal(modelOne.isModelDifferent(modelTwo), true, 'Models differ as one issue is in a different column');
    });
});
