/* globals
 * GH.VersionController, GH.PageIssueLinksModel
 */

/**
 * A module representing a SprintModel that contains the data for a single sprint.
 * @module jira-agile/rapid/ui/plan/sprint-model
 * @requires module:underscore
 * @requires module:jira-agile/rapid/ui/plan/issue-list-model
 */
define('jira-agile/rapid/ui/plan/sprint-model', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var IssueListModel = require('jira-agile/rapid/ui/plan/issue-list-model');

    /**
     * Contains the data for a single sprint
     */
    var SprintModel = function (sprintData, issues) {
        this.sprintData = sprintData;

        // create a list for the contained issues
        this.issueList = new IssueListModel('sprint-' + sprintData.id, issues, GH.VersionController.getVersionModel());
    };

    /**
     * Is the passed model of type sprint?
     * @param model
     * @return {boolean}
     */
    SprintModel.isSprintModel = function (model) {
        return model.getType() === 'sprint';
    };

    /**
     *  Contains id, name, etc
     * @returns {{id: number, name: string}}
     */
    SprintModel.prototype.getSprintData = function () {
        return this.sprintData;
    };

    /**
     * Updates the sprint data
     */
    SprintModel.prototype.setSprintData = function (sprintData) {
        this.sprintData = sprintData;
    };

    /**
     * Contains the issues contained in this sprint
     *
     * @return {IssueListModel}
     */
    SprintModel.prototype.getIssueList = function () {
        return this.issueList;
    };

    /**
     * Get the id of the model
     */
    SprintModel.prototype.getId = function () {
        return this.sprintData.id;
    };

    /**
     * Get the type of the model
     */
    SprintModel.prototype.getType = function () {
        return 'sprint';
    };

    /**
     * Get the name of the model
     */
    SprintModel.prototype.getName = function () {
        return this.sprintData.name;
    };

    /**
     * Get the id of the sprint
     */
    SprintModel.prototype.getSprintId = function () {
        return this.sprintData.id;
    };

    /**
     * Is this a future sprint
     */
    SprintModel.prototype.isFutureSprint = function () {
        return this.sprintData.state === 'FUTURE';
    };

    /**
     * Is this an active sprint
     */
    SprintModel.prototype.isActiveSprint = function () {
        return this.sprintData.state === 'ACTIVE';
    };

    /*
     * Returns the count of linked pages for the sprint
     */
    SprintModel.prototype.getLinkedPagesCount = function () {
        var linkedPagesCountFromSprint = this.sprintData.linkedPagesCount;
        var linkedPagesCountFromModel = GH.PageIssueLinksModel.list(this.sprintData.id).length;
        var linkedPagesCount = linkedPagesCountFromSprint;
        if (!_.isNumber(linkedPagesCountFromSprint) || linkedPagesCountFromSprint < linkedPagesCountFromModel) {
            linkedPagesCount = linkedPagesCountFromModel;
        }
        return linkedPagesCount;
    };

    return SprintModel;
});