/* globals
 * GH.Tooltip, GH.EpicController, GH.RapidBoard, GH.tpl
 */

/**
 * This module representing SprintBacklogView
 * @module jira-agile/rapid/ui/plan/sprint-backlog-view
 * @requires module:jquery
 * @requires module:jira-agile/rapid/ui/plan/sprint-controller
 * @requires module:jira-agile/rapid/ui/plan/backlog-model
 * @requires module:jira-agile/rapid/ui/plan/backlog-statistics
 * @requires module:jira-agile/rapid/ui/work/issue-list-util
 * @requires module:jira-agile/rapid/ui/plan/backlog-controller
 */
define('jira-agile/rapid/ui/plan/sprint-backlog-view', ['require', 'underscore'], function (require, _) {
    'use strict';

    var $ = require('jquery');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var SprintController = require('jira-agile/rapid/ui/plan/sprint-controller');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var BacklogStatistics = require('jira-agile/rapid/ui/plan/backlog-statistics');
    var IssueListUtil = require('jira-agile/rapid/ui/plan/issue-list-util');
    var SubtasksExpandingController = require('jira-agile/rapid/ui/plan/subtasks-expanding-controller');
    var BacklogController;

    // Resolve circular dependency
    GlobalEvents.on("pre-initialization", function () {
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
    });

    /**
     * Backlog view component.
     */
    var SprintBacklogView = {};

    SprintBacklogView.containerSelector = '.ghx-backlog-group';

    /**
     * Initializes this view
     */
    SprintBacklogView.init = function () {
        // Add sprint button handler
        $(document).on('click', '.js-add-sprint', SprintController.addSprint);
    };

    /**
     * Draws the backlog
     */
    SprintBacklogView.renderBacklog = function (issueRenderData) {
        var html = _getBacklogHtml(issueRenderData);
        var $container = $('.ghx-backlog-group');
        $container.append($(html));
        enableTooltipForCreateSprintButton();
    };

    /**
     * Updates the rendered backlog
     */
    SprintBacklogView.updateBacklog = function () {
        // load the issue render data required for drawing
        var issueRenderData = BacklogController.calculateIssueRenderData();

        // re-render backlog
        var html = _getBacklogHtml(issueRenderData);
        var $container = $('.ghx-backlog-group');
        $container.empty().append($(html));
        enableTooltipForCreateSprintButton();
    };

    /**
     * Updates the backlog header & statistics
     */
    SprintBacklogView.updateBacklogHeader = function () {
        var $newHeaderContent = $(_renderBacklogHeader()).html();
        $('.js-marker-backlog-header').html($newHeaderContent);
        enableTooltipForCreateSprintButton();
    };

    /**
     * Calculates the issue list stats for a backlog
     */
    SprintBacklogView.calculateBacklogIssueListStats = function (backlogIssueList) {
        var issueListStats = BacklogStatistics.getIssueListStats(backlogIssueList, false);

        if (GH.RapidBoard.State.isKanbanBoard()) {
            issueListStats = _.extend({},
                issueListStats,
                BacklogStatistics.calculateStatisticsFieldValue(backlogIssueList, BacklogController.rapidViewConfig.statisticConfig))
        }
        return issueListStats;
    };

    /**
     * Get the html for the complete backlog
     */
    var _getBacklogHtml = function (issueRenderData) {
        // put together all required data
        var backlogModel2 = BacklogModel.getBacklogModel2();
        var backlogIssueList = backlogModel2.getIssueList();
        var issueListStats = SprintBacklogView.calculateBacklogIssueListStats(backlogIssueList);

        var visibleRanks = backlogIssueList.getVisibleRankables();
        var issuesMap = backlogIssueList.getAllIssues();
        var issues = IssueListUtil.issueListToTree(visibleRanks.map(function(key) {
            return issuesMap[key];
        }), BacklogModel.getAllIssuesWithMissingParents());

        IssueListUtil.applySubtaskCount(issues, backlogIssueList, issueRenderData.hiddenIssues);

        SubtasksExpandingController.applyExpandedState(issues);

        // render
        var params = {
            issueKeys: visibleRanks,
            issues: issues,
            issueListStats: issueListStats,
            epics: GH.EpicController.getEpicModel().getEpicList().getAllIssues(), // epics required for labels
            issueRenderData: issueRenderData,
            statLabel: BacklogModel.estimationStatistic.name,
            isRankable: BacklogModel.isRankable(),
            canManageSprints: BacklogModel.canManageSprints(),
            showEpic: GH.RapidBoard.State.getEpicShownOnRapidBoard(),
            shouldDisplayCreateSprintButton: GH.RapidBoard.State.isScrumBoard(),
            extraClasses: GH.RapidBoard.State.isKanbanBoard() ? 'ghx-kanban-backlog' : undefined,
            isKanbanBoard: GH.RapidBoard.State.isKanbanBoard()
        };

        return GH.tpl.sprintbacklogview.renderBacklogIssues(params);
    };

    /**
     * Get the html for the backlog header
     */
    var _renderBacklogHeader = function () {
        // put together all required data
        var backlogModel2 = BacklogModel.getBacklogModel2();
        var backlogIssueList = backlogModel2.getIssueList();
        var issueListStats = SprintBacklogView.calculateBacklogIssueListStats(backlogIssueList);

        // render
        var params = {
            issueListStats: issueListStats,
            isRankable: BacklogModel.isRankable(),
            canManageSprints: BacklogModel.canManageSprints(),
            shouldDisplayCreateSprintButton: GH.RapidBoard.State.isScrumBoard()
        };
        return GH.tpl.sprintbacklogview.renderBacklogHeader(params);
    };

    /**
     * Enable tipsy with default behaviour for the create sprint button,
     * since we want to take the advantage of default delayOut
     * interval so that user can click on the link within the tipsy.
     */
    function enableTooltipForCreateSprintButton() {
        $('.js-add-sprint').tooltip({
            html: true,
            title: 'data-tooltip'
        });
    }

    return SprintBacklogView;
});