AJS.test.require('com.pyxis.greenhopper.jira:gh-rapid', function () {

    var IssueListUtil = require('jira-agile/rapid/ui/plan/issue-list-util');
    var _ = require('underscore');

    module('issueListToTree', {
        /**
         *
         * @param input
         * @param expected
         */
        assertResult: function assertResult(input, expected, hiddenIssues) {
            if (typeof hiddenIssues === 'undefined') {
                hiddenIssues = {};
            }

            var actual = IssueListUtil.issueListToTree(input, {}, hiddenIssues);

            deepEqual(actual, expected);
        }
    });

    test('Simple data input', function () {
        var input = [{
            key: '12'
        }, {
            key: '13'
        }, {
            key: '14'
        }];

        this.assertResult(input, input);
    });

    test('Complex data input', function () {
        var input = [{
            key: '12'
        }, {
            key: '13',
            parentKey: '12'
        }, {
            key: '14'
        }];
        var expected = [{
            key: '12',
            subTasks: [{
                key: '13',
                parentKey: '12'
            }]
        }, {
            key: '14'
        }];

        this.assertResult(input, expected);
    });

    test('Fake parent', function () {
        var input = [{
            key: '12'
        }, {
            key: '13',
            parentKey: '1'
        }, {
            key: '14'
        }, {
            key: '15',
            parentKey: '1'
        }, {
            key: '16',
            parentKey: '2'
        }];
        var expected = [{
            key: '12'
        }, {
            fakeParent: true,
            key: '1',
            subTasks: [{
                key: '13',
                parentKey: '1'
            }, {
                key: '15',
                parentKey: '1'
            }]
        }, {
            key: '14'
        }, {
            fakeParent: true,
            key: '2',
            subTasks: [{
                key: '16',
                parentKey: '2'
            }]
        }];

        this.assertResult(input, expected);
    });

    test('Parent ranked below issue', function () {
        var input = [{
            key: '12'
        }, {
            key: '13',
            parentKey: '14'
        }, {
            key: '14'
        }, {
            key: '15',
            parentKey: '14'
        }];

        var expected = [{
            key: '12'
        }, {
            key: '14',
            subTasks: [{
                key: '13',
                parentKey: '14'
            }, {
                key: '15',
                parentKey: '14'
            }]
        }];

        this.assertResult(input, expected);
    });

    test('Subtasks with or without parents', function () {
        var input = [{
            key: '14'
        }, {
            key: '13',
            parentKey: '14'
        }, {
            key: '17',
            parentKey: '1'
        }, {
            key: '15',
            parentKey: '14'
        }];

        var expected = [{
            key: '14',
            subTasks: [{
                key: '13',
                parentKey: '14'
            }, {
                key: '15',
                parentKey: '14'
            }]
        }, {
            key: '1',
            fakeParent: true,
            subTasks: [{
                key: '17',
                parentKey: '1'
            }]
        }];

        this.assertResult(input, expected, {
            '13': true,
            '17': true,
            '15': true
        });
    });

    module('#applySubtasksCount', {

        setup: function setup() {
            this.issueListModel = {
                getSubtasksKeys: sinon.stub()
            };
        },

        mockSubtasksKeys: function mockSubtasksKeys(key, number) {
            this.issueListModel.getSubtasksKeys.withArgs(key).returns(_(number).times(function (number) {
                return number;
            }));
        }

    });

    function mockIssue(key, subtasksList) {
        return {
            key: key,
            subTasks: subtasksList ? subtasksList : []
        };
    }

    test('subtasksCount should contain number of all subtasks of issue, no matter of parent', function () {
        var ISSUE_ONE = "one";
        var ISSUE_TWO = "two";
        var ISSUE_THREE = "three";

        var issueList = [mockIssue(ISSUE_ONE), mockIssue(ISSUE_TWO), mockIssue(ISSUE_THREE)];

        this.mockSubtasksKeys(ISSUE_ONE, 10);
        this.mockSubtasksKeys(ISSUE_TWO, 100);
        this.mockSubtasksKeys(ISSUE_THREE, 42);

        IssueListUtil.applySubtaskCount(issueList, this.issueListModel, {});

        equal(issueList[0].subtasksCount, 10);
        equal(issueList[1].subtasksCount, 100);
        equal(issueList[2].subtasksCount, 42);
    });

    test('subtasksFiltered should be set only if all subtasks all filtered', function () {
        var ALL_FILTERED = "all filtered";
        var NOT_ALL_FILTERED = "not all filtered";

        var SUB_HIDDEN_1 = 'sub hidden 1';
        var SUB_HIDDEN_2 = 'sub hidden 2';

        var SUB_HIDDEN_3 = 'sub hidden 3';
        var SUB_NOT_HIDDEN = 'sub not hidden';

        var issueList = [mockIssue(ALL_FILTERED, [mockIssue(SUB_HIDDEN_1), mockIssue(SUB_HIDDEN_2)]), mockIssue(NOT_ALL_FILTERED, [mockIssue(SUB_HIDDEN_3), mockIssue(SUB_NOT_HIDDEN)])];

        this.mockSubtasksKeys(ALL_FILTERED, 2);
        this.mockSubtasksKeys(NOT_ALL_FILTERED, 2);

        var hidden = {};
        hidden[SUB_HIDDEN_1] = true;
        hidden[SUB_HIDDEN_2] = true;
        hidden[SUB_HIDDEN_3] = true;
        IssueListUtil.applySubtaskCount(issueList, this.issueListModel, hidden);

        ok(issueList[0].subtasksFiltered);
        ok(!issueList[1].subtasksFiltered);
    });

    test('hiddenSubtasks should include issues hidden by search and hidden by filters', function () {
        var ISSUE = "issue";

        var HIDDEN_SUB = 'hidden sub';
        var VISIBLE_SUB = 'visible sub';

        var issueList = [mockIssue(ISSUE, [mockIssue(HIDDEN_SUB), mockIssue(VISIBLE_SUB)])];

        this.mockSubtasksKeys(ISSUE, 5);

        var hidden = {};
        hidden[HIDDEN_SUB] = true;

        IssueListUtil.applySubtaskCount(issueList, this.issueListModel, hidden);

        equal(issueList[0].hiddenSubtasks, 4);
    });

    test('hiddenByQuickFilters should only include issues hidden by quick filters', function () {
        var ISSUE = "issue";

        var HIDDEN_SUB = 'hidden sub';
        var VISIBLE_SUB = 'visible sub';

        var issueList = [mockIssue(ISSUE, [mockIssue(HIDDEN_SUB), mockIssue(VISIBLE_SUB)])];

        this.mockSubtasksKeys(ISSUE, 5);

        var hidden = {};
        hidden[HIDDEN_SUB] = true;

        IssueListUtil.applySubtaskCount(issueList, this.issueListModel, hidden);

        equal(issueList[0].hiddenByQuickFiltersSubtasks, 3);
    });
});