/* globals
 * GH.EpicController
 */

/**
 * @module jira-agile/rapid/ui/plan/plan-issue-list-filtering
 * @requires module:underscore
 */
define('jira-agile/rapid/ui/plan/plan-issue-list-filtering', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');

    /**
     * Encapulates the issue list filtering logic
     */
    var PlanIssueListFiltering = {};


    /** The epic key when we want explicitly filter issue not associated to epic */
    PlanIssueListFiltering.EPIC_FILTER_KEY_NONE = 'none';

    /** The version key used to filter for no associated versions */
    PlanIssueListFiltering.VERSION_FILTER_ID_NONE = 'none';

    /**
     * A map of all currently active filters on the backlog model. Includes Epic Filters and Instant Filter.
     */
    PlanIssueListFiltering.allFilters = {};

    //
    // Misc
    //

    /**
     * Is the passed key matching the none epics key?
     * @param key the currently filtered epic key
     * @return {boolean}
     */
    PlanIssueListFiltering.isNoneEpicFilterKey = function (key) {
        return key === PlanIssueListFiltering.EPIC_FILTER_KEY_NONE;
    };

    /**
     * Is the passed id matching the none version id
     * @param id
     * @return {boolean}
     */
    PlanIssueListFiltering.isNoneVersionFilterId = function (id) {
        return id === PlanIssueListFiltering.VERSION_FILTER_ID_NONE;
    };

    //
    // Filter management
    //

    /**
     * Sets or removes the filter for a given key.
     */
    PlanIssueListFiltering.setFilter = function (key, filterFn) {
        if (!filterFn) {
            delete PlanIssueListFiltering.allFilters[key];
        } else {
            PlanIssueListFiltering.allFilters[key] = filterFn;
        }
    };

    /**
     * Clears all current filters
     * @return {Boolean} if filters have been removed
     */
    PlanIssueListFiltering.clearAllFilters = function () {
        var hadFilters = !_.isEmpty(_.keys(PlanIssueListFiltering.allFilters));
        PlanIssueListFiltering.allFilters = {};
        return hadFilters;
    };

    /**
     * Combines all current filters into one filter that can be applied to an issue list. The result of the combined function
     * will only be true for an issue if all functions return true.
     * @return {Function}
     * @private
     */
    PlanIssueListFiltering._getCompoundFilter = function () {
        return function (issue, issueCollection) {
            // _.any returns true for matches which in this case means an unmatched filter
            // So we want to return the opposite
            var anyUnmatched = _.any(PlanIssueListFiltering.allFilters, function (filterFn) {
                // return false if the issue matches, true otherwise
                return !filterFn(issue, issueCollection);
            });
            return !anyUnmatched;
        };
    };


    //
    // Setting different types of filters
    //

    /**
     * Applies a filter against issues in all lists for the specified epic key.
     *
     * If null key is specified, it clears the filter. If PlanIssueListFiltering.EPIC_FILTER_KEY_NONE is specified, it filters out
     * the issues affected to an epic.
     *
     * @param {string} epicKey the epic key, null or PlanIssueListFiltering.EPIC_FILTER_KEY_NONE
     * @return {Boolean} whether or not issues changed as a result of applying the filter
     */
    PlanIssueListFiltering.setEpicFilter = function (epicKey) {
        var epicFilterFn = null;
        if (PlanIssueListFiltering.isNoneEpicFilterKey(epicKey)) {
            epicFilterFn = function (issue) {
                var parentEpic = typeof issue.parent === 'object' && issue.parent.epic;
                return _.isEmpty(issue.epic) && !parentEpic;
            };
        } else if (PlanIssueListFiltering.isEpicIssueSelectable(epicKey)) {
            epicFilterFn = function (issue) {
                return issue.epic === epicKey || (issue.parent && issue.parent.epic === epicKey);
            };
        }

        return PlanIssueListFiltering.setFilter('epics', epicFilterFn);
    };

    /**
     * Is the current epic selectable?
     */
    PlanIssueListFiltering.isEpicIssueSelectable = function (issueKey) {
        var epicIssueList = GH.EpicController.getEpicModel().getEpicList();
        return (epicIssueList.isIssueValid(issueKey) && epicIssueList.isIssueVisible(issueKey));
    };

    PlanIssueListFiltering.setVersionFilter = function (versionId) {
        var versionFilterFn = null;
        if (PlanIssueListFiltering.isNoneVersionFilterId(versionId)) {
            versionFilterFn = function (issue) {
                return _.isEmpty(issue.fixVersions);
            };
        } else if (PlanIssueListFiltering.isVersionSelectable(versionId)) {
            versionFilterFn = function (issue) {
                return _.some(issue.fixVersions, function (issueFixVersionId) {
                    return issueFixVersionId === parseInt(versionId, 10);
                });
            };
        }

        return PlanIssueListFiltering.setFilter('versions', versionFilterFn);
    };

    PlanIssueListFiltering.isVersionSelectable = function (versionId) {
        // todo implement version visibility check
        return versionId != null;
    };

    /**
     * Applies the instant filter against issues in all lists. If null is specified, it clears the filter.
     * @param instantFilterFn the filter function or null
     * @return {Boolean} whether or not issues changed as a result of applying the filter
     */
    PlanIssueListFiltering.setInstantFilter = function (instantFilterFn) {
        PlanIssueListFiltering.setFilter('instant', instantFilterFn);
    };


    //
    // Applying the filter
    //


    /**
     * Applies the currently configured filters to all passed lists
     *
     * @param issueLists Array of issue list models
     * @return {Boolean} true if at least one issue changed as a result of applying the search
     */
    PlanIssueListFiltering.applyToIssueLists = function (issueLists) {
        // get filter to apply
        var compoundFilter = PlanIssueListFiltering._getCompoundFilter();

        // apply
        var changed = false;
        _.each(issueLists, function (issueList) {
            var result = issueList.applySearchFilter(compoundFilter);
            changed = changed || result;
        });

        return changed;
    };

    /**
     * Applies the configured filters on an issue list
     */
    PlanIssueListFiltering.applyToIssueList = function (issueList) {
        PlanIssueListFiltering.applyToIssueLists([issueList]);
    };

    return PlanIssueListFiltering;
});
