AJS.test.require(['com.pyxis.greenhopper.jira:gh-rapid'], function () {
    var $ = require('jquery');
    var _ = require('underscore');
    var Backbone = require('backbone');
    var SprintModel = require('jira-agile/rapid/ui/plan/sprint-model');
    var SPRINT_ID = 'sprint ID';

    module('transitionAndRankIssues', {
        setup: function setup() {
            this.sandbox = sinon.sandbox.create();

            this.context = AJS.test.mockableModuleContext();

            this.mockAjax();
            this.mockRankController();

            this.mockIssueMoveController();
            this.mockBacklogModel();
            this.mockBacklogController();
            this.mockPlanController();
            this.mockPlanDragAndDrop();
            this.mockPlanView();
            this.mockKanPlanDialogUtils();
            this.mockPlanControls();
            this.mockWorkDragAndDrop();
            this.mockNotification();
            this.mockSubtasksExpandingController();
            this.mockBacklogSelectionController();
            this.mockAnalyticsHelper();
            this.mockAnalytics();

            this.kanbanTransitionAndRank = this.context.require('jira-agile/rapid/ui/plan/kanban-transition-and-rank');
            this.kanbanTransitionAndRank._preInitialization();
        },

        teardown: function teardown() {
            this.sandbox.restore();
        },

        /**
         * Mocking
         */

        mockAjax: function mockAjax() {
            this.Ajax = {
                putResult: $.Deferred(),
                putResult2: $.Deferred()
            };
            this.sandbox.stub(GH.Ajax, 'put');

            this.Ajax.put = GH.Ajax.put;

            this.Ajax.put.onCall(0).returns(this.Ajax.putResult);
            this.Ajax.put.onCall(1).returns(this.Ajax.putResult2);
        },

        mockRankController: function mockRankController() {
            this.RankController = {
                rankIssuesResult: $.Deferred()
            };
            this.sandbox.stub(GH.RankController, 'rankIssues').returns(this.RankController.rankIssuesResult);

            this.RankController.rankIssues = GH.RankController.rankIssues;
        },

        mockWorkDragAndDrop: function mockWorkDragAndDrop() {
            this.WorkDragAndDrop = {};

            this.sandbox.stub(GH.WorkDragAndDrop, 'executeWorkflowTransition');

            this.WorkDragAndDrop.executeWorkflowTransition = GH.WorkDragAndDrop.executeWorkflowTransition;
        },

        mockNotification: function mockNotification() {
            this.Notification = {};

            this.sandbox.stub(GH.Notification, 'showSuccess');

            this.Notification.showSuccess = GH.Notification.showSuccess;
        },

        mockIssueMoveController: function mockIssueMoveController() {
            this.IssueMoveController = {
                calculateIssueMoveModel: sinon.stub(),
                isMoveToBacklog: sinon.stub(),
                isOnlyRankOperation: sinon.stub().returns(false)
            };
            this.context.mock('jira-agile/rapid/ui/plan/issue-move-controller', this.IssueMoveController);
            this.IssueMoveController.calculateIssueMoveModel.returns({ changes: [{ targetSprint: { name: SPRINT_ID } }] });
        },

        mockBacklogModel: function mockBacklogModel() {
            this.BacklogModel = {
                getIssueData: function getIssueData(issueKey) {
                    return {
                        id: issueKey
                    };
                },
                getIssueDataForId: function getIssueDataForId(id) {
                    return undefined;
                },
                getIssueIdForKey: function getIssueIdForKey(issueKey) {
                    return issueKey;
                },
                getRankCustomFieldId: function getRankCustomFieldId() {
                    return 1;
                },
                filterOnlySubtasks: function filterOnlySubtasks(list) {
                    return [];
                },
                getSprintModels: function getSprintModels() {
                    return [new SprintModel({ column: { id: 1 } })];
                }
            };
            this.context.mock('jira-agile/rapid/ui/plan/backlog-model', this.BacklogModel);
        },

        mockBacklogController: function mockBacklogController() {
            this.BacklogController = {
                moveIssues: sinon.stub(),
                reorderIssuesInSprint: sinon.stub(),
                updateIssues: sinon.stub()
            };
            this.context.mock('jira-agile/rapid/ui/plan/backlog-controller', this.BacklogController);
        },

        mockSubtasksExpandingController: function mockSubtasksExpandingController() {
            this.SubtasksExpandingController = {
                toggleExpandStateForKey: sinon.stub()
            };
            this.context.mock('jira-agile/rapid/ui/plan/subtasks-expanding-controller', this.SubtasksExpandingController);
        },

        mockBacklogSelectionController: function mockBacklogSelectionController() {
            this.BacklogSelectionController = {
                removeFromSelectionIssuesInDifferentModel: sinon.stub()
            };
            this.context.mock('jira-agile/rapid/ui/plan/backlog-selection-controller', this.BacklogSelectionController);
        },

        mockAnalyticsHelper: function mockAnalyticsHelper() {
            this.AnalyticsHelper = {
                baseEventData: sinon.stub().returns({}),
                getSrcColumnName: sinon.stub().returns('src'),
                getDestColumnName: sinon.stub().returns('dest')

            };
            this.context.mock('jira-agile/rapid/ui/plan/analytics-helper', this.AnalyticsHelper);
        },
        mockAnalytics: function mockAnalytics() {
            this.Analytics = {
                send: sinon.stub()
            };
            this.context.mock('jira/analytics', this.Analytics);
        },

        mockPlanController: function mockPlanController() {
            this.PlanController = {
                reload: function reload(callback) {
                    callback();
                }
            };
            this.sandbox.spy(this.PlanController, 'reload');
            this.context.mock('jira-agile/rapid/ui/plan/plan-controller', this.PlanController);
        },

        mockPlanDragAndDrop: function mockPlanDragAndDrop() {
            this.PlanDragAndDrop = {
                refreshDetailsView: sinon.stub()
            };
            this.context.mock('jira-agile/rapid/ui/plan/plan-drag-and-drop', this.PlanDragAndDrop);
        },

        mockPlanView: function mockPlanView() {
            this.PlanView = {
                showLoadingBacklog: sinon.stub(),
                hideLoadingBacklog: sinon.stub()
            };
            this.context.mock('jira-agile/rapid/ui/plan/plan-view', this.PlanView);
        },

        mockKanPlanDialogUtils: function mockKanPlanDialogUtils() {
            var transitionStatusDialog = {};
            _.extend(transitionStatusDialog, Backbone.Events);
            transitionStatusDialog.show = sinon.stub();
            transitionStatusDialog.updateHeight = sinon.stub();

            this.KanPlanDialogUtils = {
                createTransitionStatusDialog: sinon.stub().returns(transitionStatusDialog),
                createTransitionStatusDialogElement: transitionStatusDialog
            };
            this.context.mock('jira-agile/rapid/ui/kanplan/kan-plan-dialog-utils', this.KanPlanDialogUtils);
        },

        mockPlanControls: function mockPlanControls() {
            this.PlanControls = {
                quickfilters: {
                    getActiveQuickFilters: function getActiveQuickFilters() {
                        return [2, 5];
                    }
                }
            };
            this.context.mock('jira-agile/rapid/ui/plan/plan-controls', this.PlanControls);
        },

        /**
         * Assert
         */
        assertRankControllerRankIssuesNotCalled: function assertRankControllerRankIssuesNotCalled() {
            ok(!this.RankController.rankIssues.called, "RankController.rankIssues was not called");
        },

        assertUpdateIssuesTriggered: function assertUpdateIssuesTriggered(issueKeyOrKeys) {
            ok(this.BacklogController.updateIssues.calledWith(issueKeyOrKeys));
        },

        assertNoCalculatingIssuesOrder: function assertNoCalculatingIssuesOrder() {
            ok(!this.BacklogController.reorderIssuesInSprint.called, "calculateNewIssuesOrder was not passed so issues in sprint are not reordered");
        },

        assertBacklogLoading: function assertBacklogLoading() {
            ok(this.PlanView.showLoadingBacklog.called, "Backlog should always change it's state to loading");
            ok(this.PlanView.hideLoadingBacklog.called, "and should always revert to visible state");
        },

        assertSelectionChangesCalled: function assertSelectionChangesCalled() {
            ok(this.BacklogSelectionController.removeFromSelectionIssuesInDifferentModel.calledOnce, "BacklogSelectionController#removeFromSelectionIssuesInDifferentModel was called once");
            ok(this.BacklogSelectionController.removeFromSelectionIssuesInDifferentModel.withArgs(SPRINT_ID), "BacklogSelectionController#removeFromSelectionIssuesInDifferentModel was called with argument " + SPRINT_ID);
        },

        assertRankingAnalytics: function assertRankingAnalytics(issuesCount) {
            var properties = {
                src: 'src',
                dest: 'dest',
                rankOnly: false,
                issuesCount: issuesCount,
                subtasksCount: 0
            };
            this.Analytics.send.calledWith({
                name: 'jira-software.plan.issuecard.drop.success',
                properties: properties
            });
        },

        assertNoRankingAnalytics: function assertNoRankingAnalytics() {
            var callCount = this.Analytics.send.callCount;
            for (var i = 0; i < callCount; i++) {
                var callArgs = this.Analytics.send.getCall(i).args[0];
                notEqual(callArgs.name, 'jira-software.plan.issuecard.drop.success', "Ranking event should not be called (" + i + " call)");
            }
        },

        /**
         * Builders
         */
        buildAjaxResponse: function buildAjaxResponse(possibleTransitions, updatedIssues) {
            return {
                success: {
                    possibleTransitions: possibleTransitions,
                    // in reality updatedIssues contains whole issues, not just keys,
                    // but for the sake of tests this is good enough
                    updatedIssues: updatedIssues
                }
            };
        }
    });

    test('calls RankController for rank only events', function () {
        var issueKeys = [1];
        this.IssueMoveController.isOnlyRankOperation.returns(true);

        this.kanbanTransitionAndRank.transitionAndRankIssues(issueKeys, SPRINT_ID);

        ok(!this.Ajax.put.called, "There was not put request - there was no transition");
        ok(this.RankController.rankIssues.calledOnce);
        ok(this.RankController.rankIssues.calledWith(this.BacklogModel.getRankCustomFieldId(), issueKeys, undefined, undefined));
        this.assertRankingAnalytics(1);
    });

    test('plan is reloaded when request fails', function () {
        var issueKeys = [1];

        this.kanbanTransitionAndRank.transitionAndRankIssues(issueKeys, SPRINT_ID);
        this.Ajax.putResult.reject({ error: [] });

        ok(this.Ajax.put.calledOnce, "Put request was called");
        ok(this.PlanController.reload.calledOnce, "Plan was reloaded because request failed");

        this.assertRankControllerRankIssuesNotCalled();
        this.assertNoRankingAnalytics();
    });

    test('does transition and rank operation when no transition choices in response', function () {
        var issueKeys = [1, 2];

        this.kanbanTransitionAndRank.transitionAndRankIssues(issueKeys, SPRINT_ID);
        this.Ajax.putResult.resolve(this.buildAjaxResponse([], issueKeys));

        ok(this.PlanDragAndDrop.refreshDetailsView.calledOnce);
        this.assertSelectionChangesCalled();
        this.assertNoCalculatingIssuesOrder();
        this.assertBacklogLoading();
        this.assertUpdateIssuesTriggered(issueKeys);
        this.assertRankControllerRankIssuesNotCalled();
        this.assertRankingAnalytics(2);
    });

    test('issues are re-ordered when required', function () {
        var issueKeys = [1, 2];
        this.BacklogController.moveIssues.returns(true);

        this.kanbanTransitionAndRank.transitionAndRankIssues(issueKeys, SPRINT_ID);
        this.Ajax.putResult.resolve(this.buildAjaxResponse([], issueKeys));

        ok(this.PlanDragAndDrop.refreshDetailsView.calledOnce);
        ok(this.BacklogController.reorderIssuesInSprint.called);
        this.assertSelectionChangesCalled();
        this.assertBacklogLoading();
        this.assertUpdateIssuesTriggered(issueKeys);
        this.assertRankControllerRankIssuesNotCalled();
        this.assertRankingAnalytics(2);
    });

    function defaultTransitions() {
        return [{
            id: 1,
            hasTransitionView: false,
            targetStatus: '1-target-status'
        }, {
            id: 2,
            hasTransitionView: true,
            targetStatus: '2-target-status'
        }];
    }

    test('shows multi-transition dialog and successfully transitions when there are multiple transition choices', function () {
        var issueKeys = [1, 2];

        this.kanbanTransitionAndRank.transitionAndRankIssues(issueKeys, SPRINT_ID);
        this.Ajax.putResult.resolve(this.buildAjaxResponse(defaultTransitions()));
        this.KanPlanDialogUtils.createTransitionStatusDialogElement.trigger('submit', 1);
        this.Ajax.putResult2.resolve(this.buildAjaxResponse([], issueKeys));

        ok(this.Ajax.put.calledTwice, "Put request was called twice - one with chosen transition, second one to get sure that there are multiple transitions");
        ok(this.PlanDragAndDrop.refreshDetailsView.calledOnce);

        ok(this.Notification.showSuccess.calledOnce);

        this.assertSelectionChangesCalled();
        this.assertNoCalculatingIssuesOrder();
        this.assertBacklogLoading();
        this.assertUpdateIssuesTriggered(issueKeys);
        this.assertRankControllerRankIssuesNotCalled();
        this.assertRankingAnalytics(2);
    });

    test('quick filters are passed to the transitionAndRank operation', function () {
        var issueKeys = [1, 2];

        this.kanbanTransitionAndRank.transitionAndRankIssues(issueKeys, SPRINT_ID);

        deepEqual(this.Ajax.put.args[0][0].data.activeQuickFilters, this.PlanControls.quickfilters.getActiveQuickFilters(), "Put request was called with active quick filters");
    });

    test('shows multi-transition dialog and handles selected transition when it has a view', function () {
        var issueKeys = [1, 2];

        this.kanbanTransitionAndRank.transitionAndRankIssues(issueKeys, SPRINT_ID);
        this.Ajax.putResult.resolve(this.buildAjaxResponse(defaultTransitions()));
        this.KanPlanDialogUtils.createTransitionStatusDialogElement.trigger('submit', 2);
        this.Ajax.putResult2.resolve([]);

        ok(this.Ajax.put.calledOnce);

        ok(this.WorkDragAndDrop.executeWorkflowTransition.calledOnce);
        ok(this.RankController.rankIssues.calledOnce);

        this.assertNoCalculatingIssuesOrder();
        this.assertBacklogLoading();
        this.assertRankingAnalytics(2);
    });

    test('shows multi-transition dialog and reloads board when dialog cancelled', function () {
        var issueKeys = [1, 2];

        this.kanbanTransitionAndRank.transitionAndRankIssues(issueKeys, SPRINT_ID);
        this.Ajax.putResult.resolve(this.buildAjaxResponse(defaultTransitions()));
        this.KanPlanDialogUtils.createTransitionStatusDialogElement.trigger('cancel', 1);

        ok(this.Ajax.put.calledOnce);
        ok(!this.PlanDragAndDrop.refreshDetailsView.called);
        ok(this.PlanController.reload.calledOnce);

        this.assertNoCalculatingIssuesOrder();
        this.assertBacklogLoading();
        this.assertRankControllerRankIssuesNotCalled();
        this.assertRankingAnalytics(2);
    });

    test('can handle transition with view', function () {
        var issueKeys = [1, 2];

        this.kanbanTransitionAndRank.transitionAndRankIssues(issueKeys, SPRINT_ID);
        this.Ajax.putResult.resolve(this.buildAjaxResponse([{
            id: 1
        }]));

        ok(this.Ajax.put.calledOnce);
        ok(!this.PlanDragAndDrop.refreshDetailsView.called);
        ok(!this.PlanController.reload.calledOnce);
        ok(this.WorkDragAndDrop.executeWorkflowTransition.calledOnce);
        ok(this.RankController.rankIssues.calledOnce);

        this.assertNoCalculatingIssuesOrder();
        this.assertBacklogLoading();
        this.assertRankingAnalytics(2);
    });

    test('expand only parents of issues that are not in selection', function () {
        var issueKeys = ['parent', 'subtask-with-parent-moved', 'subtask-without-parent-moved'];

        this.BacklogModel.getIssueData = sinon.stub();
        this.BacklogModel.getIssueData.withArgs(issueKeys[0]).returns({
            id: issueKeys[0]
        });
        this.BacklogModel.getIssueData.withArgs(issueKeys[1]).returns({
            id: issueKeys[1],
            parentKey: issueKeys[0]
        });
        this.BacklogModel.getIssueData.withArgs(issueKeys[2]).returns({
            id: issueKeys[2],
            parentKey: 'parent-key-was-not-moved'
        });

        this.kanbanTransitionAndRank.transitionAndRankIssues(issueKeys, SPRINT_ID);
        this.Ajax.putResult.resolve(this.buildAjaxResponse([]));

        ok(this.SubtasksExpandingController.toggleExpandStateForKey.calledOnce);
        ok(this.SubtasksExpandingController.toggleExpandStateForKey.calledWith(issueKeys[2], true));
    });
});