/* globals GH.Collection, GH.ExtraFieldsHelper */

/**
 * @typedef {string} IssueKey
 */

/**
 * @typedef {Object} IssueData
 */

/**
 * @typedef {string} OrderKey
 */

/**
 * @typedef {number} Index
 */

/**
 * @typedef {Object} Data held by this model
 * @property {string} id
 * @property {Object.<IssueKey, IssueData>} issueByKey
 * @property {string[]} order
 * @property {Object.<OrderKey, Index>} orderByKey
 * @property [IssueKey[]] visibleIssueKeys
 * @property {Object.<IssueKey, boolean>} visibilityByIssueKey
 * @property {Object.<IssueKey, boolean>} hiddenIssueKeys
 */

/**
 * @module jira-agile/rapid/ui/plan/issue-list-model
 * @requires module:underscore
 */
define('jira-agile/rapid/ui/plan/issue-list-model', ['require'], function (require) {
    'use strict';

    var SubtasksExpandingController = require('jira-agile/rapid/ui/plan/subtasks-expanding-controller');
    var _ = require('underscore');

    // TODO: remove all the marker related code from this model
    //       rankable could be renamed to "issueKey" too
    var IssueListModel = function (id, issues, fixVersions) {
        this.fixVersions = fixVersions;

        // init the data structure
        var data = this.data = {};
        data.id = id;
        data.issueByKey = {};
        data.order = [];
        data.visibleIssueKeys = new GH.Collection.OrderedSet();
        data.hiddenIssueKeys = {};
        data.parentSubtaskKeys = {};

        // now add all issues
        this.addIssues(issues);
    };


    /**
     * Adds issues to this model
     */
    IssueListModel.prototype.addIssues = function (issues) {
        var allIssues = this.data.issueByKey;
        var order = this.data.order;
        _.each(issues, function (issue) {
            if (this.fixVersions) {
                this.processFixVersion(issue);
            }
            var issueKey = issue.key;
            allIssues[issueKey] = issue;

            order.push(issueKey);
        }, this);

        // build/update the cache
        this.buildCache();
    };

    /**
     * Adds issues at the top of the list
     */
    IssueListModel.prototype.prependIssues = function (issues) {
        var allIssues = this.data.issueByKey;
        var order = this.data.order;
        _.each(issues, function (issue) {
            if (this.fixVersions) {
                this.processFixVersion(issue);
            }
            var issueKey = issue.key;
            allIssues[issueKey] = issue;
        }, this);

        // update the order
        var newKeys = _.map(issues, function (issue) {
            return issue.key;
        });
        this.data.order = _.flatten([newKeys, order], true);

        // build/update the cache
        this.buildCache();
    };


    IssueListModel.getOrderKeyForMarkerId = function (markerId) {
        return '%%-' + markerId;
    };


    IssueListModel.prototype.getIssuesInOrder = function () {
        var orderedIssues = [];
        var order = this.data.order;
        var issueByKey = this.data.issueByKey;
        _.forEach(order, function (rankableId, index) {
            orderedIssues.push(issueByKey[rankableId]);
        }, this);
        return orderedIssues;
    };

    /**
     * @return {String} return the ID of this issue list
     */
    IssueListModel.prototype.getId = function () {
        return this.data.id;
    };

    IssueListModel.prototype.buildCache = function () {
        // store the visible issues in a separate array
        var visibleIssueKeys = this.data.visibleIssueKeys.clear();
        var orderByKey = this.data.orderByKey = {};
        _.forEach(this.data.order, function (rankableId, index) {
            if (!this._isMarkerId(rankableId)) {
                orderByKey[rankableId] = index;
                var issue = this.data.issueByKey[rankableId];
                if (issue && !issue.hidden) {
                    visibleIssueKeys.push(rankableId);
                }
            } else {
                orderByKey[IssueListModel.getOrderKeyForMarkerId(rankableId)] = index;
            }
        }, this);

        // store the parents of subtasks in different object
        this.data.parentSubtaskKeys = {};
        _.forEach(this.data.issueByKey, function(issue, key) {
            var parentKey = issue.parentKey;
            if (parentKey) {
                if (!this.data.parentSubtaskKeys[parentKey]) {
                    this.data.parentSubtaskKeys[parentKey] = [];
                }

                this.data.parentSubtaskKeys[parentKey].push(key);
            }
        }.bind(this));
    };

    /**
     * Determine if the rankable ID is a marker ID (i.e. is a number)
     * @param rankableId
     * @return {*}
     * @private
     */
    IssueListModel.prototype._isMarkerId = function (rankableId) {
        return _.isNumber(rankableId);
    };

    IssueListModel.prototype.indexOfRankable = function (rankableId) {
        if (this._isMarkerId(rankableId)) {
            rankableId = IssueListModel.getOrderKeyForMarkerId(rankableId);
        }
        var index = this.data.orderByKey[rankableId];
        return !_.isUndefined(index) ? index : -1;
    };

    /**
     *
     * @param rankableId marker id or issueKey
     * @return {Boolean}
     */
    IssueListModel.prototype.containsRankable = function (rankableId) {
        return this.indexOfRankable(rankableId) !== -1;
    };

    /**
     * Insert a 'rankable' object (i.e. sprint marker) into the order index of this model.
     *
     * @param markerId the ID of the marker
     * @param rankableAfterId the ID of the rankable which the marker will come after
     */
    IssueListModel.prototype.insertMarker = function (markerId, rankableAfterId) {
        var orderPos = 0;
        var order = this.data.order;
        if (this.containsRankable(markerId)) {
            // this already exists, we must remove it from the array first
            order = _.without(order, markerId);
        }

        if (rankableAfterId) {
            orderPos = _.indexOf(order, rankableAfterId);
            if (orderPos < 0) {
                orderPos = 0;
            } else {
                orderPos++;
            }
        }
        // When we have determined the the new position we want to insert it in the most
        // optimal way. We don't need to modify the visibleIssueKeys or visibilityByIssueKey as we are
        // inserting markerIds only which aren't contained in those data structures
        if (orderPos === 0) {
            order.unshift(markerId);
        } else {
            order = _.flatten([order.slice(0, orderPos), markerId, order.slice(orderPos)]);
        }
        var orderByKey = {};
        _.forEach(order, function (rankableId, index) {
            if (this._isMarkerId(rankableId)) {
                rankableId = IssueListModel.getOrderKeyForMarkerId(rankableId);
            }
            orderByKey[rankableId] = index;
        }, this);
        this.data.order = order;
        this.data.orderByKey = orderByKey;
    };

    /**
     * Remove a 'rankable' object (i.e. sprint marker) from the order index of this model.
     *
     * @param markerId the ID of the rankable
     */
    IssueListModel.prototype.removeMarker = function (markerId) {
        if (this.containsRankable(markerId)) {
            // this already exists, we must remove it from the array first
            var order = _.without(this.data.order, markerId);
            var orderByKey = {};
            _.forEach(order, function (rankableId, index) {
                if (this._isMarkerId(rankableId)) {
                    rankableId = IssueListModel.getOrderKeyForMarkerId(rankableId);
                }
                orderByKey[rankableId] = index;
            }, this);
            this.data.order = order;
            this.data.orderByKey = orderByKey;
        }
    };

    /**
     * Does the given issueKey appear in this model as a subtask.
     * @param issueKey {IssueKey}
     * @returns {boolean}
     */
    IssueListModel.prototype.isSubtask = function(issueKey) {
        var issue = this.getIssueData(issueKey);
        return !!(issue && issue.parentKey);
    };

    /**
     * Reorders a set of issues to either before or after another issue. afterKey/beforeKey can be part of issueKeys,
     * in which case the result for afterKey/beforeKey is the same.
     * The <b>afterKey</b> is taken over the <b>beforeKey</b> if both are specified.
     *
     * @param issueKeys {IssueKey[]} the keys to move around (in order)
     * @param afterKey {IssueKey} the key to rank these issues keys after
     * @param beforeKey {IssueKey} the key to rank these issue keys before
     */
    IssueListModel.prototype.reorderIssues = function (issueKeys, afterKey, beforeKey) {
        var order = this.data.order;
        var insertPosition;
        var rankingKey = afterKey || beforeKey;

        if (afterKey) {
            var keyToInsertAfter = this.getRealAfterKey(afterKey);
            insertPosition = order.indexOf(keyToInsertAfter) + 1;
        } else {
            var keyToInsertBefore = this.getRealBeforeKey(beforeKey);
            insertPosition = order.indexOf(keyToInsertBefore);
        }

        if (this.isSubtask(rankingKey)) {
            // If we are ranking against a subtask, only move subtasks of the same sibling.
            var rankingParentKey = this.getIssueData(rankingKey).parentKey;
            issueKeys = issueKeys.filter(function(key) {
                var issue = this.getIssueData(key);
                return issue && issue.parentKey === rankingParentKey;
            }.bind(this));
        } else {
            // If we are ranking against a parent, for each parent issue that we are moving, move all of its subtasks too,
            // whether these are selected or not
            var issueKeysWithSubtasks = [];
            issueKeys.forEach(function(key) {
                if (!this.isIssueValid(key) || this.isSubtask(key)) {
                    return;
                }

                var subtaskKeys = this.getSubtasks(key).map(function (issue) {
                    return issue.key;
                });

                issueKeysWithSubtasks.push(key);
                subtaskKeys.forEach(function (subtaskKey) {
                    issueKeysWithSubtasks.push(subtaskKey);
                });
            }.bind(this));
            issueKeys = issueKeysWithSubtasks;
        }

        // split lists, then remove keys, then put together (avoids having to fiddle with all weird edge cases
        var beforePosition = order.slice(0, insertPosition);
        var afterPosition = order.slice(insertPosition);
        beforePosition = this.withoutArray(beforePosition, issueKeys);
        afterPosition = this.withoutArray(afterPosition, issueKeys);

        var newOrder = _.flatten([beforePosition, issueKeys, afterPosition]);
        this.setOrder(newOrder);
    };

    /**
     * Get the issue key of the last real issue or subtask, for the given rankable.
     * e.g. if the given rankable is a fake parent, this will find the real last child.
     * If the given rankable is an issue, this will find the real last child.
     * If the given rankable is a subtask, this will return the given rankable.
     * @param rankAfterKey {IssueKey}
     * @return {IssueKey}
     *
     * @private
     */
    IssueListModel.prototype.getRealAfterKey = function(rankAfterKey) {
        if (this.isSubtask(rankAfterKey)) {
            return rankAfterKey;
        }

        var issuesListReversedOrder = _.clone(this.data.order).reverse();

        var lastSubtask = this.findFirstSubtaskOfParentInList(issuesListReversedOrder, rankAfterKey);

        // If we found the last subtask, rank after that. Otherwise, just rank after the issue itself.
        return lastSubtask ? lastSubtask : rankAfterKey;
    };

    /**
     * Get the issue key of the first real issue or subtask, for the given rankable.
     * e.g. if the given rankable is a fake parent, this will find the real first child.
     * If the given rankable is an issue, this will return the given rankable.
     * If the given rankable is a subtask, this will return the given rankable.
     * @param rankBeforeKey {IssueKey}
     * @return {IssueKey}
     *
     * @private
     */
    IssueListModel.prototype.getRealBeforeKey = function(rankBeforeKey) {
        // No need to calculate anything if we are ranking before a subtask, or an issue which exists in the order.
        if (this.isSubtask(rankBeforeKey) || _.contains(this.data.order, rankBeforeKey)) {
            return rankBeforeKey;
        }

        // Otherwise, we must be ranking before a missing parent, so we'll instead rank before the first subtask.
        return this.findFirstSubtaskOfParentInList(this.data.order, rankBeforeKey);
    };

    /**
     * Finds the first subtask which belongs to the parent with the given key.
     * @param list {Array} list to search. Will return first match.
     * @param parentKey {IssueKey} issue key of the parent to search
     * @returns {IssueData} issue that is the first subtask of the given parent
     *
     * @private
     */
    IssueListModel.prototype.findFirstSubtaskOfParentInList = function(list, parentKey) {
        return _.find(list, function (key) {
            var issue = this.getIssueData(key);
            return issue && issue.parentKey === parentKey;
        }.bind(this))
    };

    /**
     * Return the ID of the rankable object which is BEFORE the one specified.
     * @param rankableId the ID of the rankable object you are looking for
     * @return {*} the ID of the rankable object before; false if none found or input did not exist
     */
    IssueListModel.prototype.getRankableBeforeId = function (rankableId) {
        var pos = this.indexOfRankable(rankableId);
        if (pos > 0) {
            return this.data.order[pos - 1];
        }
        return false;
    };

    /**
     * Is an issue visible, given its key
     */
    IssueListModel.prototype.isIssueHiddenBySearchByKey = function (issueKey) {
        return this.data.hiddenIssueKeys[issueKey] || false;
    };

    /**
     * Is issue hidden - by search or if it's a collapsed subtask
     * @param issueKey
     * @returns {boolean}
     */
    IssueListModel.prototype.isIssueHidden = function (issueKey) {
        if (this.isSubtask(issueKey)) {
            var parentKey = this.getIssueData(issueKey).parentKey;

            if (this.isIssueValid(parentKey) && SubtasksExpandingController.isIssueCollapsed(parentKey)) {
                return true;
            }
        }

        return this.isIssueHiddenBySearchByKey(issueKey);
    };


    /**
     * Updates a single issue in the model
     */
    IssueListModel.prototype.updateIssue = function (issue) {
        GH.ExtraFieldsHelper.prepareExtraFields(issue.extraFields);
        this.processFixVersion(issue);
        if (_.isUndefined(this.data.issueByKey[issue.key])) {
            return false;
        }
        this.data.issueByKey[issue.key] = issue;

        this.buildCache();
        return true;
    };

    IssueListModel.prototype.processFixVersion = function (issue) {
        if (this.fixVersions) {
            issue.fixVersionsUI = this.fixVersions.getVersionsForIds(issue.fixVersions);
        }
    };

    /**
     * Get the number of "visible" issues.
     * Here, visible means not "hidden" and not "hiddenBySearch".
     */
    IssueListModel.prototype.getNumVisibleIssues = function () {
        return this.getNumVisibleIssuesBeforeRankable(false);
    };

    /**
     * Get the "visible" issues before the specified rankable.
     * Here, visible means not "hidden" and not "hiddenBySearch".
     * @param rankableId either the issue key or the marker id; if <tt>false</tt> is specified then all issues will be considered
     */
    IssueListModel.prototype.getVisibleIssuesBeforeRankable = function (rankableId) {
        var rankablesBefore;
        if (rankableId !== false) {
            var pos = this.indexOfRankable(rankableId);
            if (pos < 0) {
                return false;
            }
            rankablesBefore = this._first(this.data.order, pos);
        } else {
            rankablesBefore = this.data.order;
        }
        return _.filter(rankablesBefore, function (idOrKey) {
            return this.isIssueVisible(idOrKey);
        }.bind(this));
    };

    /**
     * Get the number of "visible" issues before the specified rankable.
     * Here, visible means not "hidden" and not "hiddenBySearch".
     * @param rankableId either the issue key or the marker id; if <tt>false</tt> is specified then all issues will be considered
     */
    IssueListModel.prototype.getNumVisibleIssuesBeforeRankable = function (rankableId) {
        return this.getVisibleIssuesBeforeRankable(rankableId).length;
    };

    /**
     * Get all issues (hidden and visible) in order
     * @returns map of issues by issueKey
     */
    IssueListModel.prototype.getAllIssues = function () {
        return this.data.issueByKey;
    };

    /**
     * Get visible issues
     * @returns {Object} map of issues by issueKey
     */
    IssueListModel.prototype.getVisibleIssues = function () {
        var visible = {};
        _.each(this.data.issueByKey, function (issue, issueKey) {
            if (this.isIssueVisible(issueKey)) {
                visible[issueKey] = issue;
            }
        }, this);
        return visible;
    };

    /**
     * Get all visible rankables from the list, in their rank order.
     * This will just return the ID or Issue Key of each.
     * @return {*}
     */
    IssueListModel.prototype.getVisibleRankables = function () {
        var self = this;
        return _.filter(self.data.order, function (idOrKey) {
            if (!self.isIssueValid(idOrKey)) {
                return false;
            }

            if (self._isMarkerId(idOrKey)) {
                return true;
            }
            return self.getIssueIndex(idOrKey) !== -1;
        });
    };

    /**
     * Get all visible issues from the list, in their rank order.
     * This will just return the Issue Key of each.
     * @return {*}
     */
    IssueListModel.prototype.getVisibleIssueKeys = function () {
        return this.data.visibleIssueKeys;
    };

    /**
     * Return the map containing all the hidden issue keys.
     * @return {*}
     */
    IssueListModel.prototype.getHiddenBySearchIssues = function () {
        return this.data.hiddenIssueKeys;
    };

    /**
     * Set the issues hidden by search. Note: Normally you would not call this from outside!
     * This method is used to temporarily create a fake list to update a dragged marker statistic
     */
    IssueListModel.prototype.setHiddenBySearchIssues = function (hiddenIssueKeys) {
        this.data.hiddenIssueKeys = hiddenIssueKeys;
    };

    IssueListModel.prototype.getIssueKeyAtIndex = function (index) {
        var visibleIssueKeys = this.data.visibleIssueKeys;
        return visibleIssueKeys.length > index ? visibleIssueKeys.get(index) : visibleIssueKeys.last();
    };

    /**
     * Returns the first visible issue key in the issue list
     *
     * @returns {string|false} Returns the first visible issue key or false if none found
     */
    IssueListModel.prototype.getFirstVisibleIssueKey = function () {
        var visibleIssueKeys = this.data.visibleIssueKeys;
        for (var i = 0; i < visibleIssueKeys.length; i++) {
            var issueKey = visibleIssueKeys.get(i);
            if (!this.isIssueHidden(issueKey)) {
                return issueKey;
            }
        }
        return false;
    };

    /**
     * Returns the last visible issue key in the issue list
     *
     * @returns {string|false} Returns the last visible issue key or false if none found
     */
    IssueListModel.prototype.getLastVisibleIssueKey = function () {
        var visibleIssueKeys = this.data.visibleIssueKeys;
        for (var i = visibleIssueKeys.length - 1; i >= 0; i--) {
            var issueKey = visibleIssueKeys.get(i);
            if (!this.isIssueHidden(issueKey)) {
                return issueKey;
            }
        }
        return false;
    };

    /**
     * Get the index of an issue relative to all the <b>visible</b> issues.
     * @param issueKey
     */
    IssueListModel.prototype.getIssueIndex = function (issueKey) {
        return this.data.visibleIssueKeys.indexOf(issueKey);
    };

    IssueListModel.prototype.getIssueData = function (issueKey) {
        return this.data.issueByKey[issueKey];
    };

    IssueListModel.prototype.getIssueDataForId = function (issueId) {
        return _.find(this.data.issueByKey, function (issue) {
                return issue.id === issueId;
            }) || undefined;

    };

    IssueListModel.prototype.getIssueKeyForId = function (issueId) {
        var issue = this.getIssueDataForId(issueId);

        return issue ? issue.key : undefined;
    };

    IssueListModel.prototype.getIssueIdForKey = function (issueKey) {
        var issueData = this.getIssueData(issueKey);
        if (issueData) {
            return issueData.id;
        }
        return undefined;
    };

    IssueListModel.prototype.isIssueVisible = function (issueKey) {
        return this.getIssueIndex(issueKey) !== -1 && !this.isIssueHiddenBySearchByKey(issueKey);
    };

    /**
     * Is this issue contained in this list?
     * @param issueKeyOrId the issue key or issue ID to look for
     * @return {Boolean}
     */
    IssueListModel.prototype.isIssueValid = function (issueKeyOrId) {
        // if its an issueKey
        if (_.isString(issueKeyOrId)) {
            return !_.isUndefined(this.data.issueByKey[issueKeyOrId]);
        } else {
            // TODO: create a lookup for issueId -> issueKey
            return _.any(this.data.issueByKey, function (issue) {
                return issueKeyOrId == issue.id;
            });
        }
    };

    /**
     * Are these issues contained in this list?
     * @param issueKeysOrIds the issue keys or issue IDs to look for
     * @return {Boolean}
     */
    IssueListModel.prototype.areIssuesValid = function (issueKeysOrIds) {
        if (issueKeysOrIds.length === 0) {
            return false;
        }

        var self = this;
        return _.all(issueKeysOrIds, function (issueKeyOrId) {
            return self.isIssueValid(issueKeyOrId);
        });
    };

    /**
     * Get the first rankable issue key in the column.
     * @param issueKeyOrKeys the issueKey(s) to ignore when finding the first RankableIssueKey.
     *                       This can be a string, an array or null
     * @param useParentIssue if issue is a subtask, use parent issue instead
     */
    IssueListModel.prototype.getFirstRankableIssueKeyInColumn = function (issueKeyOrKeys, useParentIssue) {
        return this._getFirstOrLastRankableIssueKeyInColumn(issueKeyOrKeys, useParentIssue, true);
    };

    /**
     * Get the last rankable issue key in the column, ignoring the passed issue key.
     * This means if you pass the last rankable issue key in the column, it'll return the one *before* it.
     * @param issueKeyOrKeys the issueKey(s) to ignore when finding the last RankableIssueKey.
     *                       This can be a string, an array or null
     * @param useParentIssue if issue is a subtask, use parent issue instead
     */
    IssueListModel.prototype.getLastRankableIssueKeyInColumn = function (issueKeyOrKeys, useParentIssue) {
        return this._getFirstOrLastRankableIssueKeyInColumn(issueKeyOrKeys, useParentIssue, false);
    };

    /**
     * Get the first or last rankable issue key in the column, ignoring the passed issue key.
     * Should be used internally by {#getLastRankableIssueKeyInColumn} or {#getFirstRankableIssueKeyInColumn}
     * @param issueKeyOrKeys the issueKey(s) to ignore when finding the last RankableIssueKey.
     *                       This can be a string, an array or null
     * @param useParentIssue if issue is a subtask, use parent issue instead
     * @param firstRankable find first rankable - if false find last
     */
    IssueListModel.prototype._getFirstOrLastRankableIssueKeyInColumn = function (issueKeyOrKeys, useParentIssue, firstRankable) {
        var issueKeys;
        if (_.isArray(issueKeyOrKeys)) {
            issueKeys = issueKeyOrKeys;
        } else {
            issueKeys = _.isString(issueKeyOrKeys) ? [issueKeyOrKeys] : [];
        }
        var visibleIssueKeys = this.data.visibleIssueKeys;

        // Find the first/last matching visibleIssue which is not in the issueKeys supplied
        var range = _.range(0, visibleIssueKeys.length);
        if (!firstRankable) {
            range.reverse();
        }

        for (var i = 0; i < range.length; i++) {

            // NB: this if statement could be inefficient for large arrays. Create an index?
            var visibleIssueKey = visibleIssueKeys.get(range[i]);
            if (!_.contains(issueKeys, visibleIssueKey)) {
                if (useParentIssue && this.isSubtask(visibleIssueKey)) {
                    return this.getIssueData(visibleIssueKey).parentKey;
                }
                return visibleIssueKey;
            }
        }
        // TODO: Why do we return false everywhere instead of null?
        return false;
    };

    /**
     * Finds the next issue to select given a set of to-be-ranked issues
     */
    IssueListModel.prototype.getBestFitSelectionAfterRank = function (toBeRankedIssueKeys) {
        var bestFit = this.getNextIssueKeyAfterRank(toBeRankedIssueKeys);
        if (!bestFit) {
            bestFit = this.getPreviousIssueKeyAfterRank(toBeRankedIssueKeys);
        }
        return bestFit;
    };

    /**
     * Returns the id of the issue to go to after sending the specified issue to the bottom
     * We have to do some tricky handling to ensure that we don't return an issue which is a child of the current issue.
     */
    IssueListModel.prototype.getNextIssueKeyAfterRank = function (toBeRankedIssueKeys) {
        // the issue keys are in order, so we really only care about the last one
        var issueKey = _.last(toBeRankedIssueKeys);
        return this.getNextIssueKey(issueKey);
    };

    /**
     * Get the next visible issue key after the specified key. Visibility here means that it must not be "hidden" and also
     * must not be "hidden by search".
     * @param issueKey the key to look for
     * @return {*} key or false
     */
    IssueListModel.prototype.getNextIssueKey = function (issueKey) {
        var index = this.getIssueIndex(issueKey);
        var visibleIssueKeys = this.data.visibleIssueKeys;
        while (++index < visibleIssueKeys.length) {
            var key = visibleIssueKeys.get(index);
            if (!this.isIssueHidden(key)) {
                return key;
            }
        }
        return false;
    };

    IssueListModel.prototype.getPreviousIssueKeyAfterRank = function (toBeRankedIssueKeys) {
        // the issue keys are in order, so we really only care about the first one
        var issueKey = _.first(toBeRankedIssueKeys);
        return this.getPreviousIssueKey(issueKey);
    };

    /**
     * Get the next visible issue key after the specified key
     * @param issueKey the key to look for
     * @return {*} key or false
     */
    IssueListModel.prototype.getPreviousIssueKey = function (issueKey) {
        var index = this.getIssueIndex(issueKey);
        var visibleIssueKeys = this.data.visibleIssueKeys;
        while (--index >= 0) {
            var key = visibleIssueKeys.get(index);
            if (!this.isIssueHidden(key)) {
                return key;
            }
        }
        return false;
    };

    /**
     * Get an issue range. The issue keys returned should be in the order as specified by from and to.
     */
    IssueListModel.prototype.getIssueRange = function (fromKey, toKey) {
        var keys = [], i, currentKey;
        var visibleIssueKeys = this.data.visibleIssueKeys;

        // fetch the from and to index
        var fromIndex = this.getIssueIndex(fromKey);
        var toIndex = this.getIssueIndex(toKey);

        // nothing to do if either issue is not valid
        if (_.isUndefined(fromIndex) || _.isUndefined(toIndex)) {
            return keys;
        }

        // forwards
        if (fromIndex < toIndex) {
            for (i = fromIndex; i <= toIndex; i++) {
                currentKey = visibleIssueKeys.get(i);
                if (!this.isIssueHiddenBySearchByKey(currentKey)) {
                    keys.push(currentKey);
                }
            }
            // backwards
        } else {
            for (i = fromIndex; toIndex <= i; i--) {
                currentKey = visibleIssueKeys.get(i);
                if (!this.isIssueHiddenBySearchByKey(currentKey)) {
                    keys.push(currentKey);
                }
            }
        }

        return keys;
    };

    /**
     * Is the passed issue selectable
     * @param selectedIssueKeys the currently selected issues
     * @param issueKey
     */
    IssueListModel.prototype.canAddToSelection = function (selectedIssueKeys, issueKey) {
        // check whether all selected issue keys are visible issues in this model
        var intersection = _.filter(selectedIssueKeys, function (issueKey) {
            return this.getIssueIndex(issueKey) !== -1;
        }, this);
        if (selectedIssueKeys.length > 0 && intersection.length != selectedIssueKeys.length) {
            return false;
        }
        return this.isIssueVisible(issueKey);
    };

    /**
     * Removes the given issues from the model.
     * Adapts the marker accordingly
     */
    IssueListModel.prototype.removeIssues = function (issueKeys) {
        var self = this;
        issueKeys = typeof issueKeys === 'object' ? issueKeys : [issueKeys];

        var result = [];

        _.each(issueKeys, function (key) {
            var issueByKey = self.data.issueByKey[key];
            if (issueByKey) {
                result.push(issueByKey);
                delete self.data.issueByKey[key];
                self.data.order = _.without(self.data.order, key);
                delete self.data.hiddenIssueKeys[key];
            }
        });
        this.buildCache();

        return result;
    };

    /**
     * Returns the last (i.e. previous) visible issue key before the specified rankable ID.
     * @param rankableId the rankable ID to look for
     * @return the issue key or undefined
     */
    IssueListModel.prototype.getLastVisibleKeyBefore = function (rankableId) {
        if (rankableId === undefined) {
            return undefined;
        }

        var lastVisibleKey;
        for (var i = 0; i < this.data.order.length; i++) {
            var key = this.data.order[i];
            if (!this._isMarkerId(key) && this.isIssueVisible(key)) {
                lastVisibleKey = key;
            }
            if (key === rankableId) {
                break;
            }
        }
        return lastVisibleKey;
    };

    /**
     * Get the order of rankable things in the model
     * @return {Array}
     */
    IssueListModel.prototype.getOrder = function () {
        return this.data.order;
    };

    /**
     * Reset the order of rankable things in the model and rebuild the cache
     * @param order the new ordering
     */
    IssueListModel.prototype.setOrder = function (order) {
        this.data.order = order;
        this.buildCache();
    };

    /**
     * Applies the specified search filtering function to the issues in this list. Note that only issues which are not hidden
     * by the server will be processed.
     *
     * @param searchFilterFn function of two arguments: issue object and issue collection (mapping of issue keys to
     * issue objects). The second argument can be ignored. The function returns true if issue from the first argument
     * should be shown.
     * @return {Boolean} true if at least one issue changed as a result of applying the search
     */
    IssueListModel.prototype.applySearchFilter = function (searchFilterFn) {
        var self = this;
        var changed = false;
        self.data.visibleIssueKeys.forEach(function (issueKey) {
            var issue = self.data.issueByKey[issueKey];
            var oldHidden = self.data.hiddenIssueKeys[issueKey] || false;
            var newHidden = !searchFilterFn(issue, self.data.issueByKey);
            changed = changed || oldHidden !== newHidden;
            if (newHidden) {
                self.data.hiddenIssueKeys[issueKey] = true;
            } else {
                delete self.data.hiddenIssueKeys[issueKey];
            }
        });
        return changed;
    };

    /**
     * Checks whether the given issues are between two markers. The order of the markers is not important.
     * Other markers or issues can be in between the two given markers.
     * @param issueKeys
     * @param markerFrom
     * @param markerTo
     */
    IssueListModel.prototype.areIssuesBetweenMarkers = function (issueKeys, markerFrom, markerTo) {
        var issueRange = this.getIssuesBetweenMarkers(markerFrom, markerTo);
        var containsAllIssues = true;
        _.forEach(issueKeys, function (issueKey) {
            if (!_.contains(issueRange, issueKey)) {
                containsAllIssues = false;
            }
        });
        return containsAllIssues;
    };

    /**
     * Returns an array of issue keys of the issues ranked between the given markers.
     * @param markerFrom
     * @param markerTo
     */
    IssueListModel.prototype.getIssuesBetweenMarkers = function (markerFrom, markerTo) {
        var markerFromIndex = this.indexOfRankable(markerFrom);
        var markerToIndex = this.indexOfRankable(markerTo);

        if (markerFromIndex > markerToIndex) {
            var tmp = markerToIndex;
            markerToIndex = markerFromIndex;
            markerFromIndex = tmp;
        }

        return this.getOrder().slice(markerFromIndex + 1, markerToIndex);
    };

    /**
     * Returns the marker before the marker with the given markerId
     * @param markerId
     */
    IssueListModel.prototype.getMarkerIdBeforeId = function (markerId) {
        if (this._isMarkerId(markerId)) {
            markerId = IssueListModel.getOrderKeyForMarkerId(markerId);
        }
        var index = this.data.orderByKey[markerId];
        if (_.isUndefined(index)) {
            return false;
        }

        for (var i = index - 1; i >= 0; i--) {
            if (this._isMarkerId(this.data.order[i])) {
                return this.data.order[i];
            }
        }
        return false;
    };

    /**
     * Returns the id of the last marker in the list model, or false if the model doesn't contain any markers
     */
    IssueListModel.prototype.getLastMarkerId = function () {
        var markerId = false;
        var order = this.data.order;
        var length = order.length;
        for (var i = length - 1; i >= 0; i--) {
            var rankableId = order[i];
            if (this._isMarkerId(rankableId)) {
                markerId = rankableId;
                break;
            }
        }
        return markerId;
    };

    /**
     * Returns an array of markers id's as they appear in the list model
     */
    IssueListModel.prototype.getMarkerIds = function () {
        var markerIds = [];
        _.each(this.data.order, function (rankableId) {
            if (this._isMarkerId(rankableId)) {
                markerIds.push(rankableId);
            }
        }, this);
        return markerIds;
    };

// not available in underscore
    IssueListModel.prototype.withoutArray = function (a, b) {
        var res = a;
        _.each(b, function (elem) {
            res = _.without(res, elem);
        });
        return res;
    };

    /**
     * Underscore.js implementation of _.first(array, pos) behaves strangely when pos == 0 -- it returns '1' instead of
     * an empty array.
     * @param array
     * @param pos
     * @return {Array}
     * @private
     */
    IssueListModel.prototype._first = function (array, pos) {
        pos = parseInt(pos, 10);
        return pos === 0 ? [] : _.first(array, pos);
    };

    /**
     * Returns the amount of issues in the model
     */
    IssueListModel.prototype.getIssueCount = function () {
        var count = 0;
        _.each(this.data.issueByKey, function (issueKey) {
            count++;
        });
        return count;
    };

    /**
     * Get the data for all invisible issues in this list, that is hidden and hiddenBySearch
     */
    IssueListModel.prototype.getAllInvisibleIssuesData = function () {
        var order = this.data.order;
        return this._getInvisibleIssuesDataForKeys(order);
    };

    /**
     * Get the data for all visible issues.
     * Here, visible means not "hidden" and not "hiddenBySearch".
     */
    IssueListModel.prototype.getAllVisibleIssuesData = function () {
        var issuesInOrder = this.data.order;
        return this._getVisibleIssuesDataForKeys(issuesInOrder);
    };

    /**
     * Get the data for all visible issues before and including the passed issue key
     */
    IssueListModel.prototype.getVisibleIssuesDataBeforeIncludingKey = function (lastIncludedIssueKey) {
        var pos = this.indexOfRankable(lastIncludedIssueKey);
        if (pos < 0) {
            return [];
        }
        var issuesInOrder = this.data.order.slice(0, pos + 1);
        return this._getVisibleIssuesDataForKeys(issuesInOrder);
    };

    /**
     * Get the data for all visible issues after the passed issue key
     */
    IssueListModel.prototype.getVisibleIssuesDataAfterKey = function (lastExcludedIssueKey) {
        var pos = this.indexOfRankable(lastExcludedIssueKey);
        if (pos < 0) {
            return [];
        }
        var issuesInOrder = this.data.order.slice(pos + 1);
        return this._getVisibleIssuesDataForKeys(issuesInOrder);
    };

    /**
     * Get the sub-tasks of an issue
     * @param issueKeyOrId it can be the parent id or key
     */
    IssueListModel.prototype.getSubtasks = function (issueKeyOrId) {
        return this.getSubtasksKeys(issueKeyOrId).map(function (issueKey) {
            return this.getIssueData(issueKey);
        }.bind(this));
    };

    /**
     * Get the sub-tasks keys of an issue
     * @param issueKeyOrId it can be the parent id or key
     */
    IssueListModel.prototype.getSubtasksKeys = function (issueKeyOrId) {
        var subtasksKeys = this.data.parentSubtaskKeys[issueKeyOrId];
        if (!_.isUndefined(subtasksKeys)) {
            return subtasksKeys;
        }

        var issueData = this.getIssueDataForId(issueKeyOrId);
        if (!issueData) {
            return [];
        }

        return this.data.parentSubtaskKeys[issueData.key] || [];
    };

    /**
     * Check if given issue is parent in this IssueListModel
     * @param issueKey
     * @returns {boolean}
     */
    IssueListModel.prototype.isParent = function(issueKey) {
        return !!this.data.parentSubtaskKeys[issueKey];
    };

    /**
     * Return parent key if issue is subtask or issue key
     * @param issueKey
     * @returns {string}
     */
    IssueListModel.prototype.getParentKey = function(issueKey) {
        if (this.isSubtask(issueKey)) {
            return this.getIssueData(issueKey).parentKey;
        }

        return issueKey;
    };

    /**
     * Returns the data for all visible issues
     */
    IssueListModel.prototype._getVisibleIssuesDataForKeys = function (issuesKeysInOrder) {
        var issues = [];
        _.each(issuesKeysInOrder, function (issueKey) {
            if (this.isIssueVisible(issueKey)) {
                var issue = this.getIssueData(issueKey);
                issues.push(issue);
            }
        }, this);
        return issues;
    };

    /**
     * Returns the data for all invisible issues
     */
    IssueListModel.prototype._getInvisibleIssuesDataForKeys = function (issuesKeysInOrder) {
        var issues = [];
        _.each(issuesKeysInOrder, function (issueKey) {
            var issue = this.getIssueData(issueKey);
            if (issue.hidden || this.isIssueHiddenBySearchByKey(issueKey)) {
                issues.push(issue);
            }
        }, this);
        return issues;
    };

    IssueListModel.prototype.getIssuesExcludingSubtasks = function () {
        return _.filter(this.data.issueByKey, function(issue) {
            return _.isUndefined(issue.parentId);
        });
    };


    return IssueListModel;
});
