/* globals
 * GH.sprintActions, GH.planOnboarding, GH.Notification, GH.NumberFormat, GH.ScrollUtils, Waypoint
 */

/**
 * @module jira-agile/rapid/ui/plan/BacklogView
 * @requires module:underscore
 * @requires module:jquery
 * @requires module:waypoints
 * @requires module:jira/ajs/dark-features
 * @requires module:jira/util/navigator
 * @requires module:jira-agile/rapid/global-events
 * @requires module:jira-agile/rapid/ui/plan/backlog-model2
 * @requires module:jira-agile/rapid/ui/plan/sprint-view
 * @requires module:jira-agile/rapid/ui/plan/sprint-backlog-view
 * @requires module:jira-agile/rapid/ui/plan/sprint-model
 * @requires module:jira-agile/rapid/ui/plan/backlog-controller
 * @requires module:jira-agile/rapid/ui/plan/backlog-selection-controller
 * @requires module:jira-agile/rapid/ui/plan/plan-drag-and-drop
 */
define('jira-agile/rapid/ui/plan/BacklogView', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var $ = require('jquery');
    var Waypoint = require('Waypoint');
    var DarkFeature = require('jira/ajs/dark-features');
    var navigator = require('jira/util/navigator');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var BacklogModel2 = require('jira-agile/rapid/ui/plan/backlog-model2');
    var SprintView = require('jira-agile/rapid/ui/plan/sprint-view');
    var SprintBacklogView = require('jira-agile/rapid/ui/plan/sprint-backlog-view');
    var SprintModel = require('jira-agile/rapid/ui/plan/sprint-model');
    var BacklogSelectionController = require('jira-agile/rapid/ui/plan/backlog-selection-controller');
    var PlanDragAndDrop = require('jira-agile/rapid/ui/plan/plan-drag-and-drop');
    var BacklogController;

    var FILTERED_ISSUE_CLASS = "ghx-filtered";

    // Resolve circular dependencies
    GlobalEvents.on('pre-initialization', function () {
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
    });

    var BacklogView = {};

    BacklogView.containerSelector = '#ghx-backlog';

    BacklogView.getContainer = function () {
        if (!BacklogView.container) {
            BacklogView.container = $(BacklogView.containerSelector);
        }
        return BacklogView.container;
    };
    /**
     * Draw the Backlog.
     */
    BacklogView.draw = function () {
        GH.Logger.log("BacklogView.draw", GH.Logger.Contexts.ui);

        // clear out the current view
        BacklogView.container = undefined;
        var $backlogViewContainer = $(BacklogView.containerSelector);
        $backlogViewContainer.html(GH.tpl.backlogview.renderBacklogView({}));

        var issueRenderData = BacklogController.calculateIssueRenderData();

        // draw all sprints
        SprintView.renderAllSprints(issueRenderData);

        // draw backlog
        SprintBacklogView.renderBacklog(issueRenderData);

        // draw onboarding
        GH.planOnboarding.renderOnboarding();

        // Delay event bindings until after rendering for performance reasons
        _.delay(function () {
            // enable d&d
            PlanDragAndDrop.enableDragAndDrop();

            // register click handlers
            BacklogView.registerClickHandlers();

            // register twixies for meta data disclosure
            SprintView.registerTwixies();

            GH.sprintActions.createDropdown();

            // Enable sticky sprint and backlog headers
            BacklogView.enableStickyHeaders();

            // set the "rendered" attribute to indicate when the eagle has land... err, board has loaded
            $('#ghx-backlog').attr("data-rendered", new Date().getTime());
        });
    };

    BacklogView.registerClickHandlers = function () {
        // listener for filter clearing
        $(BacklogView.containerSelector).delegate('.js-clear-all-filters', 'click', function (event) {
            BacklogController.clearFilters();
        });
    };

    BacklogView.redrawChangedModel = function (model) {
        BacklogView.redrawChangedModels([model]);
    };

    BacklogView.redrawChangedModels = function (models) {
        // rerender each model
        _.each(models, function (model) {
            if (SprintModel.isSprintModel(model)) {
                SprintView.updateSprintViewForModel(model);
            } else if (BacklogModel2.isBacklogModel(model)) {
                SprintBacklogView.updateBacklog();
            }
        });

        GH.planOnboarding.renderOnboarding();

        // Re-generate sticky header clones and re-calculate offsets due to the changes in DOM
        BacklogView.enableStickyHeaders();

        // re-enable drag and drop for backlog issue list
        PlanDragAndDrop.enableDragAndDrop();

        // re-create Dropdown for sprint actions
        GH.sprintActions.createDropdown();
    };

    BacklogView.updateSprintsAfterAddRemove = function () {
        var issueRenderData = BacklogController.calculateIssueRenderData();

        // draw all sprints
        SprintView.renderAllSprints(issueRenderData);

        GH.planOnboarding.renderOnboarding();

        // Re-generate sticky header clones and re-calculate offsets due to the changes in DOM
        BacklogView.enableStickyHeaders();

        // re-enable drag and drop for backlog issue list
        PlanDragAndDrop.enableDragAndDrop();

        // re-create Dropdown for sprint actions
        GH.sprintActions.createDropdown();
    };

    /**
     * Updates the selection of the ui
     */
    BacklogView.updateIssueSelectionState = function (doScroll) {
        var scrollToIssueKey = false;
        var scrollToViewElem = null;
        var selectedIssue = BacklogSelectionController.getSelectedIssueKey();
        scrollToIssueKey = selectedIssue;
        $(BacklogView.containerSelector).find('.js-issue').removeClass('ghx-selected ghx-selected-primary').each(function (index, elem) {
            var $elem = $(elem);
            var issueKey = $elem.data('issue-key');
            if (BacklogSelectionController.isInSelection(issueKey)) {
                $elem.addClass('ghx-selected');
                if (selectedIssue == issueKey) {
                    $elem.addClass('ghx-selected-primary');
                }
            }
            if (scrollToIssueKey != null && issueKey === scrollToIssueKey) {
                scrollToViewElem = $elem;
            }
        });

        // scroll the currently selected issue to view
        // this is a slow operation so only do it if necessary
        // not on mouse click
        if (scrollToViewElem && doScroll) {
            BacklogView._scrollToView(scrollToViewElem);
        }
    };

    /**
     * Updates the visibility of all issues and container statistics
     *
     * Re-drawing the whole backlog is slow
     * We instead hide the container div, update the display property on all the elements and re-show the container
     * This means we only do two re-flows, instead of one per issue changed
     * Using .css instead of .hide() / .show() is much faster, as .hide/.show have to store the current display information
     */
    BacklogView.updateHiddenIssues = function () {
        // fetch the issueRenderData which contains information about hidden issues
        var issueRenderData = BacklogController.calculateIssueRenderData();

        // hide the backlog
        var $backlog = $("#ghx-backlog");
        $backlog.css({ 'display': 'none' });

        // update all issue lists
        var containers = $backlog.find('.js-issue-list');
        _.each(containers, function (container) {
            var $issuesContainer = $(container);

            // if there are no issues there is nothing to do
            if ($issuesContainer.is('.js-empty-list')) {
                return;
            }

            // update issue visibilities
            var allFiltered = BacklogView._updateIssuesVisibilityInIssueList($issuesContainer, issueRenderData);

            // set flag according to all issues beening filtered
            if (allFiltered) {
                $issuesContainer.removeClass('ghx-has-issues').addClass('ghx-no-issues');
            } else {
                $issuesContainer.removeClass('ghx-no-issues').addClass('ghx-has-issues');
            }
        });

        // update the statistics of all sprints
        SprintView.updateSprintStatistics();

        // update the backlog header
        SprintBacklogView.updateBacklogHeader();

        GH.planOnboarding.renderOnboarding();

        $backlog.css({ 'display': '' });

        // Re-calculate offsets due to the changes in DOM
        BacklogView.adjustStickyHeader();
    };

    /**
     * updates the issue visibility and returns whether all issues are filtered
     * @return boolean true if all issues are hidden
     */
    BacklogView._updateIssuesVisibilityInIssueList = function ($container, issueRenderData) {
        // update the backlog
        var $issues = $container.find(".js-issue");
        var allHidden = true;
        _.each($issues, function (issue) {
            var $issue = $(issue);
            var issueKey = $issue.attr('data-issue-key');
            if (issueRenderData.hiddenIssues[issueKey]) {
                $issue.addClass(FILTERED_ISSUE_CLASS);
            } else {
                $issue.removeClass(FILTERED_ISSUE_CLASS);
                allHidden = false;
            }
        });

        //show/hide containers without issues
        var $fakeContainers = $container.find('.ghx-parent-group');
        $fakeContainers.each(function (index, fakeContainer) {
            var $fakeContainer = $(fakeContainer);

            var $subtasks = $fakeContainer.find('.ghx-issue-subtask');
            var $parentStub = $fakeContainer.find('.ghx-parent-stub');

            // all subtasks are filtered - do not show stub
            var allFiltered = $subtasks.not('.' + FILTERED_ISSUE_CLASS).length === 0;
            $parentStub.toggleClass(FILTERED_ISSUE_CLASS, allFiltered);

            // hidden subtasks counter
            var $counterContainer = $fakeContainer.find('.ghx-hidden-subtasks-count-container');
            var hiddenByQuickFilters = parseInt($counterContainer.attr('data-subtasks-hidden-by-filters'), 10);
            var hiddenBySearch = $subtasks.filter('.ghx-filtered').length;
            var hiddenInTotal = hiddenByQuickFilters + hiddenBySearch;

            $counterContainer.html(GH.tpl.planissuelist.renderSubtasksFilteredContent({
                hiddenSubtasks: hiddenInTotal,
                allHidden: allFiltered
            }));
        });

        return allHidden;
    };

    /**
     * 'Element query' for backlog column - used to resize/change elements
     */
    BacklogView.checkBacklogWidth = function () {
        var currentWidth = BacklogView.getContainer().width();

        // These classes must be placed on the body or they won't apply to the elements created when dragging an issue
        if (currentWidth < 600) {
            $('body').addClass('ghx-plan-band-1').removeClass('ghx-plan-band-2');
        } else {
            $('body').addClass('ghx-plan-band-2').removeClass('ghx-plan-band-1');
        }

        // Update sticky header width and positioning during resize
        BacklogView.adjustStickyHeaderPosition();

        GH.planOnboarding.refresh();
    };

    /** Inline Issue Create expects this exact method to be called in its Acceptance tests, so we cant remove it **/
    BacklogView._scrollToView = function ($elem) {
        var $backlog = $('#ghx-backlog');

        // bypass jQuery which is a pain for that kind of work. getBoundingClientRect is exactly what we need and is supported on all our supported browsers.
        // we assume that the document is not scrollable by design, so we don't have to manage document scroll offset
        var backlogRect = $backlog[0].getBoundingClientRect();
        var elementRect = $elem[0].getBoundingClientRect();

        GH.ScrollUtils.scrollElementToViewPort($backlog, backlogRect, elementRect);
    };

    BacklogView.isStickySprintHeadersEnabled = function () {
        return GH.RapidBoard.stickySprintHeaders && !DarkFeature.isEnabled('jira.greenhopper.backlog.disableStickyHeader');
    };

    /**
     * Makes the sprint and backlog headers sticky
     */
    BacklogView.enableStickyHeaders = function () {

        if (!BacklogView.isStickySprintHeadersEnabled()) {
            return;
        }
        // Make sure existing stickies and waypoints are cleaned up while redrawing
        BacklogView.destroyStickies();

        var $backlogViewContainer = this.getContainer();
        if (!$backlogViewContainer.parent().hasClass('ghx-sticky-wrapper')) {
            $backlogViewContainer.wrap($('<div/>', { 'class': 'ghx-sticky-wrapper' }));
        }

        // Make the headers sticky upon reaching top of the backlog container, and non-sticky when scrolling back
        $backlogViewContainer.find('.ghx-backlog-header').each(function (i, el) {
            var $el = $(this);

            var $sprintContainer = $el.closest('.ghx-backlog-container');
            if (!BacklogView.stickies) {
                BacklogView.stickies = [];
            }
            var sticky = new Waypoint.Sticky({
                element: el,
                context: $backlogViewContainer,
                offset: 0,
                handler: function handler(direction) {
                    var footerDragInfo = BacklogView.getFooterDragInfo($sprintContainer);

                    if (footerDragInfo.isFooterDrag && footerDragInfo.isOverTakenSprint) {
                        // Don't make this sticky while dragging footer of another sprint
                        if ($el.hasClass('stuck')) {
                            $el.removeClass('stuck');
                        }
                    }
                    if (footerDragInfo.isFooterDrag && footerDragInfo.isOvertakingSprint) {
                        BacklogView.ensureIsSticky($el, direction);
                    }

                    // While scrolling back in, remove the absolute positioning applied earlier when scrolling back up
                    if ($el.hasClass('stuck-bottom')) {
                        $el.removeClass('stuck-bottom');
                    }

                    // Remove js positioning while scrolling up in IE
                    BacklogView.adjustStickyHeaderPosition(direction === 'up', $el);
                }
            });
            BacklogView.stickies.push(sticky);
        });

        // Makes the header non-sticky when the entire section scrolls out of the view, and makes it sticky when the
        // entire section starts scrolling back in
        $backlogViewContainer.find('.ghx-backlog-container').waypoint({
            context: $backlogViewContainer,
            group: 'containers',
            offset: function offset() {
                var $el = $(this.element);
                var $header = $el.find('.ghx-backlog-header').parent();
                var headerHeight = $header.outerHeight();
                return ($el.outerHeight() - headerHeight) * -1;
            },
            handler: BacklogView.waypointHandler
        });

        setTimeout(Waypoint.refreshAll, 0);
    };

    BacklogView.getFooterDragInfo = function ($sprintContainer) {
        var $backlogViewContainer = this.getContainer();
        var $groupContainer = $backlogViewContainer.find('#ghx-content-group');
        var sprintId = $sprintContainer.data('sprint-id') || 'backlog';
        var $overTakingSprint = $groupContainer.find('.ghx-backlog-container.ghx-active-drag');
        var isFooterDrag = $groupContainer.hasClass('ghx-drag-in-progress') && $overTakingSprint.length;
        var isOvertakingSprint = $sprintContainer.hasClass('ghx-active-drag');
        // Due to an existing bug, while smooth scrolling fast sometimes the class
        // ghx-overtaken doesn't get added correctly. Hence the additional checks
        var isOverTakenSprint = $sprintContainer.hasClass('ghx-overtaken') || $sprintContainer.prevAll().is($overTakingSprint) || isFooterDrag && sprintId === 'backlog';

        return {
            isFooterDrag: isFooterDrag,
            isOvertakingSprint: isOvertakingSprint,
            isOverTakenSprint: isOverTakenSprint
        };
    };

    BacklogView.ensureIsSticky = function ($header, direction) {
        // Makes sure the header stays sticky while dragging footer over another sprint
        if (direction === 'down') {
            if (!$header.hasClass('stuck')) {
                //make this sticky even if contents are out of view
                $header.addClass('stuck');
            }
        }
    };

    BacklogView.waypointHandler = function (direction) {
        var $el = $(this.element);
        var $header = $el.find('.ghx-backlog-header');
        var footerDragInfo = BacklogView.getFooterDragInfo($el);

        if (footerDragInfo.isFooterDrag && footerDragInfo.isOverTakenSprint) {
            // Don't make this sticky while dragging footer of another sprint
            return;
        }

        if (footerDragInfo.isFooterDrag && footerDragInfo.isOvertakingSprint) {
            BacklogView.ensureIsSticky($header, direction);
            return;
        }

        if (direction === 'down') {
            $header.removeClass('stuck');
            // remove js positioning in IE
            BacklogView.adjustStickyHeaderPosition(true, $header);
            $header.addClass('stuck-bottom');
        } else if (direction === 'up') {
            $header.removeClass('stuck-bottom');
            BacklogView.adjustStickyHeaderPosition(null, $header);
            $header.addClass('stuck');
            // HACK. For some reason while smooth scrolling down fast the sticky header waypoint
            // only fires after this way point, which leaves 'stuck' class on the previous sprint header
            var next = this.next();
            if (next) {
                var $next = $(next.element);
                var $nextHeader = $next.find('.ghx-backlog-header');
                if ($nextHeader.hasClass('stuck')) {
                    $nextHeader.removeClass('stuck');
                    BacklogView.adjustStickyHeaderPosition(true, $nextHeader);
                }
            }
        }
    };

    // This function is used for adjusting the width of fixed header in IE after changes are made to the UI layout (For
    // example opening a side panel), because unlike other browsers IE doesn't respect the parent container with transform
    BacklogView.adjustStickyHeaderPosition = !navigator.isIE() || navigator.isEdge() ? $.noop : _.debounce(function (removePositioning, el) {

        var $backlogViewContainer = BacklogView.getContainer();
        var $el = el || $backlogViewContainer.find('.ghx-backlog-header.stuck');

        // if the function is invoked before sticky header is setup, do nothing
        if (!$el.length) {
            return;
        }

        var fixedOffSet;
        var width;
        if (!removePositioning) {
            var $scrollParent = $backlogViewContainer.parent();
            var $container = $el.closest('.ghx-backlog-container');
            fixedOffSet = $scrollParent.get(0).getBoundingClientRect().top;
            width = $el.parent().outerWidth();
            if ($container.hasClass('ghx-everything-else')) {
                // Is backlog header
                width += parseInt($container.css('padding-left'), 10); //overlap the parent padding
            }
        } else {
            fixedOffSet = 'auto';
            width = '';
        }
        $el.css({ width: width, top: fixedOffSet });
    }, 100);

    // This function is used for adjusting scroll offsets after a UI update
    BacklogView.adjustStickyHeader = DarkFeature.isEnabled('jira.greenhopper.backlog.disableStickyHeader') ? $.noop : _.debounce(function () {
        setTimeout(function () {
            BacklogView.adjustStickyHeaderPosition();
            Waypoint.refreshAll();
        }, 0);
    }, 100);

    BacklogView.destroyStickies = function () {
        if (!BacklogView.stickies || !BacklogView.stickies.length) {
            return;
        }

        BacklogView.stickies.forEach(function (sticky) {
            sticky.destroy();
        });

        Waypoint.destroyAll();
    };

    return BacklogView;
});