define('jira-agile/rapid/ui/kanplan/kan-plan-feature-service', ['require'], function (require) {
    'use strict';

    var flag = require('jira/flag');
    var timedFlag = require('jira-agile/rapid/configuration/timed-flag');
    var formatter = require('jira/util/formatter');
    var analytics = require('jira/analytics');
    var _ = require('underscore');
    var $ = require('jquery');
    var FeatureFlagManager = require('jira/featureflags/feature-manager');
    var wrmData = require('wrm/data');

    var KANPLAN_FEATURE_FLAG = 'com.atlassian.jira.agile.darkfeature.kanplan';
    var KANPLAN_EPICS_AND_VERSIONS_FEATURE_FLAG = 'com.atlassian.jira.agile.darkfeature.kanplan.epics.and.versions';

    var KanPlanFeatureService = {};
    var FLAG_AUTO_CLOSE_TIME = 10000;

    var epicsOnboardingState = wrmData.claim("com.pyxis.greenhopper.jira:wrm-data.kanplan-epics-onboarding");

    KanPlanFeatureService.acknowledgeDisplayData = wrmData.claim("com.pyxis.greenhopper.jira:wrm-data.kanplan-acknowledment");

    KanPlanFeatureService.autoConvertKanPlan = function (rapidViewId) {
        return GH.Ajax.post({
            url: '/xboard/plan/kanban/' + rapidViewId + '/autoConvert',
            deferErrorHandling: true
        }, 'auto-convert-kanPlan');
    };

    KanPlanFeatureService.isFeatureEnabled = function () {
        return FeatureFlagManager.isFeatureEnabled(KANPLAN_FEATURE_FLAG);
    };

    KanPlanFeatureService.isEpicsAndVersionsEnabled = function () {
        return KanPlanFeatureService.isFeatureEnabled() && FeatureFlagManager.isFeatureEnabled(KANPLAN_EPICS_AND_VERSIONS_FEATURE_FLAG);
    };

    KanPlanFeatureService.shouldShowEpicsPanel = function () {
        return this.isFeatureEnabled() && this.isEpicsAndVersionsEnabled() && GH.RapidBoard.State.getShowEpicAsPanel();
    };

    KanPlanFeatureService.isVersionsFilterApplied = function () {
        return this.isEpicsAndVersionsEnabled() && !!GH.BoardState.getPerViewValue('planSelectedVersion');
    };

    KanPlanFeatureService.isEpicsFilterApplied = function () {
        return this.shouldShowEpicsPanel() && !!GH.BoardState.getPerViewValue('planSelectedEpic');
    };

    KanPlanFeatureService.shouldShowKanbanBacklog = function () {
        return GH.RapidBoard.kanPlanLabEnabled && GH.RapidViewConfig.kanPlanEnabled;
    };

    /**
     * Can show KanPlan Acknowledge when
     * <ul>
     *     <li>Board has been configured successfully for KanPlan</li>
     *     <li>Users have never clicked close button on plan/work mode</li>
     * </ul>
     * @returns {boolean}
     */
    KanPlanFeatureService.shouldShowAcknowledge = function () {
        if (!KanPlanFeatureService.acknowledgeDisplayData) {
            return false;
        }
        if (KanPlanFeatureService.shouldShowKanbanBacklog()) {
            return GH.RapidBoard.State.isWorkMode() ? !(KanPlanFeatureService.acknowledgeDisplayData.isClosedOnWorkMode || KanPlanFeatureService.acknowledgeDisplayData.isClosedOnPlanMode) : !KanPlanFeatureService.acknowledgeDisplayData.isClosedOnPlanMode;
        }
        return false;
    };

    KanPlanFeatureService.shouldShowEpicsOnboarding = function (mode) {
        if (mode === 'work' && (epicsOnboardingState.isClosedOnWorkMode || epicsOnboardingState.isClosedOnPlanMode)) {
            return false;
        } else if (mode === 'plan' && epicsOnboardingState.isClosedOnPlanMode) {
            return false;
        }
        return true;
    };

    KanPlanFeatureService.showKanPlanDisabledFlag = function () {
        flag.showInfoMsg('', formatter.I18n.getText('gh.rapid.kanplan.not.enabled.message'), { close: 'auto' });
    };

    KanPlanFeatureService.showKanPlanInlineDialogDismissedFlag = function () {
        var url = GH.RapidBoard.getRapidViewConfigurationUrl(GH.WorkController.rapidViewData.id, 'columns');
        var flagInstance = flag.showInfoMsg('', GH.tpl.kanplan.onboarding.flags.dismissedFlag({ url: url }));
        var analyticsEventName = 'jira-software.kanplan.onboarding.enable-prompt.dismissflag.click';
        $(flagInstance).find('.kanplan-autoconvert-dismiss-flag').on('click', KanPlanFeatureService.sendAnalyticsInfo.bind(null, analyticsEventName, null));
    };

    /**
     * Displays flag after successful kanplan auto conversion.
     * Flag will contain a link to backlog if the user have dismissed acknowledgement banner at any board in the past
     * @param shouldLinkToBacklog - Boolean indicates whether to display link or not
     */
    KanPlanFeatureService.showAutoConvertSuccessFlag = function (shouldLinkToBacklog) {
        var flagContent = GH.tpl.kanplan.onboarding.flags.autovenversionSuccessFlag({ shouldLinkToBacklog: shouldLinkToBacklog });
        var hideDuration = shouldLinkToBacklog ? FLAG_AUTO_CLOSE_TIME : undefined; // Pass undefined so that flag is set to auto
        var $flagInstance = timedFlag.successMsgTimed('', flagContent, {}, hideDuration);

        $flagInstance.on('click', function () {
            $flagInstance.get(0).close();
            var analyticsEventName = 'jira-software.kanplan.onboarding.convert.successflag.click';
            KanPlanFeatureService.sendAnalyticsInfo(analyticsEventName);
            GH.RapidBoard.ViewController.setMode('plan');
        });
    };

    /**
     * Method used to send kanplan onboarding analytics data such as click in flags with basic data.
     * @param name - name of the event
     */
    KanPlanFeatureService.sendAnalyticsInfo = function (name, props, includeDefaults) {
        if (!name) {
            return;
        }

        var defaults = {
            boardId: GH.RapidBoard.State.getRapidViewId(),
            isBoardAdmin: GH.RapidViewConfig.isBoardAdmin,
            isGlobalAdmin: GH.UserData.userConfig.isGlobalAdmin
        };

        analytics.send({
            name: name,
            properties: props ? includeDefaults ? _.extend(defaults, props) : props : defaults
        });
    };

    KanPlanFeatureService.dismissEpicsOnboardingDialog = function (mode) {
        if (!mode) {
            mode = GH.RapidBoard.State.data.mode;
        }
        return GH.Ajax.post({ url: '/xboard/plan/kanban/closeEpicsOnboardingDialog?mode=' + mode }).done(function () {
            KanPlanFeatureService.updateKanplanEpicsOnboardingState(mode);
        });
    };

    KanPlanFeatureService.updateKanplanEpicsOnboardingState = function (mode) {
        var value = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;

        if (mode === GH.RapidBoard.State.VIEW_MODE.PLAN) {
            epicsOnboardingState.isClosedOnPlanMode = value;
        } else if (mode === GH.RapidBoard.State.VIEW_MODE.WORK) {
            epicsOnboardingState.isClosedOnWorkMode = value;
        } else if (mode === 'both') {
            epicsOnboardingState.isClosedOnPlanMode = value;
            epicsOnboardingState.isClosedOnWorkMode = value;
        }
    };

    return KanPlanFeatureService;
});