/**
 * AUI Tooltip with HoverIntent and event delegation support
 */
(function(_, jq, Events) {
    'use strict';

    var HOVERINTENT_INTERVAL = 200;
    var TOOLTIP_MARGIN = 20;

    /**
     * Reposition a tooltip with either 'n' or 's' gravity so that the left or right edge does not go off screen.
     * If necessary this can be repurposed to work for 'w' and 'e' gravities too.
     * @param tipsyObject
     */
    function nudgeNSTooltipIntoView(tipsyObject) {
        var gravity = tipsyObject.options.gravity;
        if (typeof gravity === 'function') {
            gravity = gravity.call(tipsyObject.$element[0]);
        }
        if (gravity.charAt(0) === 'n' || gravity.charAt(0) === 's') {
            var $tip = tipsyObject.$tip;
            var offset = $tip.offset();
            var overflowLeft = Math.max(0, TOOLTIP_MARGIN - offset.left);
            var overflowRight = Math.max(0, TOOLTIP_MARGIN + offset.left + $tip.outerWidth() - jq(window).width());
            var $arrow = $tip.find('.tipsy-arrow');

            // Adjust if there is either overflowLeft or overflowRight, but not both
            var adjustment = (overflowLeft && overflowRight) ? 0 : (overflowLeft || -overflowRight);
            if (adjustment) {
                $tip.css('left', parseInt($tip.css('left'), 10) + adjustment);
                $arrow.css('marginLeft', -$arrow.outerWidth() / 2 - adjustment);
            } else {
                // Reset arrow positioning as tipsy caches the element
                $arrow.css('marginLeft', '');
            }
        }
    }

    /**
     * Tipsy leaves its tooltip element around even if the element it is attached to
     * is removed from the document or becomes hidden.
     * This function checks at a short interval whether the tooltip should be hidden to prevent "detached" tooltips.
     * @param tipsyObject
     */
    function hideTipsyWhenDetached(tipsyObject) {
        var CHECK_INTERVAL = 250;

        setTimeout(function() {
            // If the tipsy isn't visible any more we stop checking as well
            if (tipsyObject.$tip.is(':visible')) {
                if (tipsyObject.$element.is(':visible')) {
                    hideTipsyWhenDetached(tipsyObject);
                } else {
                    tipsyObject.hide();
                }
            }
        }, CHECK_INTERVAL);
    }

    /**
     * Show the tipsy, with our special behaviour
     * @param tipsyObject
     * @param {Events} eventAggregator
     */
    function showTipsy(tipsyObject, eventAggregator) {
        if (tipsyObject) {
            tipsyObject.show();
            if (tipsyObject.$tip) { // If title is empty there is no tip element even after calling show
                nudgeNSTooltipIntoView(tipsyObject);
                hideTipsyWhenDetached(tipsyObject);
                eventAggregator.trigger('show', tipsyObject);
            }
        }
    }

    GH.Tooltip = {};

    /**
     * Initialise tooltip
     * @param {Object} options
     * @param {string|Node} options.selector the element(s) to show tooltips for
     * @param {string|Node} [options.context] if specified, use event delegation on this element
     * @returns {Events} an event aggregator for tooltip events
     */
    GH.Tooltip.tipsify = function (options) {
        options = _.extend({}, options, {
            trigger: 'manual',
            title: 'data-tooltip'
        });

        var eventAggregator = new Events();

        // return if no selector defined
        if (!options.selector) {
            return eventAggregator;
        }

        if (options.context && _.isString(options.selector)) {
            // Delegated tooltip
            var selector = options.selector;
            jq(options.context).ghHoverIntent({
                interval: HOVERINTENT_INTERVAL,
                selector: selector,
                over: function() {
                    // Re-init the tooltip and show it immediately, provided the element is still attached
                    var $element = jq(this);
                    if ($element.is(':visible')) {
                        var tipsy = $element.tooltip(options).tipsy(true);
                        showTipsy(tipsy, eventAggregator);
                    }
                },
                out: function() {
                    // Check if tipsy is still present to be safe
                    var tipsy = jq(this).tipsy(true);
                    if (tipsy) {
                        tipsy.hide();
                    }
                }
            });
        } else {
            // Normal tooltip
            jq(options.selector)
                .tooltip(options)
                .ghHoverIntent({
                    interval: HOVERINTENT_INTERVAL,
                    over: function() {
                        var tipsy = jq(this).tipsy(true);
                        showTipsy(tipsy, eventAggregator);
                    },
                    out: function() {
                        jq(this).tipsy('hide');
                    }
                });
        }
        return eventAggregator;
    };

    /**
     * Enable and disable used in cases where Tipsy only required in one of multiple states e.g. toggles
     */

    /**
     * Enable tooltip on the given element(s)
     * @param {element|string} element
     */
    GH.Tooltip.enable = function(element) {
        var tipsy = jq(element).tipsy(true);
        if (tipsy) {
            tipsy.enable();
        }
    };

    /**
     * Disable tooltip on the given element(s)
     * @param {element|string} element
     */
    GH.Tooltip.disable = function(element) {
        var tipsy = jq(element).tipsy(true);
        if (tipsy) {
            tipsy.disable();
        }
    };

    /**
     * Hide tooltip either after it is shown or before (while hoverintent is still polling)
     * @param {element|string} element
     */
    GH.Tooltip.hide = function(element) {
        var tipsy = jq(element).tipsy(true);
        if (tipsy) {
            jq(element).trigger('mouseleave'); // Force hoverintent to stop polling
            tipsy.hide();
        }
    };
}(_, AJS.$, GH.Events));