AJS.test.require(["com.pyxis.greenhopper.jira:gh-test-common", "com.pyxis.greenhopper.jira:gh-rapid"], function () {
    var BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');

    module("handleIssueCreated", {
        setup: function setup() {
            this.sandbox = sinon.sandbox.create();
            this.context = AJS.test.mockableModuleContext();
            this.context.mock('jira-agile/rapid/ui/plan/analytics-helper', {
                baseEventData: sinon.stub().returns({})
            });
            this.logger = {
                trace: sinon.stub()
            };
            this.context.mock("jira/util/logger", this.logger);
            this.EpicController = this.context.require('jira-agile/rapid/ui/epic/epic-controller');
            GH.Test.setUpFakeServer(this);
            GH.AnalyticsEnabled = true;
            this.fakeEpic = {
                issueKey: 1,
                fields: {
                    issuetype: {
                        name: "Epic"
                    }
                }
            };
            GH.EpicConfig.epicConfig = {
                epicStatusFieldId: 1
            };
        },
        teardown: function teardown() {
            GH.Notification.clear();
            GH.Test.restoreServer(this);
            this.sandbox.restore();
        }
    });

    test("Successful Ajax request triggers the analytics", sinon.test(function () {
        this.stub(this.EpicController.analytics, "trigger");
        this.stub(GH.EpicConfig, "getEpicLabelFieldId").returns("customfield_1");

        GH.Test.respondToBareUrlGetWith200(this, '/rest/api/2/issue/', this.fakeEpic);

        this.EpicController.handleIssueCreated(null, { issues: [{ issueKey: 1 }] });

        this.server.respond();

        ok(this.EpicController.analytics.trigger.called, "Called success method");
    }));

    test("Failed request does not trigger analytics (no specific error handling)", sinon.test(function () {
        this.stub(this.EpicController.analytics, "trigger");
        this.stub(GH.EpicConfig, "getEpicLabelFieldId").returns("customfield_1");

        GH.Test.respondToBareUrlGetWith500(this, '/rest/api/2/issue/', {});

        this.EpicController.handleIssueCreated(null, { issues: [{ issueKey: 1 }] });

        this.server.respond();

        ok(!this.EpicController.analytics.trigger.called, "Does not trigger analytics on failure");
    }));

    test('addIssuesToEpic: should show proper success message', sinon.test(function () {
        var EPIC_KEY = 'epic1';
        var ISSUE_KEYS = ['issue-key1', 'issue-key2', 'issue-key3'];
        var traceSpy = this.logger.trace;
        var messageStub = this.sandbox.stub(GH.Notification, 'showSuccess');
        this.sandbox.stub(this.EpicController, 'addIssuesToEpicOnServer').returns(AJS.$.Deferred().resolve().promise());
        this.sandbox.stub(GH.BacklogModel, 'getIssueData', function (issueKey) {
            return {
                key: issueKey,
                typeId: 'random-id-which-must-be-different-than-epic-issue-type-id'
            };
        });

        var TEST_DATA = [{
            countIssues: 0,
            expectedArgs: ['gh.epic.issues.added', 0, undefined, undefined]
        }, {
            countIssues: 1,
            expectedArgs: ['gh.epic.issues.added', 1, 'issue-key1', undefined]
        }, {
            countIssues: 2,
            expectedArgs: ['gh.epic.issues.added', 2, 'issue-key1', 'issue-key2']
        }, {
            countIssues: 3,
            expectedArgs: ['gh.epic.issues.added', 3, 'issue-key1', 'issue-key2']
        }];

        TEST_DATA.forEach(function (spec) {
            this.EpicController.addIssuesToEpic(EPIC_KEY, ISSUE_KEYS.slice(0, spec.countIssues));

            ok(traceSpy.calledWithExactly.apply(traceSpy, spec.expectedArgs), "when countIssues=" + spec.countIssues + " 'getText' should be called with [" + spec.expectedArgs + "]; was: [" + traceSpy.getCall(0).args + "]");
            ok(messageStub.calledOnce, "when countIssues=" + spec.countIssues + " 'showSuccess' should be called once");

            traceSpy.reset();
            messageStub.reset();
        }, this);
    }));

    test('removeIssuesFromAssociatedEpics: should show proper success message', sinon.test(function () {
        var ISSUE_KEYS = ['issue-key1', 'issue-key2', 'issue-key3'];

        var traceSpy = this.logger.trace;
        var messageStub = this.sandbox.stub(GH.Notification, 'showSuccess');
        this.sandbox.stub(GH.PlanController, 'isActive').returns(true);
        this.sandbox.stub(this.EpicController, '_reloadDetailsView').returns();
        this.sandbox.stub(GH.Ajax, 'put').returns(AJS.$.Deferred().resolve().promise());
        this.sandbox.stub(GH.BacklogModel, 'removeEpicFromIssues', function (issueKeys) {
            return issueKeys.map(function (issueKey) {
                return {
                    key: issueKey,
                    typeId: 'random-id-which-must-be-different-than-epic-issue-type-id'
                };
            });
        });

        var TEST_DATA = [{
            countIssues: 1,
            expectedArgs: ['gh.epic.issues.removed', 1, 'issue-key1', undefined]
        }, {
            countIssues: 2,
            expectedArgs: ['gh.epic.issues.removed', 2, 'issue-key1', 'issue-key2']
        }, {
            countIssues: 3,
            expectedArgs: ['gh.epic.issues.removed', 3, 'issue-key1', 'issue-key2']
        }];

        TEST_DATA.forEach(function (spec) {
            this.EpicController.removeIssuesFromAssociatedEpics(ISSUE_KEYS.slice(0, spec.countIssues));

            ok(traceSpy.calledWithExactly.apply(traceSpy, spec.expectedArgs), "when countIssues=" + spec.countIssues + " 'getText' should be called with [" + spec.expectedArgs + "]; was: [" + traceSpy.getCall(0).args + "]");
            ok(messageStub.calledOnce, "when countIssues=" + spec.countIssues + " 'showSuccess' should be called once");

            traceSpy.reset();
            messageStub.reset();
        }, this);
    }));

    test("addIssuesToEpicOnServer: Ajax request sent and stringified", sinon.test(function () {
        this.EpicController.addIssuesToEpicOnServer();

        ok(GH.Test.isRequestDataStringified(this.server.requests[0]));

        this.server.respond();
    }));

    test('addIssuesToEpicOnServer: should handle general error message when adding sub-tasks', sinon.test(function () {
        var EPIC_KEY = 'epic1';
        var SUBTASK_KEY = 'issue-key';

        this.sandbox.stub(GH.Notification, 'showErrors');
        [401, 403, 500].forEach(function (statusCode) {
            GH.Test.respondWith(this, "/epics/" + EPIC_KEY + "/add", {}, 'PUT', statusCode);

            this.EpicController.addIssuesToEpicOnServer(EPIC_KEY, [SUBTASK_KEY]);
            this.server.respond();

            sinon.assert.calledOnce(GH.Notification.showErrors);
            var args = GH.Notification.showErrors.getCall(0).args[0];
            strictEqual(args.errors[0].status, statusCode, "error status should be " + statusCode);

            GH.Notification.showErrors.reset();
        }, this);
    }));

    test('addIssuesToEpicOnServer: should handle specific error message when adding sub-tasks', sinon.test(function () {
        var EPIC_KEY = 'epic1';
        var SUBTASK_KEY = 'issue-key';
        var ERROR_MSG = 'should show correct error message when adding a sub-task';
        var ERRORS = {
            errors: {
                'soft-error': ERROR_MSG
            },
            errorMessages: []
        };
        var CLOSEABLE = true;
        var OPTS = { autoHide: true, showTitle: false };

        this.sandbox.stub(GH.Notification, 'showErrors');
        GH.Test.respondWith(this, "/epics/" + EPIC_KEY + "/add", ERRORS, 'PUT', 400);

        this.EpicController.addIssuesToEpicOnServer(EPIC_KEY, [SUBTASK_KEY]);
        this.server.respond();

        sinon.assert.calledOnce(GH.Notification.showErrors);

        var args = GH.Notification.showErrors.getCall(0).args;
        var errors = args[0];
        var closeable = args[1];
        var opts = args[2];

        strictEqual(errors.errors[0].message, ERROR_MSG);
        strictEqual(closeable, CLOSEABLE, 'message should be closeable');
        deepEqual(opts, OPTS, 'message should be automatically closed and rendered without title');
    }));

    test("removeIssuesFromAssociatedEpics: Successful AJAX call reinitializes the form", sinon.test(function () {

        this.stub(GH.Notification, "showSuccess");
        this.stub(BacklogModel, "getIssueData").returns({ epic: 2 });

        GH.Test.respondToPutWith200(this, "/epics/remove", {});

        this.EpicController.removeIssuesFromAssociatedEpics([1]);

        ok(GH.Test.isRequestDataStringified(this.server.requests[0]));

        this.server.respond();

        ok(GH.Notification.showSuccess.called, "Called success method");
    }));

    test("removeIssuesFromAssociatedEpics: Failed AJAX call does nothing (no error handling)", sinon.test(function () {

        this.stub(GH.Notification, "showSuccess");
        this.stub(BacklogModel, "getIssueData").returns({ epic: 2 });

        GH.Test.respondToPutWith500(this, "/epics/remove", {});

        this.EpicController.removeIssuesFromAssociatedEpics([1]);

        ok(GH.Test.isRequestDataStringified(this.server.requests[0]));

        this.server.respond();

        ok(!GH.Notification.showSuccess.called, "Did not call success method");
    }));

    test("updateEpicAsDone: Successful AJAX call updates epic data", sinon.test(function () {

        this.stub(BacklogController, "updateEpicData");

        GH.Test.respondToPutWith200(this, "/xboard/issue/update-field.json", {});

        this.EpicController.updateEpicAsDone(10);

        ok(GH.Test.isRequestDataStringified(this.server.requests[0]));

        this.server.respond();

        ok(BacklogController.updateEpicData.called, "Called success method");
    }));

    test("updateEpicAsDone: Failed AJAX call does nothing (no error handling)", sinon.test(function () {

        this.stub(BacklogController, "updateEpicData");

        GH.Test.respondToPutWith500(this, "/xboard/issue/update-field.json", {});

        this.EpicController.updateEpicAsDone(10);

        ok(GH.Test.isRequestDataStringified(this.server.requests[0]));

        this.server.respond();

        ok(!BacklogController.updateEpicData.called, "Did not call success method");
    }));
});