(function(){

var _ = require('underscore');

if (!GH.Components) {
    GH.Components = {};
}

/**
 * Initialization function required for this component
 */
GH.Components.sprintFiltersInit = function() {
    // register inline dialog hacks
    GH.InlineDialog.registerInlineDialogHacks('sprintDialog', { disregardClickSelectors: ['.calendar', '.js-edit-sprintName-trigger', '.js-edit-startDate-trigger', '.js-edit-endDate-trigger']});
};

GH.Components.SPRINT_NAME_MAX_LENGTH = 30;

/**
 * Create a new SprintFilters component
 *
 * @param stateParamName the name to keep the selected filters state
 */
GH.Components.SprintFilters = function(stateParamName) {
    this.stateParamName = stateParamName;
};

GH.Components.SprintFilters.prototype.stateParamName = "workSprintFilters";
GH.Components.SprintFilters.prototype.rapidViewConfig = {};

/**
 * Set the configuration of the current rapid view
 */
GH.Components.SprintFilters.prototype.setRapidViewConfig = function(rapidViewConfig) {
    this.rapidViewConfig = rapidViewConfig;
};

/**
 * Get the active sprint filters
 */
GH.Components.SprintFilters.prototype.getActiveSprintFilters = function () {
    return GH.BoardState.getPerViewValue(this.stateParamName, []);
};

/**
 * Set the active sprint filters
 */
GH.Components.SprintFilters.prototype.setActiveSprintFilters = function (activeSprintFilters) {
    GH.BoardState.setPerViewValue(this.stateParamName, activeSprintFilters);
};

GH.Components.SprintFilters.prototype.renderComponent = function () {
    return AJS.$(GH.tpl.component.sprintfilters.renderSprintSelector());
};

GH.Components.SprintFilters.prototype.setSprintData = function(sprints) {
    this.sprints = sprints || [];
};

GH.Components.SprintFilters.prototype.getSprintData = function() {
    return this.sprints;
};

/**
 * Validates the current active selection. Replaces the url & triggers a filter change if outdated
 */
GH.Components.SprintFilters.prototype.validateActiveSprints = function() {
    // validate the current selection, remove invalid ids
    var activeIds = this.getActiveSprintFilters();
    var validIds = [];
    _.each(activeIds, function(id) {
        if (this.getSprintById(id)) {
            validIds.push(id);
        }
    }, this);

    // replace the active sprint ids and then replace the state
    if (!_.isEqual(activeIds, validIds)) {
        this.setActiveSprintFilters(validIds);
        GH.RapidBoard.State.replaceState();
        // load the board again
        AJS.$(this).trigger('sprintFilterChange');
    }
};

GH.Components.SprintFilters.prototype.hasSprints = function() {
    return this.sprints.length > 0;
};

GH.Components.SprintFilters.prototype.hasParallelActiveSprints = function() {
    return _.where(this.sprints, {state: 'ACTIVE'}).length > 1;
};

GH.Components.SprintFilters.prototype.getSprintById = function(sprintId) {
    return _.find(this.sprints, function(sprint) {
        return sprint.id === sprintId;
    });
};

GH.Components.SprintFilters.prototype.updateSprintData = function(sprintId, sprintData) {
    var updated = false;
    // we might not have any sprints yet (if plan/work mode has never been loaded)
    if (this.sprints) {
        for(var i=0; i < this.sprints.length; i++) {
            if (this.sprints[i].id == sprintId) {
                this.sprints[i] = sprintData;
                updated = true;
            }
        }
    }
    if (updated) {
        this.drawSprints();
    }
};

GH.Components.SprintFilters.prototype.drawSprints = function() {
    // update the value
    var self = this;
    // currently Sprint Filters are only on Work Controls. Scoping the container selector to make sure we don't accidentally
    // change something else in the future.
    var container = AJS.$('.ghx-controls-work .js-sprint-info');

    var activeSprintFilters = this.getActiveSprintFilters();
    var activeSprintMap = {};
    _.each(activeSprintFilters, function(filterId) {
        activeSprintMap[filterId] = true;
    });

    // render sprints
    container.html(GH.tpl.component.sprintfilters.renderSprints({
        sprints: this.sprints,
        activeSprintMap: activeSprintMap
    }));

    // render subnavigator if required
    if (GH.RapidBoard.ViewController.shouldShowSubnavigator()) {
        GH.RapidBoard.SubnavigatorController.render();
    }

    // attach handlers - but only if we got more than 1 sprint
    if (this.sprints.length > 1) {
        container.find('.js-toggle-sprint').click(function(event) {
            var button = AJS.$(this);
            self.toggleActiveSprintFilterButton(event, button);
        });
    }

    if (GH.RapidBoard.ViewController.hasSideBar) {
        var toolsContainer = AJS.$('#ghx-modes-tools');
        if (this.hasSprints()) {
            // Clear existing sprint data
            toolsContainer.find('.ghx-sprint-meta').remove();

            // If only 1 sprint, default to that
            var activeSprintId = this.sprints.length === 1 ? this.sprints[0].id : this.getActiveSprintFilters()[0];

            // Render new sprint controls
            if (activeSprintId) {
                var activeSprint = this.getSprintById(activeSprintId);
                var sprintMeta = AJS.$(GH.tpl.component.sprintfilters.renderSprintMeta({
                    sprint: activeSprint,
                    noSprintClosePermissions: !activeSprint.canUpdateSprint
                }));

                // Add dates tooltip
                sprintMeta.find('.time').tooltip({
                    html: true,
                    title: function () {
                        return GH.tpl.component.sprintfilters.renderSprintDatesTooltip({
                            sprint: activeSprint
                        });
                    }
                });

                // Enable tipsy with default behaviour for the Complete Sprint link,
                // since we want to take the advantage of default delayOut
                // interval so that user can click on the link within the tipsy.
                sprintMeta.find('.complete-sprint-container').tooltip({
                    html: true,
                    title: 'data-tooltip'
                });

                // Wire up Complete Sprint button
                sprintMeta.find('.js-complete-sprint').on('click', function (e) {
                    e.preventDefault();
                    GH.Dialogs.CompleteSprintDialog.showDialog(GH.RapidBoard.State.getRapidViewId(), activeSprintId);
                });

                toolsContainer.prepend(sprintMeta);
            }
        }
    } else {
        // delegate the Sprint Details dialog clicks
        container.find('.js-sprint-trigger').click(function(event) {
            var elem = AJS.$(this);
            self.openSprintDialog(elem);
        });
    }
};

GH.Components.SprintFilters.prototype.toggleActiveSprintFilterButton = function(event, button) {
    // prevent default, as href="#" would add a hash to the url/url change
    event.preventDefault();

    var sprintFilterId = parseInt(button.parents('.js-sprintfilter').attr('data-sprint-id'), 10);

    //Get the number encoded after the final dash in the button's id.
    this.toggleActiveSprintFilter(sprintFilterId);

    // load the board again
    AJS.$(this).trigger('sprintFilterChange');
};

GH.Components.SprintFilters.prototype.updateButtonStates = function() {
    var activeFilters = this.getActiveSprintFilters();
    AJS.$.each(AJS.$('.js-sprint-info').find('.js-sprintfilter'), function(index, elem) {
        var $elem = AJS.$(elem);
        var filterId = parseInt($elem.attr('data-sprint-id'), 10);
        if (_.contains(activeFilters, filterId)) {
            $elem.addClass('ghx-active');
        } else {
            $elem.removeClass('ghx-active');
        }
    });
};

GH.Components.SprintFilters.prototype.toggleActiveSprintFilter = function toggleActiveQuickFilter(sprintFilterId) {
    // we either select that filter, or if selected unselect it. We don't have multiselect currently
    var activeFilters = this.getActiveSprintFilters();
    var newFilters = [];
    if (activeFilters.length === 0 || activeFilters[0] !== sprintFilterId) {
        newFilters.push(sprintFilterId);
    }

    this.setActiveSprintFilters(newFilters);
    GH.RapidBoard.State.pushState();
};

GH.Components.SprintFilters.prototype.removeDialog = function() {
    GH.InlineDialog.remove('sprintDialog');
};

/**
 * Renders the dropdown menu for the Sprint
 */
GH.Components.SprintFilters.prototype.openSprintDialog = function(elem) {
    // remove previous dialog
    this.removeDialog();

    var sprintId = parseInt(elem.parents('.js-sprintfilter').attr('data-sprint-id'), 10);
    var sprint = this.getSprintById(sprintId);

    this.analytics("open");

    // create the inline dialog
    var dialogOpts = {
        fadeTime : 0,
        noBind : true,
        width: 400,
        cacheContent : false, // don't cache the dialog content
        hideDelay : 60000 // set longer timeout (default is 10 seconds)
    };
    var self = this;
    var dialog = GH.InlineDialog.create('sprintDialog', elem, function(contents, trigger, showPopup) {
        self.renderSprintDialog(contents, sprint, showPopup);
    }, dialogOpts);

    // then show
    dialog.show();
};

/**
 * Analytics for the sprint edit dropdown
 */
GH.Components.SprintFilters.prototype.analytics = function(action, label) {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    new AnalyticsTracker('gh.edit.sprint.inlinedialog.work.change').trigger(action, label); // SAFE
};

/**
 * Renders a Sprint Detail Dialog
 */
GH.Components.SprintFilters.prototype.renderSprintDialog = function(contents, sprint, showPopup) {

    contents.html(GH.tpl.component.sprintfilters.renderSprintDetails({
        sprint: sprint,
        canUpdateSprint: sprint.canUpdateSprint
    }));

    // mark this for GH styling
    contents.closest('.aui-inline-dialog').addClass('ghx-inline-dialog');

    // hide dialog, otherwise its not realigned
    contents.parent().hide();

    showPopup();
};

/**
 * Inline edit data getter.
 */
GH.Components.SprintFilters.prototype.createGetInlineEditDataFunction = function() {
    var self = this;
    return function(viewElement, options) {
        // force other edits to close
        GH.RapidBoard.Util.InlineEditable.submitActiveEdits();

        var editData = GH.RapidBoard.Util.InlineEditable.defaultFieldData(viewElement);
        if (editData.fieldName === 'sprintName') {
            editData.maxLength = GH.Components.SPRINT_NAME_MAX_LENGTH;
            editData.cssClass = 'medium-field';
        }
        // fetch the sprint id from the dialog
        editData.sprintId = parseInt(viewElement.parents('.js-sprint-details').attr('data-sprint-id'), 10);
        editData.sprintData = _.clone(self.getSprintById(editData.sprintId));
        editData.sprintFiltersComponent = self;
        return editData;
    };
};

/**
 * Submits updated sprint data
 */
GH.Components.SprintFilters.prototype.submitSprintData = function(sprintData) {
    var self = this;
    var sprintId = sprintData.id;
    var editSprintRequest = {
        name : sprintData.name,
        startDate : sprintData.startDate,
        endDate : sprintData.endDate
    };

    GH.Ajax.put({
        url: '/sprint/' + sprintId,
        data: editSprintRequest,
        errorContextMap: {
            'name' : '#ghx-sprint-name',
            'startDate' : '#ghx-sprint-start-date-error',
            'endDate' : '#ghx-sprint-end-date-error',
            'dateFormatMismatch': GH.Notification.handleDateFormatMismatchError
        }
    })
        .done(function(data) {
            self.updateSprintData(data.success.id, data.success);
        });
};

GH.Components.SprintFilters.prototype.initializeInlineEdits = function() {
    var self = this;

    // Validation for name, start and end date. These function also extract the value from the dom
    var validateName = function(editData) {
        var newValue = editData.editElement.val();
        if (!GH.Validation.notBlank(editData.editElement, AJS.I18n.getText('gh.sprint.error.name.required'))) {
            return false;
        }
        if (!GH.Validation.maxLength(editData.editElement, GH.Components.SPRINT_NAME_MAX_LENGTH, AJS.I18n.getText('gh.sprint.error.name.too.long', GH.Components.SPRINT_NAME_MAX_LENGTH))) {
            return false;
        }
        editData.sprintData.name = newValue;
        editData.newValue = newValue;
        return true;
    };
    var validateStartDate = function(editData) {
        var newValue = editData.editElement.find('input').val();
        if (!GH.Validation.notBlank(editData.editElement.find('input'), AJS.I18n.getText('gh.sprint.error.end.date.required'), editData.editElement)) {
            return false;
        }
        editData.sprintData.startDate = newValue;
        editData.newValue = newValue;
        return true;
    };
    var validateEndDate = function(editData) {
        var newValue = editData.editElement.find('input').val();
        if (!GH.Validation.notBlank(editData.editElement.find('input'), AJS.I18n.getText('gh.sprint.error.end.date.required'), editData.editElement)) {
            return false;
        }
        editData.sprintData.endDate = newValue;
        editData.newValue = newValue;
        return true;
    };

    // Saves the changed data, we always update the complete information, not just a field
    var saveSprint = function(editData) {
        // only submit changes if there are any, otherwise just update view
        var existingValues = _.clone(self.getSprintById(editData.sprintId));
        if (!_.isEqual(existingValues, editData.sprintData)) {
            self.submitSprintData(editData.sprintData);

            self.analytics("update", editData.fieldName);
        }
        // give fast feedback
        GH.RapidBoard.Util.InlineEditable.updateView(editData);
    };

    var preEdit = function(editData) {
        GH.Validation.clearContextualErrors('.ghx-sprint-active[data-sprint-id=' + editData.sprintId + ']');
        GH.RapidBoard.Util.InlineEditable.cancelActiveEdits();
    };

    // edit sprint name
    GH.RapidBoard.Util.InlineEditable.register('#inline-dialog-sprintDialog .js-edit-sprintName-trigger', {
        getData: this.createGetInlineEditDataFunction(),
        validate : validateName,
        save : saveSprint
    });
    GH.RapidBoard.Util.InlineEditable.register('#inline-dialog-sprintDialog .js-edit-startDate-trigger', {
        getData: this.createGetInlineEditDataFunction(),
        renderEdit: GH.RapidBoard.Util.InlineEditable.renderDatePickerInlineEdit,
        activate: GH.RapidBoard.Util.InlineEditable.datePickerActivate,
        validate : validateStartDate,
        preEdit: preEdit,
        save : saveSprint
    });
    GH.RapidBoard.Util.InlineEditable.register('#inline-dialog-sprintDialog .js-edit-endDate-trigger', {
        getData: this.createGetInlineEditDataFunction(),
        renderEdit: GH.RapidBoard.Util.InlineEditable.renderDatePickerInlineEdit,
        activate: GH.RapidBoard.Util.InlineEditable.datePickerActivate,
        validate : validateEndDate,
        preEdit: preEdit,
        save : saveSprint
    });
};
})();
