/* global GH */
/**
 * Epic Burndown Report Controller
 * @module jira-agile/rapid/ui/chart/v2/epicburndown/epic-burndown-report-controller
 * @requires module:underscore
 * @requires module:jira-agile/rapid/analytics-tracker
 * @requires module:jira-agile/rapid/ui/chart/v2/epicburndown/epic-burndown-chart-data-service
 * @requires module:jira-agile/rapid/ui/chart/v2/epicburndown/epic-burndown-chart-table-data-service
 * @requires module:jira-agile/rapid/global-events
 */
define('jira-agile/rapid/ui/chart/v2/epicburndown/epic-burndown-report-controller', ['require'], function (require) {
    'use strict';

    // REQUIRES

    var _ = require('underscore');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var EpicBurndownChartDataService = require('jira-agile/rapid/ui/chart/v2/epicburndown/epic-burndown-chart-data-service');
    var EpicBurndownChartTableDataService = require('jira-agile/rapid/ui/chart/v2/epicburndown/epic-burndown-chart-table-data-service');
    var GlobalEvents = require('jira-agile/rapid/global-events');

    var epicBurndownChartDataService = new EpicBurndownChartDataService();
    var epicBurndownChartTableDataService = new EpicBurndownChartTableDataService();

    // GLOBALS... FIX ME
    var PromiseKeeper;
    var RapidBoardState;
    var EpicBurndownReportView;
    var EpicsDataService;
    var epicsDataService;

    GlobalEvents.on("pre-initialization", function () {
        PromiseKeeper = GH.PromiseKeeper;
        RapidBoardState = GH.RapidBoard.State;
        EpicBurndownReportView = GH.Reports.EpicBurndownReportView;
        EpicsDataService = GH.EpicsDataService;
        epicsDataService = new EpicsDataService();
    });

    function EpicBurndownReportController() {}
    EpicBurndownReportController.id = 'epicBurndown';

    EpicBurndownReportController.prototype = {
        SETTINGS_KEY: 'chart.config.epicBurndown',
        displayed: false,

        _model: null,
        _rapidViewData: null,
        _tracker: null,

        setRapidView: function setRapidView(rapidViewData) {
            this._rapidViewData = rapidViewData || {};
        },

        show: function show() {
            this.displayed = true;
            this._tracker = new AnalyticsTracker('gh.report.epicburndown');

            RapidBoardState.pushState();

            // Initialise view
            var view = this._view = new EpicBurndownReportView();
            view.on('epicSelected', _.bind(function (epic) {
                this._model.epicKey = epic.id;
                this.update();
            }, this)).on('optionChanged', _.bind(function (changes) {
                this._model.zeroBaseline = changes.zeroBaseline;
                this._tracker.trigger('zeroBaseline.' + (changes.zeroBaseline ? 'on' : 'off'));
                this.update();
            }, this));
            view.init(this._model);
            view.showSpinner();

            this._tracker.trigger('show');

            this._dataPromiseKeeper = new PromiseKeeper();

            // Get epics and update the view
            var epicsPromise = epicsDataService.getSimpleEpics(this._rapidViewData.id);
            this._dataPromiseKeeper.add(epicsPromise).done(_.bind(function (epics) {
                if (!epics.length) {
                    this._model.epicKey = null;
                    view.showBlankState(true, true);
                    this._tracker.trigger('blank');
                    view.hideSpinner();
                } else {
                    epics.reverse();
                    if (this._model.epicKey == null || !epicsDataService.epicExists(epics, this._model.epicKey)) {
                        this._model.epicKey = epics[0].id;
                    }
                    view.setEpics(epics);
                    this.update(true);
                }
            }, this));
        },

        hide: function hide() {
            this.displayed = false;
            if (this._dataPromiseKeeper) {
                this._dataPromiseKeeper.reset();
            }
            if (this._view) {
                this._view.destroy();
                this._view = null;
            }
            epicBurndownChartDataService.clearCache();
            epicBurndownChartTableDataService.clearCache();
        },

        update: function update(isAfterLoadingEpics) {
            if (isAfterLoadingEpics) {
                RapidBoardState.replaceState();
            } else {
                RapidBoardState.pushState();
            }
            this.saveState();

            if (this._model.epicKey == null) {
                return;
            }

            var controller = this;

            // Get data and update chart
            controller._view.showSpinner();
            var dataPromise = epicBurndownChartDataService.get(controller._rapidViewData.id, controller._model.epicKey, {
                zeroBaseline: controller._model.zeroBaseline
            });

            return controller._dataPromiseKeeper.add(dataPromise).done(function (data) {
                // Toggle blankstate
                if (data.snapshot.estimatedIssueCount === 0) {
                    controller._view.showBlankState(false, true);
                    controller._tracker.trigger('blank');
                } else {
                    controller._view.hideBlankState();
                }
                controller._view.update(data, epicsDataService.supportsPages());

                // Get issue table data
                var tableDataPromise = epicBurndownChartTableDataService.get(controller._rapidViewData.id, controller._model.epicKey, data);
                controller._dataPromiseKeeper.add(tableDataPromise).done(function (issueData) {
                    controller._view.updateTable(issueData);
                });
            }).always(function (data) {
                if (data !== 'abort') {
                    controller._view.hideSpinner();
                }
            });
        },

        saveState: function saveState() {
            RapidBoardState.setBoardSetting(this.SETTINGS_KEY, this.getNormalizedFromInternal());
        },

        loadState: function loadState() {
            this._model = RapidBoardState.getBoardSetting(RapidBoardState.getRapidViewId(), this.SETTINGS_KEY, {
                epicKey: null,
                zeroBaseline: false
            });
        },

        /**
         * Is the report available for this type of board
         *
         * @param {Object} rapidViewConfig
         * @returns {boolean}
         */
        isApplicable: function isApplicable(rapidViewConfig) {
            return rapidViewConfig.sprintSupportEnabled;
        },

        /**
         * Return a normalized representation of the report state.
         *
         * @returns {Object}
         */
        getNormalizedFromInternal: function getNormalizedFromInternal() {
            if (this._model === null) {
                this.loadState();
            }
            return _.defaults(_.clone(this._model), {
                zeroBaseline: false
            });
        },

        /**
         * Update the report state from a normalized representation.
         * Saves the state to localStorage as part of the update.
         *
         * @param {Object} normalized
         */
        updateInternalFromNormalized: function updateInternalFromNormalized(normalized) {
            this._model = normalized;
        },

        /**
         * Convert a normalized representation of the report state into url query params.
         *
         * @param {Object} normalized
         * @returns {Object}
         */
        toUrl: function toUrl(normalized) {
            return {
                epic: normalized.epicKey !== null ? normalized.epicKey : undefined,
                zeroBaseline: normalized.zeroBaseline ? true : undefined
            };
        },

        /**
         * Convert url query params into a normalized representation of the report state.
         * Types must be cast correctly from string otherwise the URL manager will get confused when comparing equality.
         *
         * @param {Object} params
         * @returns {Object}
         */
        getNormalizedFromUrl: function getNormalizedFromUrl(params) {
            return {
                epicKey: params.epic || null,
                zeroBaseline: params.zeroBaseline === 'true'
            };
        }
    };

    return EpicBurndownReportController;
});