/* global c3: true */
/**
 * The scrubber component for the control chart.
 *
 * Adds:
 *  - a vertical line positioned next to the mouse
 *  - a dialog indicating the current time the mouse indicates or the brush range (if dragging)
 *  - a ControlChartBrush used to drill down into the chart.
 */
GH.Reports.ControlChartScrubber = function () {

    var scrubber = c3.scrubber().elementClass('ghx-scrubber').shouldHideScrubber(function () {
        return GH.Reports.ControlChartDialog.isVisible();
    });
    var brush = GH.Reports.ControlChartBrush();

    /**
     * Disable the scrubber
     */

    function disableScrubber() {
        GH.Reports.ControlChartScrubberDialog.disable();
    }

    /**
     * Enable the scrubber
     */
    function enableScrubber() {
        GH.Reports.ControlChartScrubberDialog.enable();
    }

    /**
     * Show the scrubber dialog.
     */
    function showScrubberDialog() {
        GH.Reports.ControlChartScrubberDialog.show();
    }

    /**
     * Hide the scrubber dialog.
     */
    function hideScrubberDialog() {
        GH.Reports.ControlChartScrubberDialog.hide();
    }

    /**
     * Hide the scrubber line.
     */
    function hideScrubber() {
        scrubber.hideScrubber();
    }

    function cancelDrag() {
        brush.cancelDrag();
    }

    function getStatsAtTime(series, time) {
        var stats = {};
        // Assumes that standard deviation changes at the same points in time as the rolling average
        // to save us from looping through both series
        if (series.rollingAverageLine.length && series.rollingAverageLine[0][0] <= time) {
            // Loop in reverse, breaking at the first point <= the specified time
            var i = GH.Util.indexOf(series.rollingAverageLine, function (d) {
                return d[0] <= time;
            }, true);
            stats.rollingAverage = series.rollingAverageLine[i][1];
            stats.standardDeviation = series.standardDeviationArea[i][2];
        }
        return stats;
    }

    /**
     * Refresh the dialog:
     *
     *  - Update the position of the dialog to be beside the mouse
     *  - Update the content of the dialog to display either the range (if dragging) or the current time
     *    indicated by the scrubber.
     */
    function refreshScrubberDialog() {
        var mouse = this.mouseCoordinates();
        var invert = this.xScale().invert;

        if (mouse) {
            if (brush.isDragging()) {
                var range = brush.currentRange();
                GH.Reports.ControlChartScrubberDialog.refresh({
                    startTime: invert(range[0]),
                    endTime: invert(range[1])
                });
            } else {
                var time = invert(mouse[0]);
                var stats = getStatsAtTime(this.data().series, time);
                GH.Reports.ControlChartScrubberDialog.refresh({
                    time: time,
                    rollingAverage: stats.rollingAverage,
                    standardDeviation: stats.standardDeviation
                });
            }
        }
    }

    /**
     * Initialise the dialog:
     *
     *  - Add the scrubber and brush components.
     *  - Initialise the scrubber dialog.
     *  - Bind to mouse events to update scrubber dialog.
     */
    function initialise() {

        var selection = this.selection();

        scrubber.parent(this);
        scrubber.interactiveLayer(this);
        scrubber(selection);

        brush.parent(this);
        brush.interactiveLayer(this);
        brush(selection);

        GH.Reports.ControlChartScrubberDialog.init(scrubber.elements()[0]);

        var self = this;

        // Replace the dragging dialog with the hover dialog when done dragging
        brush.onEndDrag(_.bind(refreshScrubberDialog, self));

        this.mouseover(function () {
            showScrubberDialog.call(self);
        });

        this.mousemove('ccscrubber', function () {
            refreshScrubberDialog.call(self);
        });

        this.mouseout(function () {
            hideScrubberDialog.call(self);
        });
    }

    return c3.component().extend(_.once(initialise)).extend(cancelDrag) // cancel drag if chart is updated
    .extend({
        hideScrubber: hideScrubber,
        disableScrubber: disableScrubber,
        enableScrubber: enableScrubber
    });
};