GH.Reports.ControlChartRefinementsView = (function (jq, I18n, InlineLayer, CheckboxMultiSelect) {
    var model = null;
    var previousModelNormalized = null;
    var handler = jq.noop;
    var columnsPicker;
    var lanesPicker;
    var filtersPicker;
    var $resetBtn;
    var isEventFromUI = true;

    /**
     * Initialise a JIRA Sparkler-style CheckboxMultiSelect triggered by a button
     * @param {jQuery} $scope jQuery object containing the dom elements for the multiselect and dropdown trigger
     * @param {String} options Property of the model containing options for this multiselct
     * @param {String} actives Property of the model containing options that should be checked
     * @param {String} sparklerInfo Info message to display inline with multiselect dropdown
     * @returns {{select: AJS.CheckboxMultiSelect, $button: {jQuery}}}
     * @private
     */
    function _setUpMultiPickers($scope, options, actives, sparklerInfo) {

        var $el = $scope.find('select');
        var $target = $scope.find('.ghx-refinement-selector');
        var opts = _.map(model.available[options], function (opt) {
            return {
                value: opt.id,
                label: opt.name,
                selected: false
            };
        });

        $el.empty().append(GH.tpl.reports.controlChart.renderSelectOptions({options: opts}));

        var multiselect = new CheckboxMultiSelect({
            element: $el,
            itemAttrDisplayed: 'label'
        });

        // Hide multiselect by default, and add container for info messages
        multiselect.$container
            .addClass('hidden')
            .append(GH.tpl.reports.controlChart.renderRefineInfoMessage({ message: sparklerInfo }));

        var dropdownLayer = InlineLayer.create({
            content: multiselect.$container,
            offsetTarget: $target,
            width: 'auto'
        });

        $target.on('click', function (e) {
            e.preventDefault();
            dropdownLayer.toggle();
            multiselect.$field.focus();
        });

        multiselect.model.$element.on('unselect selected', function () {
            if (isEventFromUI) {
                if (isInputValid()) {
                    // Update data model if selection is valid
                    model.selected[actives] = _.map(multiselect.model.getSelectedValues(), function (item) { return +item; });
                    digestModel();
                } else {
                    // Restore last valid selection
                    setTimeout(function () {
                        syncComponents();
                    }, 0);
                }
            }
        });

        return {
            multiSelect: multiselect,
            $button: $target
        };
    }


    /**
     * Validates refinements inputs and returns a boolean indicating validity of input
     *
     * @returns {boolean}
     * @private
     */
    function isInputValid () {
        var isColumnsInputValid = (columnsPicker.multiSelect.model.getSelectedValues().length > 0);
        var isLanesInputValid = lanesPicker ? (lanesPicker.multiSelect.model.getSelectedValues().length > 0) : true;

        return (isColumnsInputValid && isLanesInputValid);
    }


    /**
     * Initialise the controlls and events required for this view
     * @param {jQuery} $rootElement jQuery object containing the DOM elements for the view
     */
    function domInit($rootElement) {
        // Parent controller doesn't destroy subs on hide, so reset model or we don't register a change worth rendering
        previousModelNormalized = null;
        var swimlaneStrategy = GH.RapidViewConfig.currentData.data.swimlaneStrategy;
        var shouldShowSwimlaneRefine = swimlaneStrategy === 'custom';
        // Initialise all sparklers
        columnsPicker = _setUpMultiPickers($rootElement.find('#js-options-refine-columns'), 'columns', 'columnIds', AJS.I18n.getText('gh.rapid.charts.filters.error.nocolumns'));
        if(shouldShowSwimlaneRefine) {
            lanesPicker = _setUpMultiPickers($rootElement.find('#js-options-refine-swimlanes'), 'swimlanes', 'swimlaneIds', AJS.I18n.getText('gh.rapid.charts.filters.error.noswimlanes'));
        }

        filtersPicker = _setUpMultiPickers($rootElement.find('#js-options-refine-quickfilters'), 'quickFilters', 'quickFilterIds', '');

        // Reset link button
        $resetBtn = $rootElement.find('.js-reset-btn').on('click', function (e) {
            e.preventDefault();
            resetToDefault();
        });
    }


    /**
     * Checks for model changes, syncs model to UI and raises change event
     * @private
     */
    function digestModel() {
        if (!_.isEqual(model.toNormalized(), previousModelNormalized)) {
            // Don't trigger modelChanged if model is being initialized
            if (previousModelNormalized !== null) {
                handler('modelChanged');
            }
            syncComponents();
            previousModelNormalized = model.toNormalized();
        }
    }


    /**
     * Renders model state to UI components
     * @private
     */
    function syncComponents() {
        // deactivates event handlers
        isEventFromUI = false;

        updatePicker(columnsPicker, model.selected.columnIds);
        updatePicker(lanesPicker, model.selected.swimlaneIds);
        updatePicker(filtersPicker, model.selected.quickFilterIds);

        function updatePicker(picker, selectedIds) {
            if(picker) {
                picker.multiSelect.clear();
                _.each(selectedIds, function selectItemInPicker(selectedId) {
                    var descriptor = picker.multiSelect.model.getDescriptor(selectedId.toString());
                    picker.multiSelect.selectItem(descriptor, true);
                });

                var label = constructLabelFromElement(picker.multiSelect.model.$element);

                picker.$button.find('.ghx-refinement-wrap').text(label || AJS.I18n.getText('gh.boards.none'));
            }
        }

        function constructLabelFromElement($element) {
            var labelText =  $element.find('option:selected').map(function getSelectedText() {
                return this.text;
            }).get().join(', ');

            return labelText;
        }

        // Toggle reset button
        if (model.isDefault()) {
            $resetBtn.attr('aria-disabled', 'true');
        } else {
            $resetBtn.removeAttr('aria-disabled');
        }

        // reactivate event handlers
        isEventFromUI = true;
    }


    /**
     * Ask the controller to reset the model to the default selections
     */
    function resetToDefault() {
        handler('resetToDefaults');
    }


    return {
        /**
         * Initialise the view for the first time
         */
        start: function (refinements) {
            model = refinements;
            domInit(jq('#ghx-controlchart-refine-options'));
        },

        /**
         * Check if the view model has changed and sync UI components
         */
        digest: function () {
            digestModel();
        },

        /**
         * Bind handler methods from the controller to the view
         * @param {Function} delegate Handler method exposed by controller
         */
        setHandler: function (delegate) {
            handler = delegate;
        },

        /**
         * Return the view model
         * @returns {ControlChartRefinementsViewModel}
         */
        getModel: function () {
            return model;
        }
    };

}(AJS.$, AJS.I18n, AJS.InlineLayer, AJS.CheckboxMultiSelect));