/* global d3, c3, _ */

(function (Utilities, d3, c3, _, NumberFormat, ScopeBurndownBySprintChartDialog) {
    'use strict';

    var ADG_MAX_BAR_WIDTH = 80;
    var ADG_MIN_BAR_WIDTH = 40;
    var EMPTY_BAR_HEIGHT = 25;
    var MIN_INFOGRAPHIC_WIDTH = 320;
    var CHART_DIVIDER_OVERFLOW = 20;

    function getDisplayWorkRemaining(sprint) {
        return sprint.workRemaining - getDisplayWorkAdded(sprint);
    }

    function getDisplayWorkAdded(sprint) {
        var workAddedNet = Math.max(0, sprint.workAdded - sprint.workRemoved);
        // The displayable work added cannot be greater than the work remaining at the end of the sprint.
        // This case can occur when more work is added (and completed) in a sprint than there was at the start of the sprint.
        // E.g. 2 work at start of sprint. 5 is added and completed. => chart should show: -5 work completed, +2 work added.
        return Math.min(sprint.workRemaining, workAddedNet);
    }

    function isFutureSprint(sprint) {
        return sprint.isForecast && !sprint.isActive;
    }

    function ScopeBurndownBySprintChart() {
        var scopeBarChart = function scopeBarChart() {
            return c3.floatingBarChart().baselineAccessor(function (d) {
                return d.baseline;
            }).xAccessor(function (d) {
                return d.sprintId;
            }).yAccessor(function (d) {
                return d.baseline + d.workCompleted + getDisplayWorkRemaining(d) + getDisplayWorkAdded(d);
            }).maxBarWidth(ADG_MAX_BAR_WIDTH).extend({
                emptyBarHeight: c3.prop(EMPTY_BAR_HEIGHT),
                labelAccessor: function labelAccessor() {
                    return function (d) {
                        return d.isActive ? AJS.I18n.getText('gh.rapid.sprint.active.label', d.sprintName) : d.sprintName;
                    };
                },
                isCollapsedBar: function isCollapsedBar() {
                    return function (d) {
                        return !!d.collapsed;
                    };
                },
                isEmptyBar: function isEmptyBar() {
                    var baselineAccessor = this.baselineAccessor();
                    var yAccessor = this.yAccessor();
                    var isCollapsedBar = this.isCollapsedBar();

                    return function (d) {
                        return !isCollapsedBar(d) && yAccessor(d) === baselineAccessor(d);
                    };
                },
                barHeight: function barHeight() {
                    var baselineAccessor = this.baselineAccessor();
                    var yAccessor = this.yAccessor();
                    var yScale = this.yScale();
                    var isEmpty = this.isEmptyBar();
                    var emptyBarHeight = this.emptyBarHeight();

                    return function (d) {
                        return isEmpty(d) ? emptyBarHeight : yScale(baselineAccessor(d)) - yScale(yAccessor(d));
                    };
                },
                data: function data() {
                    return chart.sprints();
                }
            }).update(function (event) {
                var x = this.x();
                var y = this.y();
                var isEmpty = this.isEmptyBar();
                var emptyBarHeight = this.emptyBarHeight();

                event.selection.attr('transform', function (d, i) {
                    var yStart = y(d, i);

                    if (isEmpty(d)) {
                        yStart -= emptyBarHeight;
                    }

                    return 'translate(' + x(d, i) + ',' + yStart + ')';
                });
            });
        };

        var scopeStack = function scopeStack() {
            var stack = c3.stack().classPrefix('ghx').showLabels(true).extend({
                barWidth: c3.prop(),
                labelAccessor: function labelAccessor() {
                    return function (d) {
                        return d[2] || '';
                    };
                },
                labelFormatter: c3.prop()
            });
            stack.data.get(function (d) {
                if (!d) {
                    return [];
                }

                var workAdded = getDisplayWorkAdded(d);
                var workRemaining = getDisplayWorkRemaining(d);
                var formatter = this.labelFormatter();
                var stackData = [];
                stackData.push([workAdded + workRemaining, d.workCompleted, d.workCompleted ? '-' + formatter(d.workCompleted) : '']);
                stackData.push([workAdded, workRemaining, formatter(workRemaining)]);
                stackData.push([0, workAdded, workAdded ? '+' + formatter(workAdded) : '']);

                return stackData;
            });
            return stack;
        };

        var zeroHeightBar = function zeroHeightBar() {
            return c3.drawable().extend(c3.withDimensions()).enter(function (event) {
                var emptyBar = event.selection;

                emptyBar.append('rect').classed('ghx-series-0', true);
                // Add label
                emptyBar.append('text').classed('ghx-series-label-0 bar-label', true);
                // Add underline path to empty bar (because border-bottom isn't understood for SVG)
                emptyBar.append('polyline').classed('ghx-bar-underlined', true);
            }).update(function (event) {
                var height = this.height();
                var width = this.width();
                var emptyBar = event.selection;

                emptyBar.selectAll('rect').attr('height', height).attr('width', width);
                emptyBar.selectAll('text').attr('x', width / 2).attr('y', height / 2).attr('dy', '0.45em').text(0);
                emptyBar.selectAll('polyline').attr('points', '0,' + height + ' ' + width + ',' + height);
            });
        };

        var circleEllipsis = function circleEllipsis() {
            var radiusScale = d3.scale.linear().domain([ADG_MIN_BAR_WIDTH, ADG_MAX_BAR_WIDTH]).range([3, 5]);
            return c3.circlePlot().extend({
                yRange: function yRange() {
                    var height = this.height();
                    return [0, height];
                },
                radiusAccessor: function radiusAccessor() {
                    var width = this.width();
                    return function () {
                        return radiusScale(width);
                    };
                }
            }).elementClass('ghx-ellipsis').data([[0, 0], [1, 0], [2, 0]]).xDomain([-1, 3]).yDomain([-1, 1]);
        };

        var collapsedBar = function collapsedBar() {
            return c3.drawable().extend(c3.withDimensions()).elementClass('ghx-collapsed-sprint').extend({
                ellipsisHeight: function ellipsisHeight() {
                    return this.width() * 0.5;
                }
            }).enter(function (event) {
                event.selection.append('text').attr({
                    'text-anchor': 'middle',
                    y: -this.ellipsisHeight(),
                    dy: '-.4em'
                });
            }).update(function (event) {
                var width = this.width();

                event.selection.select('text').each(function (d) {
                    var textSelection = d3.select(this).text(AJS.I18n.getText('gh.rapid.chart.versionreport.sprintsnotshown', d));
                    c3.utils.wrapText(textSelection, width, true);
                });

                circleEllipsis().height(-this.ellipsisHeight()).width(width)(event.selection);
            });
        };

        var stackedScopeBarChart = function stackedScopeBarChart() {
            return scopeBarChart().elementTag('g').showLabels(true).update(function (event) {
                var barWidth = this.barWidth();
                var height = this.barHeight();
                var isEmpty = this.isEmptyBar();
                var isCollapsed = this.isCollapsedBar();
                var formatter = chart.data().estimationStatistic.formatter.formatCompact;
                var showLabels = this.showLabels();

                // Render each bar...
                event.selection.each(function (d) {
                    var stackHeight = height(d);
                    var isCollapsedSprint = isCollapsed(d);
                    var isZeroValueSprint = isEmpty(d);
                    var stackedBar = d3.select(this).classed('ghx-prediction', d.isForecast);

                    scopeStack().elementClass('ghx-stack').height(stackHeight).barWidth(barWidth).showLabels(showLabels).labelFormatter(formatter).data(!isCollapsedSprint && !isZeroValueSprint ? d : null)(stackedBar);

                    zeroHeightBar().elementClass('ghx-zero-value-sprint').height(stackHeight).width(barWidth).data(isZeroValueSprint ? [0] : [])(stackedBar);

                    collapsedBar().width(barWidth).data(isCollapsedSprint ? [d.collapsed] : [])(stackedBar);
                });
            });
        };

        var xAxis = c3.labelledAxis().height(60).text(AJS.I18n.getText('gh.sprint.selector.sprints')).orient('bottom').axisConstructor(function () {
            return d3.svg.axis().outerTickSize(0);
        });

        var yAxis = c3.labelledAxis().width(50).orient('left').extend({
            text: function text() {
                return chart.data().estimationStatistic.name;
            }
        }).axisConstructor(function () {
            var formatter = chart.data().estimationStatistic.formatter.formatCompact;
            return d3.svg.axis().outerTickSize(0).tickFormat(function (d) {
                return formatter(Math.abs(d));
            }).tickValues([0]);
        });

        var plots = c3.layerable().addLayer('grid', c3.gridLines().orient('left').extend({
            width: function width() {
                return chart.restrictedWidth();
            }
        })).addLayer('divider', c3.singular().extend(c3.withDimensions()).elementTag('line').elementClass('chart-divider').update(function (event) {
            event.selection.attr({
                x1: chart.restrictedWidth(),
                x2: chart.restrictedWidth(),
                y1: -CHART_DIVIDER_OVERFLOW,
                y2: this.height() + CHART_DIVIDER_OVERFLOW
            });
        })).addLayer('border', c3.drawable().extend(c3.withDimensions()).data([{ from: [0, 0], to: [1, 0] }, { from: [1, 0], to: [1, 1] }]).elementTag('line').update(function (event) {
            var x = d3.scale.linear().domain([0, 1]).range([0, this.width()]);
            var y = d3.scale.linear().domain([0, 1]).range([0, this.height()]);
            event.selection.each(function (d) {
                return d3.select(this).attr({
                    x1: x(d.from[0]),
                    y1: y(d.from[1]),
                    x2: x(d.to[0]),
                    y2: y(d.to[1])
                });
            });
        })).addLayer('sprints', stackedScopeBarChart().elementClass('ghx-bar-sprint').dataFilter(function (data) {
            return _.reject(data, isFutureSprint);
        })).addLayer('forecastSprints', stackedScopeBarChart().elementClass('ghx-bar-sprint').showLabels(false).dataFilter(function (data) {
            return _.filter(data, function (sprint) {
                return isFutureSprint(sprint) && sprint.workAtStart > 0;
            });
        })).addLayer('dialogTargets', scopeBarChart().enter(function (event) {
            // Close open dialogs when redrawing/resizing
            ScopeBurndownBySprintChartDialog.hide();

            event.selection.on('click.dialog', function (d) {
                // Add rapidViewId to sprint so we can generate sprint report URL
                var data = chart.data();
                d.rapidViewId = data.rapidViewId;
                ScopeBurndownBySprintChartDialog.show(this, d, data.estimationStatistic, data.labels);
            });
        }).elementClass('ghx-clickable-bar-sprint').dataFilter(function (data) {
            return _.reject(data, isFutureSprint);
        }));

        var chart = c3.component().extend(function () {
            var sprints = this.data().sprints;

            // Based on available width, and number of historic and future sprints,
            // determine the sprints to show.
            var fullWidth = this.positions().center.width;
            var restrictedWidth = fullWidth - MIN_INFOGRAPHIC_WIDTH;
            var bars = scopeBarChart();
            var barsPadding = bars.padding();
            var barsOuterPadding = bars.outerPadding();
            // Band width refers to bar width + padding
            var minBandWidth = ADG_MIN_BAR_WIDTH / (1 - barsPadding);
            var maxBandWidth = ADG_MAX_BAR_WIDTH / (1 - barsPadding);
            var bandWidth = Math.max(Math.min(restrictedWidth / sprints.length, maxBandWidth), minBandWidth);
            var numForecastBars = Math.max(Math.ceil(MIN_INFOGRAPHIC_WIDTH / bandWidth), Math.ceil((fullWidth - maxBandWidth * sprints.length) / bandWidth));

            // Truncate early sprints if necessary
            var maxBarCount = Math.floor((fullWidth - minBandWidth * (2 * barsOuterPadding - barsPadding)) / minBandWidth);
            var numBarsToRemove = sprints.length + numForecastBars - maxBarCount + 1; // Extra 1 because we need to fit in the collapsed bar
            if (numBarsToRemove > 1) {
                var firstSprintShown = sprints[1 + numBarsToRemove];
                sprints = sprints.slice();
                sprints.splice(1, numBarsToRemove, {
                    sprintId: 'sprints collapsed',
                    sprintName: '',
                    baseline: Math.min(0, firstSprintShown.baseline),
                    workAtStart: 0,
                    workAdded: 0,
                    workRemoved: 0,
                    workCompleted: 0,
                    workRemaining: 0,
                    isForecast: false,
                    isActive: false,
                    collapsed: numBarsToRemove
                });
            }

            var data = this.data();
            var forecastSprints = data.getForecast(numForecastBars);
            var allSprints = sprints.concat(forecastSprints);
            bandWidth = fullWidth / (allSprints.length + barsOuterPadding);
            // Different widths for the infographic section depending on whether forecasts are displayed
            if (data.forecast.forecastError) {
                this.restrictedWidth(bandWidth * (sprints.length + barsOuterPadding * 2 - barsPadding));
            } else {
                this.restrictedWidth(bandWidth * (sprints.length + barsOuterPadding - barsPadding / 2));
            }
            this.sprints(allSprints);
        }).extend(c3.borderLayout()).extend(c3.utils.defaultViewBox).center(plots).north(c3.withDimensions().height(25)).south(xAxis).west(yAxis).east(c3.withDimensions().width(1)) // so that the right border gets its full 1px of width
        .extend({
            sprints: c3.prop([]),
            restrictedWidth: c3.prop(null),
            xDomain: function xDomain() {
                return _.pluck(this.sprints(), 'sprintId');
            },
            yDomain: function yDomain() {
                var sprints = this.sprints();
                var min = _.min(_.pluck(sprints, 'baseline'));
                var max = d3.max(sprints, function (sprint) {
                    return sprint.baseline + sprint.workCompleted + getDisplayWorkAdded(sprint) + getDisplayWorkRemaining(sprint);
                });
                return [min, max];
            }
        });

        var infographicOverlay = c3.singular().extend(c3.withDimensions()).elementTag('div').elementClass('ghx-infographic-overlay').update(function (event) {
            var chartData = chart.data();

            // Render infographic contents
            if (chartData.forecast.forecastError) {
                event.selection.html(GH.tpl.reports.scopeBurndownBySprint.renderInfographicHelpMessage({
                    title: AJS.I18n.getText('gh.rapid.charts.scopeburndown.noforecast'),
                    description: chartData.forecast.forecastError,
                    helpUrl: chartData.labels.helpUrl
                }));
            } else if (chartData.forecast.sprintsRemaining === 0) {
                event.selection.html(GH.tpl.reports.scopeBurndownBySprint.renderInfographicNoSprints({
                    unestimatedIssuesCount: chartData.forecast.remainingEstimatableIssueCount,
                    imageURL: AJS.contextPath() + '/download/resources/' + Utilities.getPluginKey() + '/images/trophy.svg'
                }));
            } else {
                event.selection.html(GH.tpl.reports.scopeBurndownBySprint.renderInfographicContent({
                    sprintsRemaining: chartData.forecast.sprintsRemaining,
                    velocity: chartData.estimationStatistic.formatter.formatFull(chartData.forecast.velocity),
                    workRemaining: chartData.estimationStatistic.formatter.formatFull(chartData.forecast.workRemaining),
                    description: chartData.labels.forecastDescription(chartData.forecast.sprintsRemaining),
                    estimationStatistic: chartData.estimationStatistic.name,
                    helpUrl: chartData.labels.helpUrl
                }));
            }

            // Position the overlay so that it is centered in the infographic region.
            // Even though we have all the information needed to position the overlay using exact
            // left and top pixel values, it is positioned using a combination of percentage and pixel values
            // so it scales correctly for printing.
            var chartHeight = chart.height();
            var chartWidth = chart.width();
            var chartRegions = chart.positions();
            var infographicLeft = chartRegions.west.width + chart.restrictedWidth();
            var infographicWidth = chartWidth - infographicLeft;
            var infographicHeight = chartRegions.center.height;
            var overlayHeight = event.selection.node().offsetHeight;
            var overlayWidth = event.selection.node().offsetWidth;
            var overlayCenterX = infographicLeft + infographicWidth / 2;
            var overlayCenterY = chartRegions.north.height + infographicHeight / 2;
            event.selection.style({
                left: overlayCenterX / chartWidth * 100 + '%',
                top: overlayCenterY / chartHeight * 100 + '%',
                'margin-left': -overlayWidth / 2 + 'px',
                'margin-top': -overlayHeight / 2 + 'px'
            });
        });

        var chartWithOverlay = c3.component().extend(c3.withData()).extend(function () {
            var selection = this.selection();

            // Create svg element for the chart
            var svg = c3.singular().elementTag('svg').elementClass('ghx-svg-chart');
            svg(selection);
            chart.parent(this)(svg.elements());

            // Overlay
            infographicOverlay.parent(this)(selection);
        });

        return chartWithOverlay;
    }

    GH.Reports.ScopeBurndownBySprintChart = ScopeBurndownBySprintChart;
})(GH.Util, d3, c3, _, GH.NumberFormat, GH.Reports.ScopeBurndownBySprintChartDialog);