/**
 * A service that provides conversion between new and old time frame config.
 */
GH.ChartTimeFrameService = {};
GH.ChartTimeFrameService.rapidViewId = null;
GH.ChartTimeFrameService.normalizedDateFormat = '%Y-%m-%d';

GH.ChartTimeFrameService.setRapidViewId = function(rapidViewId){
    GH.ChartTimeFrameService.rapidViewId = rapidViewId;
};

/**
 * Returns the time frame configuration for the given chartId that is stored in local storage
 * @param chartId
 * @returns {*}
 */
GH.ChartTimeFrameService.getTimeFramesConfigForChart = function(chartId){
    var config = GH.RapidBoard.State.getBoardSetting(
        GH.ChartTimeFrameService.rapidViewId,
        'chart.config.' + chartId,
        {}
    );
    return config;

};

GH.ChartTimeFrameService.saveTimeFrameConfigForChart = function(chartId, config){
    var defaults = {
        view: 'reporting' // stop the entire board from reloading when we save
    };
    GH.RapidBoard.State.setBoardSetting('chart.config.' + chartId, _.defaults(config, defaults));
};

/**
 * Converts the given date into the string format used to store the date value
 * @param date
 * @private
 */
GH.ChartTimeFrameService._getFormattedDate = function(date){
    date = _.isNumber(date) ? new Date(date) : date;
    if(_.isDate(date)){
        return date.getFullYear() + '-' + (date.getMonth() + 1) + '-' + date.getDate();
    }
    return date;
};


/**
 * Converts the given formatted date string into a Date object
 *
 * @param date
 * @private
 */
GH.ChartTimeFrameService._convertDateFormattedStringToDate = function(formattedDate){
    if(_.isString(formattedDate) && formattedDate.length > 0){
        return new Date(Date.parse(formattedDate));
    }
    return null;

};

/**
 * Maps the values of the chart config of a chart of the old implementation, to the config of the new implementation
 *
 * @param chartId
 */
GH.ChartTimeFrameService.mapOldConfigToNewConfig = function(chartId){
    var rapidViewId = GH.ChartTimeFrameService.rapidViewId;
    var days = GH.RapidBoard.State.getBoardSetting(rapidViewId, 'chart.timeframe.days', GH.ChartTimeFrames.days);
    var from = GH.RapidBoard.State.getBoardSetting(rapidViewId, 'chart.timeframe.from', '');
    var to = GH.RapidBoard.State.getBoardSetting(rapidViewId, 'chart.timeframe.to', '');

    from = GH.ChartTimeFrameService._getFormattedDate(from);
    to = GH.ChartTimeFrameService._getFormattedDate(to);

    var chartConfig = GH.ChartTimeFrameService.getTimeFramesConfigForChart(chartId);
    chartConfig.days = days >= 0 ? days : 'custom' ;
    chartConfig.from = from;
    chartConfig.to = to;

    GH.ChartTimeFrameService.saveTimeFrameConfigForChart(chartId, chartConfig);
};

/**
 * Gets the chart timeframe config of a chart in the old implementation, but with the values of the new implementation
 *
 * @param chartId
 */
GH.ChartTimeFrameService.mapNewConfigToOldConfig = function(chartId){
    var chartConfig = GH.ChartTimeFrameService.getTimeFramesConfigForChart(chartId);
    var days = chartConfig.days,
        from = null,
        to = null;
    if(days < 0 || days === 'custom'){
        days = -1;
        try {
            from = GH.ChartTimeFrameService._convertDateFormattedStringToDate(chartConfig.from).getTime();
            to = GH.ChartTimeFrameService._convertDateFormattedStringToDate(chartConfig.to).getTime();
        } catch(e){
            // revert to the all time state if there is a problem parsing the custom dates
            days = 0;
        }
    } else if(_.isUndefined(days)){
        days = 0;
    }
    GH.ChartTimeFrames.setNewChartLimits(days, from, to, false);
};
