/* global AJS, GH */
/**
 * Report Controller
 * @module jira-agile/rapid/ui/chart/report-controller
 * @requires module:jira/featureflags/feature-manager
 * @requires module:jquery
 * @requires module:underscore
 * @requires module:jira-agile/rapid/analytics-tracker
 * @requires module:jira-agile/rapid/ui/chart/v2/controlchart/control-chart-controller
 * @requires module:jira-agile/rapid/ui/chart/v2/epicburndown/epic-burndown-report-controller
 * @requires module:jira-agile/rapid/ui/chart/v2/releaseburndown/release-burndown-report-controller
 * @requires module:jira-agile/rapid/ui/chart/burndown-chart-controller
 * @requires module:jira-agile/rapid/ui/chart/cfd-controller
 * @requires module:jira-agile/rapid/ui/chart/version-report-controller
 */
define('jira-agile/rapid/ui/chart/report-controller', ['require'], function (require) {
    var $ = require('jquery');
    var _ = require('underscore');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var BurndownChartController = require('jira-agile/rapid/ui/chart/burndown-chart-controller');
    var CFDController = require('jira-agile/rapid/ui/chart/cfd-controller');
    var ControlChartController = require('jira-agile/rapid/ui/chart/v2/controlchart/control-chart-controller');
    var EpicBurndownReportController = require('jira-agile/rapid/ui/chart/v2/epicburndown/epic-burndown-report-controller');
    var ReleaseBurndownReportController = require('jira-agile/rapid/ui/chart/v2/releaseburndown/release-burndown-report-controller');
    var SprintBurnupReportController = require('jira-agile/rapid/ui/chart/v2/burnup/burnup-report-controller');
    var VersionReportController = require('jira-agile/rapid/ui/chart/version-report-controller');
    var FeatureManager = require('jira/featureflags/feature-manager');
    var SprintActionControllerInit = require('jira-agile-sprint-actions-controller-init');
    var contextPath = require("wrm/context-path");

    /**
     * Chart view component.
     *
     * Loads chart data, then passes contorl to chart implementation for actual rendering
     */
    var ReportController = {};

    /**
     * All registered charts
     */
    ReportController.charts = [];

    /**
     * Current rapid view
     */
    ReportController.rapidViewData = {};
    ReportController.visible = false;
    ReportController.rapidViewConfig = {};

    /**
     * Analytics
     */
    ReportController.analytics = new AnalyticsTracker('gh.report');
    ReportController.ANALYTICS_TIME_SPENT_MILESTONES = [10 * 1000, 30 * 1000, 60 * 1000, 2 * 60 * 1000, 4 * 60 * 1000, 8 * 60 * 1000];

    /**
     * Initializes the report controller
     */
    ReportController.init = _.once(function () {
        // init charts
        ReportController.initCharts();
        // set the new data on all charts
        _.each(ReportController.getCharts(), function (chart) {
            chart.controller.setRapidView(ReportController.rapidViewData);
        });

        // initialize helpers
        GH.ChartTimeFrames.init();
        GH.ChartFilters.init();

        // handle new issue created
        $(GH).bind('issueCreated', ReportController.handleIssueCreated);

        $(GH).bind(GH.Dialogs.ReopenSprintDialog.EVENT_SPRINT_REOPENED, ReportController.handleSprintReopened);
    });

    /**
     * Initializes the available charts
     */
    ReportController.initCharts = function () {
        ReportController.charts = [{
            id: BurndownChartController.id,
            name: AJS.I18n.getText('gh.rapid.charts.burndown.storypoint.name'),
            description: AJS.I18n.getText('gh.rapid.charts.burndown.storypoint.description'),
            tooltip: AJS.I18n.getText('gh.rapid.charts.burndown.storypoint.tooltip'),
            preview: 'burndown',
            controller: BurndownChartController
        }, {
            id: GH.SprintRetrospectiveController.id,
            name: AJS.I18n.getText('gh.rapid.charts.sprint.report.name'),
            description: AJS.I18n.getText('gh.rapid.charts.sprint.report.description'),
            tooltip: AJS.I18n.getText('gh.rapid.charts.sprint.report.tooltip'),
            preview: 'sprint-retrospective',
            controller: GH.SprintRetrospectiveController
        }, {
            id: GH.EpicReportController.id,
            name: AJS.I18n.getText('gh.rapid.charts.epic.name'),
            description: AJS.I18n.getText('gh.rapid.charts.epic.description'),
            tooltip: AJS.I18n.getText('gh.rapid.charts.epic.tooltip'),
            preview: 'epic-report',
            controller: GH.EpicReportController
        }, {
            id: EpicBurndownReportController.id,
            name: AJS.I18n.getText('gh.rapid.charts.epicburndown.name'),
            description: AJS.I18n.getText('gh.rapid.charts.epicburndown.description'),
            tooltip: AJS.I18n.getText('gh.rapid.charts.epicburndown.tooltip'),
            howTos: [{
                title: AJS.I18n.getText('gh.rapid.charts.epicburndown.how.title.1'),
                description: AJS.I18n.getText('gh.rapid.charts.epicburndown.how.description.1')
            }, {
                title: AJS.I18n.getText('gh.rapid.charts.epicburndown.how.title.2'),
                description: AJS.I18n.getText('gh.rapid.charts.epicburndown.how.description.2')
            }, {
                title: AJS.I18n.getText('gh.rapid.charts.epicburndown.how.title.3'),
                description: AJS.I18n.getText('gh.rapid.charts.epicburndown.how.description.3')
            }],
            documentation: GH.HelpPaths.getHelpPath("viewing.epic.burndown").url,
            controller: new EpicBurndownReportController()
        }, {
            id: VersionReportController.id,
            name: AJS.I18n.getText('gh.rapid.charts.version.name'),
            description: AJS.I18n.getText('gh.rapid.charts.version.description'),
            tooltip: AJS.I18n.getText('gh.rapid.charts.version.tooltip'),
            preview: 'version-report',
            controller: VersionReportController
        }, {
            id: ReleaseBurndownReportController.id,
            name: AJS.I18n.getText('gh.rapid.charts.releaseburndown.name'),
            description: AJS.I18n.getText('gh.rapid.charts.releaseburndown.description'),
            tooltip: AJS.I18n.getText('gh.rapid.charts.releaseburndown.tooltip'),
            howTos: [{
                title: AJS.I18n.getText('gh.rapid.charts.releaseburndown.how.title.1'),
                description: AJS.I18n.getText('gh.rapid.charts.releaseburndown.how.description.1')
            }, {
                title: AJS.I18n.getText('gh.rapid.charts.releaseburndown.how.title.2'),
                description: AJS.I18n.getText('gh.rapid.charts.releaseburndown.how.description.2')
            }, {
                title: AJS.I18n.getText('gh.rapid.charts.releaseburndown.how.title.3'),
                description: AJS.I18n.getText('gh.rapid.charts.releaseburndown.how.description.3')
            }],
            documentation: GH.HelpPaths.getHelpPath("viewing.release.burndown").url,
            controller: new ReleaseBurndownReportController()
        }, {
            id: GH.VelocityChartController.id,
            name: AJS.I18n.getText('gh.rapid.charts.velocity.name'),
            description: AJS.I18n.getText('gh.rapid.charts.velocity.description'),
            tooltip: AJS.I18n.getText('gh.rapid.charts.velocity.tooltip'),
            preview: 'velocity',
            controller: GH.VelocityChartController
        }, {
            id: ControlChartController.id,
            name: AJS.I18n.getText('gh.rapid.charts.control.name'),
            description: AJS.I18n.getText('gh.rapid.charts.control.description'),
            tooltip: AJS.I18n.getText('gh.rapid.charts.control.tooltip'),
            preview: 'control-chart',
            howTos: [{
                title: AJS.I18n.getText('gh.rapid.chart.controlchart.how.title.1'),
                description: AJS.I18n.getText('gh.rapid.chart.controlchart.how.description.1')
            }, {
                title: AJS.I18n.getText('gh.rapid.chart.controlchart.how.title.2'),
                description: AJS.I18n.getText('gh.rapid.chart.controlchart.how.description.2')
            }, {
                title: AJS.I18n.getText('gh.rapid.chart.controlchart.how.title.3'),
                description: AJS.I18n.getText('gh.rapid.chart.controlchart.how.description.3')
            }],
            documentation: GH.HelpPaths.getHelpPath("viewing.control.chart").url,
            controller: ControlChartController
        }, {
            id: CFDController.id,
            name: AJS.I18n.getText('gh.rapid.charts.cfd.name'),
            description: AJS.I18n.getText('gh.rapid.charts.cfd.description'),
            tooltip: AJS.I18n.getText('gh.rapid.charts.cfd.tooltip'),
            preview: 'cumulative-flow-diagram',
            controller: CFDController
        }];

        if (FeatureManager.isFeatureEnabled('com.atlassian.jira.agile.darkfeature.burnupchart')) {
            // add the burnup as the first report (cause it's new!)
            ReportController.getCharts().unshift({
                id: 'burnupChart',
                name: AJS.I18n.getText('gh.rapid.charts.burnup.name'),
                description: AJS.I18n.getText('gh.rapid.charts.burnup.howto'),
                tooltip: AJS.I18n.getText('gh.rapid.charts.burnup.howto'),
                documentation: GH.HelpPaths.getHelpPath("viewing.burnup").url,
                controller: new SprintBurnupReportController()
            });
        }

        _.each(ReportController.getCharts(), function (chart) {
            if (chart.controller.init) {
                chart.controller.init();
            }
        });
    };

    /**
     * Get the available charts
     */
    ReportController.getCharts = function () {
        return ReportController.charts;
    };

    ReportController.getApplicableCharts = function () {
        return _.filter(ReportController.getCharts(), function (chart) {
            return chart.controller.isApplicable(ReportController.rapidViewConfig);
        });
    };

    ReportController.isApplicableChart = function (chartId) {
        var applicable = false;
        _.each(ReportController.getApplicableCharts(), function (chart) {
            if (chart.id === chartId) {
                applicable = true;
            }
        });
        return applicable;
    };

    /**
     * Get the chart object based on its id
     *
     * @param {string} chartId
     */
    ReportController.getChart = function (chartId) {
        return _.findWhere(ReportController.charts, { id: chartId });
    };

    ReportController.getSelectedChart = function () {
        return GH.RapidBoard.State.getChartType();
    };

    ReportController.renderReportView = function () {
        var shouldShowLeftHandSideChartSelector = !GH.RapidBoard.ViewController.shouldShowSubnavigator();

        // render the skeleton
        $('#ghx-report').html(GH.tpl.reportcontroller.renderReportView({
            shouldShowNavigation: shouldShowLeftHandSideChartSelector
        }));
        var $controls = $('#ghx-controls-report');

        // render the presentation mode button
        GH.PresentationMode.render(true, $controls);
    };

    ReportController.renderReopenButton = function renderReopenButton(element, rapidViewId, allSprints, sprint) {
        element.empty();
        if (sprint.state === 'CLOSED') {
            var parallelSprints = GH.RapidBoard.parallelSprints;
            var hasActiveSprints = _.any(allSprints, function (e) {
                return e.state === 'ACTIVE';
            });
            element.append(GH.tpl.reportcontroller.renderReopenSprintButton({
                noSprintReopenPermissions: !GH.RapidViewConfig.canManageSprints,
                noParallelSprintsAndSprintActive: !parallelSprints && hasActiveSprints
            }));
            element.find('#ghx-reopen-sprint').on('click', function (e) {
                e.preventDefault();
                GH.Dialogs.ReopenSprintDialog.showDialog(rapidViewId, sprint.id);
            });
        }
    };

    ReportController.renderActions = function renderActions(element, allSprints, sprint, sprintJql) {
        var rapidViewId = GH.RapidBoard.State.getRapidViewId();
        element.empty();
        if (ReportController.actionsController) {
            ReportController.actionsController.off();
        }
        // Do not initialize actions if there are no sprints
        if (!allSprints.length) {
            return;
        }

        ReportController.actionsController = SprintActionControllerInit.init({
            id: sprint.id,
            name: sprint.name,
            goal: sprint.goal,
            state: sprint.state,
            startDate: sprint.startDate,
            endDate: sprint.endDate
        }, allSprints);
        element.append(ReportController.actionsController.view.$el);

        ReportController.actionsController.bind('action:sprint-reopen', function () {
            GH.Dialogs.ReopenSprintDialog.showDialog(rapidViewId, sprint.id);
        });
        ReportController.actionsController.bind('action:sprint-view-in-issuenav', function () {
            window.location.href = contextPath() + '/issues/?jql=' + sprintJql;
        });
        ReportController.actionsController.on('sprint-edited sprint-deleted', function () {
            ReportController.handleSprintEdited();
        });
    };

    /**
     * Sets the rapid view data.
     */
    ReportController.setRapidView = function (rapidViewData) {
        ReportController.rapidViewData = rapidViewData || {};

        // set the new data on all charts
        _.each(ReportController.getCharts(), function (chart) {
            chart.controller.setRapidView(ReportController.rapidViewData);
        });
    };

    /**
     * Called to render the view
     */
    ReportController.show = function () {
        if (!ReportController.rapidViewData.id) {
            return;
        }
        ReportController.visible = true;
        GH.ChartTimeFrameService.setRapidViewId(ReportController.rapidViewData.id);

        // render the skeleton
        ReportController.renderReportView();

        ReportController.loadRapidViewConfig();
    };

    ReportController.loadRapidViewConfig = function () {
        var callback = function callback(data) {
            // ignore if we are not displayed anymore
            if (!ReportController.visible) {
                return;
            }

            ReportController.rapidViewConfig = data;

            ReportController.configureContent();
        };
        // this also needs to be delayed until the Sidebar API is ready - because handlers inside rely on it
        // when returning data and we'd have to add promises to the whole underlying API
        $.when(GH.RapidViewConfig.fetchConfiguration(ReportController.rapidViewData.id), JIRA.API.getSidebar()).done(callback);
    };

    /**
     * Renders the current chart
     */
    ReportController.configureContent = function () {

        // update visibility of certain view actions
        GH.ViewActions.setMode(GH.ViewActions.REPORT_MODE);

        // decide what to show if we haven't got a selected chart already
        var currentChart = GH.RapidBoard.State.getChartType();
        // ensure the chart is valid
        if (!ReportController.isApplicableChart(currentChart)) {
            currentChart = false;
        }
        if (!currentChart) {
            // fall back to first available chart
            var applicableCharts = ReportController.getApplicableCharts();
            GH.RapidBoard.State.setChartType(_.first(applicableCharts).id);
        }

        // render subnavigator, which in this case contains the header and report selector
        if (GH.RapidBoard.ViewController.shouldShowSubnavigator()) {
            GH.RapidBoard.SubnavigatorController.render();
        }

        // properly size the view
        GH.RapidBoard.ViewController.updateContentContainer();

        // show the view
        ReportController.showChart();
    };

    /**
     * Called when the view gets hidden
     */
    ReportController.hide = function () {
        if (ReportController.visible) {
            // tell charts
            ReportController.hideCharts();

            // mark as hidden
            ReportController.visible = false;

            // empty our DOM
            $('#ghx-report').empty();

            // Stop analytics time tracking
            if (ReportController.analyticsTimer) {
                ReportController.analyticsTimer.destroy();
            }
        }
    };

    /**
     * Tells all charts that they get hidden
     *
     * @param [exceptThis]
     */
    ReportController.hideCharts = function (exceptThis) {
        _.each(ReportController.getCharts(), function (chart) {
            if (chart.id !== exceptThis) {
                chart.controller.hide();
            }
        });
    };

    ReportController.selectChart = function (chartId) {
        GH.SprintRetrospectiveView.clearDialog();
        GH.RapidBoard.State.setChartType(chartId);
        ReportController.showChart();
    };

    ReportController.getCurrentChart = function () {
        var chartId = GH.RapidBoard.State.getChartType();
        if (!chartId) {
            return;
        }

        return ReportController.getChart(chartId);
    };
    /**
     * Shows the currently selected chart
     */
    ReportController.showChart = function () {

        var chart = ReportController.getCurrentChart();
        if (!chart) {
            return;
        }

        // highlight the newly selected chart in the nav
        ReportController.updateChartMenu(chart.id);

        // show the newly selected chart in the title
        ReportController.updateChartHeader(chart);

        // make sure all are hidden
        ReportController.hideCharts();

        // show the selected chart
        chart.controller.show();

        // Intro / How to state
        ReportController.handleIntroState();

        // Start analytics time tracking
        if (ReportController.analyticsTimer) {
            ReportController.analyticsTimer.destroy();
        }
        ReportController.analyticsTimer = new GH.DurationsTimer(ReportController.ANALYTICS_TIME_SPENT_MILESTONES).on('tick', function (duration) {
            var action = chart.id + (chart.inLabs ? '.new' : '') + ".view." + (ReportController.rapidViewConfig.sprintSupportEnabled ? "scrum" : "kanban");
            ReportController.analytics.trigger(action, 'duration', duration / 1000);
        });
    };

    ReportController.updateChartMenu = function (activeChart) {

        var $chartnav = $('#ghx-chart-nav');

        // render the selector
        $chartnav.html($(GH.tpl.reportcontroller.renderChartMenu({
            activeChart: activeChart,
            availableCharts: ReportController.getApplicableCharts(),
            board: ReportController.rapidViewConfig
        })));

        var $chartnavItems = $chartnav.find('li');

        // bind a click event to the nav items
        $chartnavItems.bind('simpleClick', GH.Util.wrapPreventDefault(function (event) {
            ReportController.handleChartSelection(event, $(this));
        }));

        // nuke all existing tipsies until we have better management of this kind of thing
        $('.tipsy').remove();

        // tooltips for each item with description of chart
        GH.Tooltip.tipsify({
            selector: $chartnavItems,
            gravity: 'w',
            className: 'ghx-tipsy-left'
        });
    };

    ReportController.updateChartHeader = function (activeChart) {
        var $messageContainer = $('#ghx-chart-message');
        $messageContainer.empty();

        // remove linked pages trigger
        $('#ghx-chart-header .ghx-linked-pages-trigger').remove();
        // remove selector contents
        $('#ghx-chart-selector').empty();

        // remove snapshot contents
        $('#ghx-chart-snapshot').empty();

        // hide legend and reopen sprint button (has to be shown again by correct chart)
        $('.ghx-chart-legend').hide();
        $('#ghx-chart-reopen-sprint').empty();
        $('#ghx-chart-actions').empty();

        // render the chart title
        $('#ghx-chart-title').html(GH.tpl.reportcontroller.renderChartTitle({
            activeChart: activeChart
        }));

        // render the chart intro
        $('#ghx-chart-intro').html(GH.tpl.reportcontroller.renderChartIntro({
            activeChart: activeChart
        }));

        // render the 'open' button for 'how to read this chart'
        $('#ghx-chart-help').html(GH.tpl.reportcontroller.renderChartDescriptionClosed());

        // bind a click event to the intro toggle
        $('#ghx-chart-panel-content').find('.js-chart-intro-toggle').on('click', function (event) {
            ReportController.toggleIntroState(event);
        });
    };

    /**
     * Handles the chart type selection
     */
    ReportController.handleChartSelection = function handleChartSelection(event, item) {

        // check which chart was selected
        var dataChart = item.find('a').attr('data-chart');

        // set the new chart
        ReportController.selectChart(dataChart);
    };

    ReportController.handleIssueCreated = function (event, data) {
        if (!ReportController.visible) {
            return;
        }

        // directly show message, we can't highlight issue on report tab
        GH.RapidBoard.QuickCreate.showCreatedIssuesMessage(data);
    };

    ReportController.handleSprintReopened = function (event, data) {
        GH.RapidBoard.ViewController.setMode('work');
        GH.CallbackManager.registerCallback(GH.WorkController.CALLBACK_POOL_RENDERED, 'afterSprintReopened', function afterSprintReopened() {
            GH.WorkControls.sprintfilters.setActiveSprintFilters([data]);
            $(GH.WorkControls.sprintfilters).trigger('sprintFilterChange');
            GH.RapidBoard.State.pushState();
        });
    };

    ReportController.handleSprintEdited = function (event, data) {
        GH.SprintRetrospectiveController.loadSprintOptions();
    };

    /**
     * Handles the visibility of the intro section for the charts
     */
    ReportController.handleIntroState = function handleIntroState() {

        if (GH.RapidBoard.State.getStoredChartIntroState() === false) {
            ReportController.closeIntro();
        }
    };

    /**
     * Animates the toggle for visibility of the intro section for the charts
     */
    ReportController.toggleIntroState = function toggleIntroState() {

        var $introContent = $('.ghx-intro-content');
        var $introTrigger = $('#ghx-chart-help');

        if ($introContent.hasClass('ghx-closed')) {
            // Show the intro content
            $introContent.removeClass('ghx-closed');
            GH.RapidBoard.Animation.animateHeightToAuto($introContent);

            // Hide the button to reopen the content
            $introTrigger.addClass('ghx-closed');

            GH.RapidBoard.State.setChartIntroState(true);
        } else {
            // Hide the intro content
            GH.RapidBoard.Animation.animateHeightToZero($introContent).done(function () {
                $introContent.addClass('ghx-closed');

                // Show the button to reopen the content
                $introTrigger.removeClass('ghx-closed');
            });

            GH.RapidBoard.State.setChartIntroState(false);
        }
    };

    /**
     * Close the intro section of a chart - no animation, used when stored value = user closed the intro
     */
    ReportController.closeIntro = function closeIntro() {
        $('.ghx-intro-content').height(0).addClass('ghx-closed');
        $('#ghx-chart-help').removeClass('ghx-closed');
    };

    return ReportController;
});