/* global GH */
/**
 * Changes Report Model
 * @module jira-agile/rapid/ui/chart/changes-report-model
 * @requires module:underscore
 * @requires module:jira-agile/rapid/ui/chart/burndown-rate
 * @requires module:jira-agile/rapid/ui/chart/burndown-table-producer
 * @requires module:jira-agile/rapid/ui/chart/burndown-timeline-producer
 * @requires module:jira-agile/rapid/ui/chart/chart-colors
 */
define('jira-agile/rapid/ui/chart/changes-report-model', ['require'], function (require) {
    'use strict';

    // REQUIRES

    var _ = require('underscore');
    var BurndownRate = require('jira-agile/rapid/ui/chart/burndown-rate');
    var BurndownTableProducer = require('jira-agile/rapid/ui/chart/burndown-table-producer');
    var BurndownTimelineProducer = require('jira-agile/rapid/ui/chart/burndown-timeline-producer');
    var ChartColors = require('jira-agile/rapid/ui/chart/chart-colors');

    // GLOBALS... FIX ME
    var FlotChartUtils = GH.FlotChartUtils;
    var ChartUtils = GH.ChartUtils;
    var TimeFormat = GH.TimeFormat;
    var NumberFormat = GH.NumberFormat;

    // Object used within VersionReportModel and EpicReportModel
    // Encapsulates common calculations from report data
    var ChangesReportModel = function ChangesReportModel() {};

    ChangesReportModel.prototype.setParams = function (params) {
        this.statistic = params.statistic;
        // Issue Added / Removed labels are specific to versions or epics
        this.issueAddedText = params.issueAddedText;
        this.issueRemovedText = params.issueRemovedText;
        this.startText = params.startText;
    };

    ChangesReportModel.prototype.setRawData = function (data) {
        this.data = data;
        this.processData();
    };

    ChangesReportModel.prototype.processData = function () {
        this.ensureIssuesHaveEstimateText();
        this.makeStatisticsValuesRenderable();
    };

    ChangesReportModel.prototype.setRawChartData = function (data) {
        this.chartData = data;
        this.processChartData();
    };

    ChangesReportModel.prototype.getStartTime = function () {
        return this.chartData.startTime;
    };

    ChangesReportModel.prototype.processChartData = function () {
        if (!this.chartData.endTime) {
            this.chartData.endTime = this.chartData.now;
        }
        BurndownRate.setRateData(this.chartData.workRateData);

        // calculate the timeline
        var statisticConsumer = BurndownTimelineProducer.ProgressConsumer;
        this.timelineData = BurndownTimelineProducer.calculateTimelineData(this.chartData, statisticConsumer);
        this.calculateSeries();
    };

    ChangesReportModel.prototype.hasIssues = function () {
        var content = this.getReportContent();
        if (content) {
            return content.completedIssues.length > 0 || content.incompleteEstimatedIssues.length > 0 || content.incompleteUnestimatedIssues.length > 0;
        } else if (this.chartData) {
            return !_.isEmpty(this.chartData.changes);
        }
        return false;
    };

    ChangesReportModel.prototype.getReportContent = function () {
        if (this.data) {
            return this.data.contents;
        }
    };

    ChangesReportModel.prototype.getRenderer = function () {
        return this.statistic.renderer;
    };

    ChangesReportModel.prototype.isIssueCountStatistic = function () {
        return this.statistic.typeId === "issueCount";
    };

    ChangesReportModel.prototype.isTimeTracking = function () {
        return this.statistic.fieldId === "timeSpent";
    };

    ChangesReportModel.prototype.shouldAddToSeries = function (serieId, issues) {
        var shouldAdd = false;

        _.each(issues, function (issue) {
            shouldAdd = shouldAdd || issue.deltas && issue.deltas[serieId.id] !== 0;
        });

        return shouldAdd;
    };

    ChangesReportModel.prototype.getAllSeriesDefinitions = function () {
        var statName = this.statistic.name;
        return {
            totalEstimate: {
                id: 'totalEstimate',
                label: AJS.I18n.getText('gh.rapid.charts.progress.total.estimate', statName),
                renderer: this.getRenderer(),
                statisticName: statName
            },
            completedEstimate: {
                id: 'completedEstimate',
                label: AJS.I18n.getText('gh.rapid.charts.progress.total.complete', statName),
                renderer: this.getRenderer(),
                statisticName: statName
            },
            issueCount: {
                id: 'issueCount',
                label: AJS.I18n.getText('gh.rapid.charts.progress.total.issuecount'),
                renderer: "number",
                statisticName: AJS.I18n.getText('gh.rapid.charts.progress.issuecount')
            },
            unestimatedIssueCount: {
                id: 'unestimatedIssueCount',
                label: AJS.I18n.getText('gh.rapid.charts.progress.unestimated.issues'),
                renderer: "number",
                statisticName: AJS.I18n.getText('gh.rapid.charts.progress.issuecount')
            }
        };
    };

    ChangesReportModel.prototype.calculateSeries = function () {
        var seriesIds = [];
        var seriesDefinition = this.getAllSeriesDefinitions();

        if (this.isIssueCountStatistic()) {
            seriesIds.push(_.defaults({
                color: ChartColors.scope,
                yaxis: 1,
                fill: 0.3,
                lineWidth: 1
            }, seriesDefinition.totalEstimate));
            seriesIds.push(_.defaults({
                color: ChartColors.completed,
                yaxis: 1,
                fill: 0.2,
                lineWidth: 1
            }, seriesDefinition.completedEstimate));
        } else {
            seriesIds.push(_.defaults({
                color: ChartColors.scope,
                yaxis: 1,
                fill: 0.3,
                lineWidth: 1
            }, seriesDefinition.totalEstimate));
            seriesIds.push(_.defaults({
                color: ChartColors.completed,
                yaxis: 1,
                fill: 0.2,
                lineWidth: 1
            }, seriesDefinition.completedEstimate));
            seriesIds.push(_.defaults({
                color: ChartColors.issueCount,
                yaxis: 2,
                lineWidth: 1
            }, seriesDefinition.issueCount));
            seriesIds.push(_.defaults({
                color: ChartColors.unestimated,
                yaxis: 2,
                lineWidth: 1
            }, seriesDefinition.unestimatedIssueCount));
        }

        this.calculateSeriesFromIds(seriesIds);
    };

    ChangesReportModel.prototype.calculateSeriesFromIds = function (seriesIds) {
        var data = this.timelineData;
        var timeline = data ? data.timeline : undefined;
        var series = this.series = [];

        if (_.isEmpty(timeline)) {
            return;
        }

        // we need four series: total issue count, unestimated and incomplete issue count,
        // total estimate for incomplete estimated work, total estimate for completed work


        // we store the series points and the data associated with these points
        var seriesPoints = {};
        var seriesData = {};
        _.each(seriesIds, function (serieId) {
            seriesPoints[serieId.id] = [];
            seriesData[serieId.id] = [];
        });

        // process the timeline
        _.each(timeline, function (timelineEntry) {

            // handle each series separately
            _.each(seriesIds, function (serieId) {

                // we have four series, only need to add certain types of events to each
                if (this.shouldAddToSeries(serieId, timelineEntry.issues)) {
                    // add a point for the current time with previous value, but only if delta is non-null
                    if (timelineEntry.deltas && _.isNumber(timelineEntry.deltas[serieId.id]) && timelineEntry.deltas[serieId.id] != 0) {
                        // old value at given time
                        seriesPoints[serieId.id].push([timelineEntry.time, timelineEntry.values[serieId.id] - timelineEntry.deltas[serieId.id]]);
                        seriesData[serieId.id].push({});
                    }

                    // new value at given time
                    seriesPoints[serieId.id].push([timelineEntry.time, timelineEntry.values[serieId.id]]);
                    seriesData[serieId.id].push(timelineEntry);
                }
            }, this);
        }, this);

        // Add a point for now or the end date of the version
        var lastTimelineEntry = _.last(timeline);
        if (!data.completeTime || data.completeTime > data.now) {
            _.each(seriesIds, function (serieId) {
                seriesPoints[serieId.id].push([Math.max(data.startTime, data.now), lastTimelineEntry.values[serieId.id]]);
            });
        } else {
            _.each(seriesIds, function (serieId) {
                seriesPoints[serieId.id].push([data.completeTime, lastTimelineEntry.values[serieId.id]]);
            });
        }

        // put in the series
        if (this.isTimeTracking()) {
            series.push({
                id: "timeSpent",
                data: seriesPoints['timeSpent'],
                color: ChartColors.timeSpent,
                label: AJS.I18n.getText('gh.issue.time')
            });
        }

        _.each(seriesIds, function (serieId) {
            series.push({
                id: serieId.id,
                data: seriesPoints[serieId.id],
                color: serieId.color,
                label: serieId.label,
                yaxis: serieId.yaxis,
                lines: { fill: serieId.fill, lineWidth: serieId.lineWidth, show: true }
            });
        });

        // Add an empty series for non-working days to populate legend
        series.push({
            id: "markings",
            color: ChartColors.nonWorkingDays,
            data: [],
            label: AJS.I18n.getText('gh.rapid.chart.burndown.nonworkingdays.label')
        });

        // set up the y-axis data
        var maxValue = data.maxValues['totalEstimate'];
        this.yAxisData = FlotChartUtils.calculateYAxis(maxValue, this.statistic);
        this.yAxisData.min = 0;
        // Remove padding at the top of the chart so 100% is at the height of the maximum value:
        this.yAxisData.max = maxValue;

        // done filtering the issues
        this.series = series;

        this.seriesData = seriesData;
    };

    ChangesReportModel.prototype.getXAxisStart = function () {
        // add a percent of the width at the left side, otherwise the start of the epic kisses the left y axis
        var leftSideBuffer = 0.02;
        return this.chartData.startTime - (this.chartData.endTime - this.chartData.startTime) * leftSideBuffer;
    };

    ChangesReportModel.prototype.getYAxes = function () {
        var yAxes = [];
        yAxes.push(this.yAxisData);
        if (!this.isIssueCountStatistic()) {
            var maxValue = this.timelineData.maxValues["issueCount"];
            var minTickSize = 1;
            if (maxValue > 15) {
                // 15 or more and there's a chance it can make the tick size things like '2.5'
                minTickSize = Math.ceil(maxValue / 8); // max 8 ticks
            }
            yAxes.push({
                position: "right",
                min: 0,
                tickFormatter: function tickFormatter(v, axis) {
                    // obliterate fractional part
                    return v.toFixed(0);
                },
                tickSize: minTickSize
            });
        }
        return yAxes;
    };

    ChangesReportModel.prototype.getDataBySeries = function (seriesId) {
        return this.seriesData && this.seriesData[seriesId];
    };

    /**
     * Renders the tooltip for a given seriesId and dataPoint
     */
    ChangesReportModel.prototype.getTooltipData = function (seriesId, timelineEntry) {

        var tooltipData = {};

        // Render an appropriate tooltip for sprint start/end
        if (timelineEntry.initial) {
            if (timelineEntry.initial) {
                tooltipData.initial = true;
                tooltipData.event = this.getEpicStartChangeEvent();
            }
            tooltipData.issueCount = _.size(timelineEntry.issues);
            tooltipData.statisticName = this.getSeriesStatisticName(seriesId);
            tooltipData.statisticValue = ChartUtils.renderStatisticText(timelineEntry.values[seriesId], false, this.getSeriesRenderer(seriesId));
        }

        // render a change tooltip
        else {
                tooltipData.changes = [];
                _.each(timelineEntry.issues, function (issue) {
                    var issueChange = {};
                    issueChange.key = issue.key;
                    issueChange.statisticName = this.getSeriesStatisticName(seriesId);
                    issueChange.statisticDelta = ChartUtils.renderStatisticText(issue.deltas[seriesId], false, this.getSeriesRenderer(seriesId));
                    issueChange.event = this.getChangeEvent(issue);
                    tooltipData.changes.push(issueChange);
                }, this);
            }
        tooltipData.dateText = ChartUtils.renderUTCMillisAsDate(timelineEntry.time);
        return tooltipData;
    };

    ChangesReportModel.prototype.getSeriesRenderer = function (seriesId) {
        var series = this.getAllSeriesDefinitions();
        return series[seriesId].renderer;
    };

    ChangesReportModel.prototype.getSeriesStatisticName = function (seriesId) {
        var series = this.getAllSeriesDefinitions();
        return series[seriesId].statisticName;
    };

    /**
     * Get the data entry for a specific index inside a series
     */
    ChangesReportModel.prototype.getDataBySeriesAtIndex = function (seriesId, index) {
        var series = this.getDataBySeries(seriesId);
        if (series) {
            return series[index];
        }
    };

    ChangesReportModel.prototype.getChangeEvent = function (issue) {
        // handle sprint in/out or column change
        var burndownEventTypeName = AJS.I18n.getText('gh.report.progress.event.burndown.type');
        var changeEvent = this.getScopeOrColumnChangeEvent(issue, burndownEventTypeName);
        if (changeEvent) {
            return changeEvent;
        }

        // handle statistic change
        if (issue.estimateChange) {
            // data to display
            var oldEstimate = issue.values.oldEstimateFieldValue;
            var newEstimate = issue.values.estimateFieldValue;
            var hasOldEstimate = !_.isUndefined(oldEstimate);
            var hasNewEstimate = !_.isUndefined(newEstimate);

            if (hasOldEstimate && hasNewEstimate) {
                return {
                    type: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.type'),
                    detail: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.detail.statistic.value.changed', this.renderStatisticText(oldEstimate), this.renderStatisticText(newEstimate))
                };
            } else if (hasOldEstimate) {
                return {
                    type: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.type'),
                    detail: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.detail.statistic.value.removed', this.renderStatisticText(oldEstimate))
                };
            } else {
                return {
                    type: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.type'),
                    detail: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.detail.statistic.value.added', this.renderStatisticText(newEstimate))
                };
            }
        }

        // should never happen
        return {
            type: 'unknown',
            detail: 'unknown change'
        };
    };

    ChangesReportModel.prototype.getScopeOrColumnChangeEvent = function (issue, burndownEventTypeName) {
        // handle issue added/removed from epic
        if (issue.scopeChange) {
            if (issue.inScope) {
                return {
                    type: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.type'),
                    detail: this.issueAddedText
                };
            } else {
                return {
                    type: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.type'),
                    detail: this.issueRemovedText
                };
            }
        }

        return BurndownTableProducer.getColumnChangeEvent(issue, burndownEventTypeName);
    };

    /**
     * Change event  for sprint start
     */
    ChangesReportModel.prototype.getStartChangeEvent = function () {
        return {
            type: this.startText,
            detail: ''
        };
    };

    ChangesReportModel.prototype.getSeries = function () {
        return this.series;
    };

    ChangesReportModel.prototype.getXAxisEnd = function () {
        // By default, let the chart figure it out
        return undefined;
    };

    ChangesReportModel.prototype.ensureIssuesHaveEstimateText = function () {
        var content = this.getReportContent();
        var renderer = this._getValueRenderer();

        // ensure all issues have an estimate
        function setEstimate(issue) {
            // if there is no estimate statistic, we should put in a dash
            if (_.isUndefined(issue.estimateStatistic) || _.isUndefined(issue.estimateStatistic.statFieldValue.value)) {
                issue.estimateStatistic = {
                    statFieldValue: {
                        text: '-'
                    }
                };
            } else {
                issue.estimateStatistic.statFieldValue.text = renderer(issue.estimateStatistic.statFieldValue.value);
            }
            return issue;
        }

        content.completedIssues = _.map(this.data.contents.completedIssues, setEstimate);
        content.incompleteUnestimatedIssues = _.map(this.data.contents.incompleteUnestimatedIssues, setEstimate);
        content.incompleteEstimatedIssues = _.map(this.data.contents.incompleteEstimatedIssues, setEstimate);
    };

    ChangesReportModel.prototype._getValueRenderer = function () {
        var renderer;
        if (this.getStatistic().renderer === "duration") {
            renderer = TimeFormat.formatShortDurationForTimeTrackingConfiguration;
        } else {
            renderer = NumberFormat.format;
        }
        return renderer;
    };

    ChangesReportModel.prototype.makeStatisticsValuesRenderable = function () {
        var content = this.getReportContent();
        var renderer = this._getValueRenderer();

        // use the renderer to populate text for statistics values
        content.completedIssuesEstimateSum.text = renderer(content.completedIssuesEstimateSum.value);
        content.incompletedIssuesEstimateSum.text = renderer(content.incompletedIssuesEstimateSum.value);
    };

    ChangesReportModel.prototype.getJql = function () {
        return this.jql;
    };

    ChangesReportModel.prototype.getDoneStatuses = function () {
        return this.data.doneStatuses;
    };

    ChangesReportModel.prototype.getNotDoneStatuses = function () {
        return this.data.notDoneStatuses;
    };

    ChangesReportModel.prototype.renderStatisticText = function (statisticValue, abs) {
        return ChartUtils.renderStatisticText(statisticValue, abs, this.getStatistic().renderer);
    };

    return ChangesReportModel;
});