/**
 * Change Event
 * @module jira-agile/rapid/ui/chart/change-event
 * @requires module:underscore
 */
define('jira-agile/rapid/ui/chart/change-event', ['require'], function (require) {
    'use strict';

    // REQUIRES
    var _ = require('underscore');

    /**
     * Object used for the purposes of calculating release date
     * to understand the type of change event and how to handle it
     */
    var ChangeEvent = {
        scopeAdded: "scopeAdded",
        scopeRemoved: "scopeRemoved",
        reopened: "reopened",
        burndown: "burndown",
        completeScopeAdded: "completeScopeAdded",
        estimateChanged: "estimateChanged",
        estimateAdded: "estimateAdded",
        estimateRemoved: "estimateRemoved",
        releaseDate: "releaseDate",
        uncertainty: "uncertainty",
        unknown: "unknown",
        startDate: "startDate"
    };

    ChangeEvent.getScopeOrColumnChangeEvent = function (issue, burndownEventTypeName) {
        // handle issue added/removed from scope
        if (issue.scopeChange) {
            if (issue.inScope) {
                return ChangeEvent.scopeAdded;
            } else {
                return ChangeEvent.scopeRemoved;
            }
        }

        // handle column change
        if (issue.columnChange) {
            if (issue.column === 'notDone') {
                if (issue.oldColumn) { // moved from done column
                    return ChangeEvent.reopened;
                } else { // open issue moved from unmapped column to a mapped one
                    return ChangeEvent.scopeAdded;
                }
            }
            if (issue.column === 'done') {
                if (issue.oldColumn) { // moved from notDone column
                    return ChangeEvent.burndown;
                } else { // moved from unmapped column
                    return ChangeEvent.completeScopeAdded;
                }
            }
            else { // moved into unmapped column
                if (issue.oldColumn === 'notDone') { // moved from not done column
                    return ChangeEvent.scopeRemoved;
                } else { // moved from done column
                    return ChangeEvent.scopeRemoved;
                }
            }
        }

        // none of the above
        return false;
    };

    ChangeEvent.getChangeEvent = function (issue) {
        // handle sprint in/out or column change
        var burndownEventTypeName = AJS.I18n.getText('gh.report.progress.event.burndown.type');
        var changeEvent = ChangeEvent.getScopeOrColumnChangeEvent(issue, burndownEventTypeName);
        if (changeEvent) { return changeEvent; }

        // handle statistic change
        if (issue.estimateChange) {
            // data to display
            var oldEstimate = issue.values.oldEstimateFieldValue;
            var newEstimate = issue.values.estimateFieldValue;
            var hasOldEstimate = !_.isUndefined(oldEstimate);
            var hasNewEstimate = !_.isUndefined(newEstimate);

            if (hasOldEstimate && hasNewEstimate) {
                return ChangeEvent.estimateChanged;
            } else if (hasOldEstimate) {
                return ChangeEvent.estimateRemoved;
            } else {
                return ChangeEvent.estimateAdded;
            }
        }

        // should never happen
        return ChangeEvent.unknown;
    };

    return ChangeEvent;
});
