/* global GH */
/**
 * CFD Controller
 * @module jira-agile/rapid/ui/chart/cfd-controller
 * @requires module:jquery
 * @requires module:jira-agile/rapid/analytics-tracker
 * @requires module:jira-agile/rapid/ui/chart/cfd-data
 * @requires module:jira-agile/rapid/ui/chart/chart-controller
 * @requires module:jira-agile/rapid/ui/chart/chart-view
 */
define('jira-agile/rapid/ui/chart/cfd-controller', ['require'], function(require) {
    'use strict';

    // REQUIRES
    var $ = require('jquery');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var CFDData = require('jira-agile/rapid/ui/chart/cfd-data');
    var CFDView = require('jira-agile/rapid/ui/chart/cfd-view');
    var ChartController = require('jira-agile/rapid/ui/chart/chart-controller');
    var ChartView = require('jira-agile/rapid/ui/chart/chart-view');

    // GLOBALS... FIX ME
    var Ajax = GH.Ajax;
    var ChartFilters = GH.ChartFilters;
    var ChartTimeFrames = GH.ChartTimeFrames;
    var ChartUtils = GH.ChartUtils;
    var log = GH.log;
    var Logger = GH.Logger;
    var RapidBoardState = GH.RapidBoard.State;
    var tpl = GH.tpl;

    /**
     * Chart view component.
     *
     * Loads chart data, then passes contorl to chart implementation for actual rendering
     */
    var CFDController = {};

    CFDController.id = 'cumulativeFlowDiagram';

    CFDController.displayed = false;
    CFDController.rapidViewData = {};

    CFDController.rapidViewConfig = {};

    CFDController.analytics = new AnalyticsTracker('gh.report.cumulativeFlowDiagram');

    CFDController.setRapidView = function (rapidViewData) {
        CFDController.rapidViewData = rapidViewData || {};
    };

    CFDController.isApplicable = function (rapidViewConfig) {
        return true;
    };

    CFDController.getNormalizedFromInternal = function () {
        return ChartUtils.getTimeFrameReportUrlParams(CFDController.id);
    };

    CFDController.toUrl = function (params) {
        return ChartUtils.timeFrameReportParamsToUrl(params);
    };

    /**
     * Update state from normalized state
     * @param params
     */
    CFDController.updateInternalFromNormalized = function (params) {
        ChartUtils.updateFromNormalizedStateDateTimeParams(params);
    };

    /**
     * Get normalized state for this chart type
     */
    CFDController.getNormalizedFromUrl = function (params) {
        return ChartUtils.getNormalizedFromUrlForTimeFrameReport(params);
    };

    CFDController.show = function () {
        log('CFDController.show', Logger.Contexts.ui);
        CFDController.displayed = true;

        // stop here if haven't got a view
        if (!CFDController.rapidViewData) { return; }

        // register filter listeners
        $(GH)
            .unbind('.' + CFDController.id)
            .bind('chartTimeFramesChanged.' + CFDController.id, CFDController.handleChartTimeFramesChangedEvent)
            .bind('chartFiltersChanged.' + CFDController.id, CFDController.handleChartFiltersChangedEvent);


        CFDView.show();

        // render the view
        CFDController.renderChartView();

        var action = 'show.' + (CFDController.rapidViewConfig.sprintSupportEnabled ? 'scrum' : 'kanban');
        CFDController.analytics.trigger(action);
    };

    CFDController.hide = function () {
        log('CFDController.hide', Logger.Contexts.ui);
        CFDController.displayed = false;

        CFDView.hide();
        $('#ghx-chart-controls').hide();

        // unbind listeners
        $(GH).unbind('.' + CFDController.id);
    };

    /**
     * Loads the chart data for a given chart type
     */
    CFDController.renderChartView = function () {
        // controls
        CFDController.initializeControls();

        // render the view
        ChartView.getChartContentElem(true).html(tpl.chartview.renderChartBody());

        // show the spinner
        ChartView.showSpinner();

        // then load the chart
        ChartController.loadRapidViewConfig(CFDController, CFDController.loadChartData);
    };

    /**
     * Renders the dropdown menu for the Chart Menu items & attach events
     */
    CFDController.initializeControls = function () {
        // controls structure
        ChartView.hideChartControls();
        ChartView.getChartControls(true).html(tpl.chartview.renderChartControls());

        // setup controls
        ChartFilters.initializeDialog();
        ChartTimeFrames.initializeDialog(CFDController.id);
    };

    /**
     * Loads the chart.
     * This function updates the tools, initializes the spinner, then sends off a request to load the data
     */
    CFDController.loadChartData = function () {
        // update the tools
        CFDController.updateTools();

        // make sure we display the tools now
        ChartView.showChartControls();

        // Put together the ajax request, including the configuration from the filters
        var data = {
            'rapidViewId': RapidBoardState.getRapidViewId()
        };
        ChartFilters.addRequestParams(data);

        // then issue the request
        Ajax.get({
            url: '/rapid/charts/cumulativeflowdiagram.json',
            data: data
        }, 'rapidChartData')
            .done(CFDController.processChartData)
            .always(ChartView.hideSpinner);
    };

    /**
     *
     * Refreshes the tools according to the selected chart type.
     * The tools are the ones that provided ajax request parameters, it is therefore important
     * that their state is properly set before loading a chart
     */
    CFDController.updateTools = function updateTools() {
        // update the display of the time frame
        ChartTimeFrames.update();

        var viewConfig = CFDController.rapidViewConfig;
        var chartFilterConfig = CFDController.buildChartFilterConfig(viewConfig);

        // initialize the chart filters
        ChartFilters.update(CFDController.id, viewConfig, chartFilterConfig);

        // push the state (as the time frames might have changed)
        RapidBoardState.pushState();
    };

    /**
     * Build the configuration object that defines what defaults are used in the Filter dialog.
     */
    CFDController.buildChartFilterConfig = function (viewConfig) {
        var chartFilterConfig = {};
        chartFilterConfig.defaultColumns = [];

        // determine default columns
        for (var i = 0; i < viewConfig.columns.length; i++) {
            chartFilterConfig.defaultColumns.push(true);
        }

        return chartFilterConfig;
    };

    /**
     * Processes the returned chart data.
     */
    CFDController.processChartData = function (data) {
        // ignore if we are not displayed anymore
        if (!CFDController.displayed) { return; }

        // cache the data for future reference without reloading
        CFDData.setRawData(data);

        // initialise the time frame to something sensible
        ChartTimeFrames.setBoundaries(data.firstChangeTime, data.now);

        // show the chart
        CFDView.renderChart(data);
    };

    /**
     * Called whenever the chart filters changed
     */
    CFDController.handleChartFiltersChangedEvent = function (event, data) {
        if (!CFDController.displayed) { return; }

        log('CFDController.handleChartFiltersChangedEvent', Logger.Contexts.event);

        CFDController.loadChartData();
    };

    /**
     * Called whenever the chart time frames changed
     */
    CFDController.handleChartTimeFramesChangedEvent = function (event) {
        if (!CFDController.displayed) { return; }
        log('CFDController.handleChartTimeFramesChangedEvent', Logger.Contexts.event);

        // fetch cached data from last request, since we're not reloading here, just zooming around.
        var data = CFDData.getRawData();
        CFDView.renderChart(data);
    };

    return CFDController;
});
