/* globals GH */

define("jira-agile/rapid/state/rapid-board-state", [
    "underscore",
    "jira/util/logger",
    "jira-agile/rapid/global-events"
], function(_, logger, GlobalEvents) {

    let PlanUrlState;
    GlobalEvents.on('pre-initialization', function() {
       PlanUrlState = require("jira-agile/rapid/ui/plan/plan-url-state");
    });

    /**
     * Central state manager for the Rapid Board.
     */
    const RapidBoardState = {};

    /**
     * All modes
     */
    RapidBoardState.VIEW_MODE = {
        PLAN: 'plan',
        WORK: 'work',
        REPORT: 'report'
    };

    /**
     * Holds all local data. Until a page reload, all RapidBoardState is solely read from this state - not from the local or session storage.
     */
    RapidBoardState.data = {};

    /**
     * Set a global data value.
     */
    RapidBoardState.setDataValue = function (name, value) {
        if (_.isUndefined(value) || _.isNull(value)) {
            delete RapidBoardState.data[name];
        } else {
            RapidBoardState.data[name] = value;
        }
    };

    /** Get a data value. */
    RapidBoardState.getDataValue = function (name) {
        return RapidBoardState.data[name];
    };


//
// Individual state variables
//

    /**
     * Get the current view name
     */
    RapidBoardState.getMode = function () {
        return RapidBoardState.getDataValue('mode') || RapidBoardState.VIEW_MODE.WORK;
    };
    RapidBoardState.getStoredMode = function () {
        return GH.storage.get('gh.mode') || RapidBoardState.VIEW_MODE.WORK;
    };

    RapidBoardState.isPlanMode = function () {
        return RapidBoardState.getMode() === RapidBoardState.VIEW_MODE.PLAN;
    };

    RapidBoardState.isWorkMode = function () {
        return RapidBoardState.getMode() === RapidBoardState.VIEW_MODE.WORK;
    };

    /**
     * Sets the name of the current view.
     * Current known views are 'work', 'report', 'plan'
     */
    RapidBoardState.setMode = function (mode) {
        RapidBoardState.setDataValue('mode', mode);
        GH.storage.put('gh.mode', mode);
    };

    RapidBoardState.getLastCardPrintingSize = function () {
        return GH.storage.get('gh.card.printing.size') || 'small'; // default is small
    };

    RapidBoardState.setLastCardPrintingSize = function (size) {
        GH.storage.put('gh.card.printing.size', size);
    };


    RapidBoardState.getProjectKey = function () {
        return RapidBoardState.getDataValue('projectKey') || undefined;
    };
    RapidBoardState.setProjectKey = function (projectKey) {
        RapidBoardState.setDataValue('projectKey', projectKey);
    };

    /**
     * Is the details view currently opened
     */
    RapidBoardState.isDetailsViewOpened = function () {
        return RapidBoardState.getDataValue('detailViewWork') || false;
    };

    RapidBoardState.isDetailsViewPlanOpened = function () {
        return RapidBoardState.getBoardSetting(null, 'detailViewPlan', true) || false;
    };

    RapidBoardState.getStoredDetailViewOpened = function () {
        return GH.storage.get('gh.detailViewWork') || false;
    };

    RapidBoardState.setWorkViewDetailsViewOpened = function (opened) {
        RapidBoardState.setDataValue('detailViewWork', opened || false);
        GH.storage.put('gh.detailViewWork', opened || false);

    };

    RapidBoardState.setPlanViewDetailsViewOpened = function (opened) {
        RapidBoardState.setBoardSetting('detailViewPlan', opened || false);
    };

    RapidBoardState.setEpicShownOnRapidBoard = function (shown) {
        GH.storage.put('gh.epicShownOnRapidBoard.' + RapidBoardState.getRapidViewId(), shown);
    };

    RapidBoardState.getEpicShownOnRapidBoard = function () {
        var epicShownOnRapidBoard = GH.storage.get('gh.epicShownOnRapidBoard.' + RapidBoardState.getRapidViewId());
        if (_.isNull(epicShownOnRapidBoard)) {
            return true;
        }
        return epicShownOnRapidBoard;
    };

    RapidBoardState.setNonWorkingDaysShown = function (reportId, shown) {
        GH.storage.put('gh.nonWorkingDaysShown.' + reportId, shown);
    };

    RapidBoardState.getNonWorkingDaysShown = function (reportId) {
        var nonWorkingDaysShown = GH.storage.get('gh.nonWorkingDaysShown.' + reportId);
        if (_.isNull(nonWorkingDaysShown)) {
            return true;
        }
        return nonWorkingDaysShown;
    };

    RapidBoardState.setQuickFiltersExpanded = function (shown) {
        GH.storage.put('gh.quickFiltersExpanded', shown);
    };

    RapidBoardState.getQuickFiltersExpanded = function () {
        var expanded = GH.storage.get('gh.quickFiltersExpanded');
        if (_.isNull(expanded)) {
            return false;
        }
        return expanded;
    };

    RapidBoardState.setDetailViewWidth = function (width) {
        GH.storage.put('gh.detailViewWidth', width);
    };

    RapidBoardState.getDetailViewWidth = function () {
        var width = GH.storage.get('gh.detailViewWidth');
        if (_.isNull(width)) {
            return 0;
        }
        return width;
    };

    RapidBoardState.getCycleTimeCalculationType = function () {
        var cycleTimeType = RapidBoardState.getDataValue('cycleTimeCalculationType');
        if (_.isUndefined(cycleTimeType)) {
            cycleTimeType = GH.storage.get('gh.cycleTimeCalculationType');
            if (_.isNull(cycleTimeType)) {
                RapidBoardState.setCycleTimeCalculationType(GH.ControlChart.CYCLE_TIME_CALCULATION_TOTAL);
                return GH.ControlChart.CYCLE_TIME_CALCULATION_TOTAL;
            }
        }
        return cycleTimeType;
    };

    RapidBoardState.setCycleTimeCalculationType = function (type) {
        RapidBoardState.setDataValue('cycleTimeCalculationType', type);
        GH.storage.put('gh.cycleTimeCalculationType', type);
    };

    RapidBoardState.getStoredChartIntroState = function () {
        return GH.storage.get('gh.charts.intro') !== false;
    };

    RapidBoardState.setChartIntroState = function (visible) {
        GH.storage.put('gh.charts.intro', visible);
    };

    /**
     * Sets whether the chart view is opened or not
     */
    RapidBoardState.isChartViewOpened = function () {
        return RapidBoardState.getMode() === RapidBoardState.VIEW_MODE.REPORT;
    };

    /**
     * Get the current rapid view id as defined in the URL or local storage
     */
    RapidBoardState.getRapidViewId = function () {
        let viewId = RapidBoardState.getDataValue('rapidViewId');
        if (_.isUndefined(viewId)) {
            viewId = RapidBoardState.getMostRecentRapidViewId();
        }
        return viewId;
    };

    /**
     * Get the current rapid view id as defined in the URL or local storage
     * and returns whether it came from the url or the local storage
     */
    RapidBoardState.getRapidViewIdWithInfo = function () {
        const data = {};
        data.viewId = RapidBoardState.getDataValue('rapidViewId');
        data.fromLocalStorage = false;
        if (_.isUndefined(data.viewId)) {
            data.viewId = RapidBoardState.getMostRecentRapidViewId();
            data.fromLocalStorage = true;
        }
        return data;
    };

    /**
     * Get the last stored rapid view id
     */
    RapidBoardState.getMostRecentRapidViewId = function () {
        return GH.storage.get('gh.latestRapidViewId');
    };

    /**
     * Set the current rapid view id
     */
    RapidBoardState.setRapidViewId = function (rapidViewId) {
        // set the new value
        RapidBoardState.setDataValue('rapidViewId', rapidViewId);

        // store the id in the local storage
        GH.storage.put('gh.latestRapidViewId', rapidViewId);
    };

    /**
     * Get the selected issue key
     */
    RapidBoardState.getSelectedIssueKey = function () {
        return RapidBoardState.getDataValue('selectedIssueKey') || null;
    };

    RapidBoardState.getStoredSelectedIssueKey = function () {
        return GH.storage.get('gh.selectedIssueKey') || null;
    };

    /**
     * Set the selected issue key
     */
    RapidBoardState.setSelectedIssueKey = function (issueKey) {
        RapidBoardState.setDataValue('selectedIssueKey', issueKey);
        GH.storage.put('gh.selectedIssueKey', issueKey);
    };

    /**
     * Sets the chart type RapidBoardState.
     * Chart type setting is stored per-RapidView.
     */
    RapidBoardState.setChartType = function (chartType) {
        const settings = RapidBoardState.getBoardSettings();
        if (_.isUndefined(chartType) || _.isNull(chartType)) {
            delete settings.chartType;
        } else {
            settings.chartType = chartType;
        }
        RapidBoardState.setBoardSettings(settings);
    };

    /**
     * Get the chart type RapidBoardState for the current RapidView
     */
    RapidBoardState.getChartType = function () {
        return RapidBoardState.getBoardSettings()['chartType'] || '';
    };

    RapidBoardState.getRetrospectiveSprintId = function () {
        return RapidBoardState.getBoardSettings()['retrospectiveSprintId'] || false;
    };

    RapidBoardState.setRetrospectiveSprintId = function (sprintId) {
        return RapidBoardState.setBoardSetting('retrospectiveSprintId', sprintId);
    };

    /**
     * Sets the current charting time frame.
     * Time frame setting is stored per-RapidView.
     */
    RapidBoardState.setTimeFrame = function (timeFrame) {
        var settings = RapidBoardState.getBoardSettings();
        if (_.isUndefined(timeFrame) || _.isNull(timeFrame)) {
            delete settings.timeFrame;
        } else {
            settings.timeFrame = timeFrame;
        }
        RapidBoardState.setBoardSettings(settings);
    };

    /**
     * Get the charting time frame for the current RapidView
     */
    RapidBoardState.getTimeFrame = function () {
        var val = RapidBoardState.getBoardSettings()["timeFrame"];
        if (_.isUndefined(val) || _.isNull(val)) {
            return 7;
        } else {
            return val;
        }
    };


// RapidBoard id specific settings.

    /**
     * Sets the board settings. This object contains all the board specific settings. The implementation
     * ensures that it always returns the right settings for the given rapidBoardId.
     */
    RapidBoardState.getBoardSettings = function (rapidViewId) {
        if (_.isUndefined(rapidViewId) || _.isNull(rapidViewId)) {
            rapidViewId = RapidBoardState.getRapidViewId();
        }
        var id = rapidViewId || "00000";
        var settings = RapidBoardState.getDataValue('boardSettings.' + id) || GH.storage.get('gh.boardSettings.' + id);
        if (!settings) {
            // set an empty settings object
            settings = {};
            RapidBoardState.setDataValue('boardSettings.' + id, settings);
        }

        return settings || {};
    };

    /** Set the board settings object for the current rapid board. */
    RapidBoardState.setBoardSettings = function (settings) {
        var id = RapidBoardState.getRapidViewId();
        if (!id) {
            // don't bother storing these settings
            return;
        }

        // stored it locally and in the storage
        RapidBoardState.setDataValue('boardSettings.' + id, settings);
        GH.storage.put('gh.boardSettings.' + id, settings);
    };


// Collapsed swimlanes

    RapidBoardState.isSwimlaneCollapsed = function (swimlaneId) {
        var lanes = RapidBoardState.getBoardSettings()["collapsedSwimlanes"] || [];
        return lanes.indexOf(swimlaneId) > -1;
    };

    RapidBoardState.setSwimlaneCollapsed = function (swimlaneId, collapsed) {
        var settings = RapidBoardState.getBoardSettings();
        var lanes = settings.collapsedSwimlanes || [];
        if (collapsed) {
            if (lanes.indexOf(swimlaneId) < 0) {
                lanes.push(swimlaneId);
                settings.collapsedSwimlanes = lanes;
            }
        } else {
            // remove the index, cleanup array if empty
            var index = lanes.indexOf(swimlaneId);
            if (index < 0) {
                return;
            }
            lanes.splice(index, 1);
            if (lanes.length > 0) {
                settings.collapsedSwimlanes = lanes;
            } else {
                delete settings.collapsedSwimlanes;
            }
        }
        RapidBoardState.setBoardSettings(settings);
    };

    /**
     * Get the generic board setting
     */
    RapidBoardState.getBoardSetting = function (rapidViewId, propertyName, defaultValue) {
        var value = RapidBoardState.getBoardSettings(rapidViewId)[propertyName];
        if (value !== undefined) {
            return value;
        }
        return defaultValue;
    };

    /**
     * Set a generic board setting
     */
    RapidBoardState.setBoardSetting = function (propertyName, value) {
        var settings = RapidBoardState.getBoardSettings();
        if (value !== undefined) {
            settings[propertyName] = value;
        } else {
            delete settings[propertyName];
        }
        RapidBoardState.setBoardSettings(settings);
    };

    /**
     * Get a number array board setting
     */
    RapidBoardState.getNumberArrayBoardSetting = function (rapidViewId, propertyName, defaultValue) {
        var value = RapidBoardState.getBoardSettings(rapidViewId)[propertyName] || defaultValue;
        if (!_.isArray(value)) {
            logger.log("Property is not of type array, returning an empty array: " + propertyName);
            return [];
        }
        // TODO: handle non-number case better
        _.each(value, function (entry) {
            if (!_.isNumber(entry)) {
                logger.log("Warning: Property value is not of type number: " + entry);
            }
        });
        return value;
    };

    /**
     * Set a number array setting
     */
    RapidBoardState.setNumberArrayBoardSetting = function (propertyName, value) {
        var settings = RapidBoardState.getBoardSettings();
        if (value !== undefined) {
            // ensure it is indeed a number array
            if (!_.isArray(value)) {
                logger.log("Property is not of type array, returning an empty array: " + propertyName);
                return;
            }

            // TODO: handle non-number case better
            _.each(value, function (entry) {
                if (!_.isNumber(entry)) {
                    logger.log("Warning: Property value is not of type number: " + entry);
                }
            });

            settings[propertyName] = value;
        } else {
            delete settings[propertyName];
        }
        RapidBoardState.setBoardSettings(settings);
    };


//
// New API
//

    GH.BoardState = {};

    /**
     * Set a global value
     * The value is persisted to local storage
     */
    GH.BoardState.setGlobalValue = function (key, value) {
        RapidBoardState.setDataValue(key, value);
        GH.storage.put('gh.' + key, value);
    };

    /**
     * Get a global value
     * The value is first checked locally, then in local storage and falls back to defaultValue if not found in either
     */
    GH.BoardState.getGlobalValue = function (key, defaultValue) {
        return RapidBoardState.getDataValue(key) || GH.storage.get('gh.' + key) || defaultValue;
    };

    /**
     * Set a view specific value
     * The value is persisted to local storage
     */
    GH.BoardState.setPerViewValue = function (key, value) {
        RapidBoardState.setBoardSetting(key, value);
    };

    /**
     * Get a view specific value
     * The value is first checked locally, then in local storage and falls back to defaultValue if not found in either
     */
    GH.BoardState.getPerViewValue = function (key, defaultValue) {
        return RapidBoardState.getBoardSetting(null, key, defaultValue);
    };

// state pushing / replacement

    /**
     * Pushes the current state onto the url.
     */
    RapidBoardState.pushState = function () {
        // TODO: remove jshint override
        /*jshint noarg:false */
        GH.log('State.pushState', GH.Logger.Contexts.state);
        GH.RapidBoard.UrlState.pushState();
    };

    /**
     * Replaces the current state onto the url.
     */
    RapidBoardState.replaceState = function () {
        // TODO: remove jshint override
        /*jshint noarg:false */
        GH.log('State.replaceState', GH.Logger.Contexts.state);
        GH.RapidBoard.UrlState.replaceState();
    };

    /**
     * Initializes the state of the rapid board.
     * This function is called exactly once, on page load
     */
    RapidBoardState.initializeState = function initializeState() {
        // load defaults from local/session storage
        // intentionally do NOT load the rapidViewId, as this is handled differently
        // When the selector is loaded, it differentiates between a url-specified view id and a local storage fallback
        // one shows an error message, the other one simply falls back to an existing view
        RapidBoardState.setSelectedIssueKey(RapidBoardState.getStoredSelectedIssueKey());
        RapidBoardState.setWorkViewDetailsViewOpened(RapidBoardState.getStoredDetailViewOpened());

        var mode = RapidBoardState.getStoredMode();
        if (mode === RapidBoardState.VIEW_MODE.PLAN && !RapidBoardState.isPlanModeAvailable()) {
            mode = RapidBoardState.VIEW_MODE.WORK;
        }

        RapidBoardState.setMode(mode);

        RapidBoardState.setChartIntroState(RapidBoardState.getStoredChartIntroState());

        // setup the handlers for the UrlState to work with
        GH.RapidBoard.UrlState.setHandlers([GH.RapidBoardUrlState, PlanUrlState, GH.WorkUrlState, GH.ReportUrlState]);

        // now load the state from available url parameters
        if (GH.RapidBoard.UrlState.hasUrlState()) {
            GH.RapidBoard.UrlState.loadState();
        }

        // replace the state as currently set
        var loadingDisabledKanPlan = RapidBoardState.isLoadingKanPlanWhileItDisabled();
        GH.RapidBoard.UrlState.replaceState(loadingDisabledKanPlan);
        if (loadingDisabledKanPlan) {
            var KanPlanFeatureService = require('jira-agile/rapid/ui/kanplan/kan-plan-feature-service');
            KanPlanFeatureService.showKanPlanDisabledFlag();
        } else {
            logger.trace("gh.rapid.kanplan.not.enabled.message.not.shown");
        }

        // now as a last step, initialize the UrlState, which will listen to state changes
        GH.RapidBoard.UrlState.init();
    };

    RapidBoardState.isPlanModeAvailable = function () {
        var KanPlanFeatureService = require('jira-agile/rapid/ui/kanplan/kan-plan-feature-service');
        return RapidBoardState.isScrumBoard() || KanPlanFeatureService.shouldShowKanbanBacklog();
    };

    RapidBoardState.isLoadingKanPlanWhileItDisabled = function (mode) {
        var urlParams = GH.RapidBoard.UrlState.getCurrentUrlState();
        mode = !mode ? GH.RapidBoard.UrlState.normalizeStringParam(urlParams['view'], '') : mode;

        if (!_.isString(mode)) {
            return false;
        }

        return mode.indexOf('planning') > -1 && !RapidBoardState.isPlanModeAvailable();
    };

    /**
     * Check if current board is Scrum or not
     * @return {boolean} True if current board is Scrum, false for otherwise
     * */
    RapidBoardState.isScrumBoard = function () {
        return GH.RapidViewConfig.currentData.data && GH.RapidViewConfig.currentData.data.sprintSupportEnabled;
    };

    /**
     * Check if current board is Kanban or not
     * @return {boolean} True if current board is Kanban, false for otherwise
     * */
    RapidBoardState.isKanbanBoard = function () {
        return GH.RapidViewConfig.currentData.data && !GH.RapidViewConfig.currentData.data.sprintSupportEnabled;
    };

    /**
     * @returns {string} scrum if Scrum Board and kanban if Kanban Board
     */
    RapidBoardState.getBoardType = function () {
        return RapidBoardState.isScrumBoard() ? 'scrum' : 'kanban';
    };

    /**
     * @returns {boolean} whether to display epics panel or not
     */
    RapidBoardState.getShowEpicAsPanel = function () {
        return GH.RapidViewConfig.currentData.data.showEpicAsPanel;
    };

    return RapidBoardState;

});

AJS.namespace("GH.RapidBoard.State", null, require("jira-agile/rapid/state/rapid-board-state"));