/**
 * Handles polling that is automatically stopped when the window loses focus and
 * started again when the window gains focus.
 *
 * Multiple pollers can be added, each with different poll intervals.
 */
GH.Poller = function (windowActivityNotifier) {
    var DEFAULT_INTERVAL = 30000;
    var pollers = {};
    var stopped = false;

    /**
     * Stops polling, clearing all setTimeouts
     */
    function stopAll() {
        if (stopped) {
            return;
        }

        GH.log('onBlur: Disabling polling', GH.Logger.Contexts.refresh);

        stopped = true;
        _.each(pollers, stop);
    }

    /**
     * Reinitiates polling, immediately calling all handlers.
     * Implies the window regained focus.
     */
    function startAll() {
        if (!stopped) {
            return;
        }

        GH.log('onFocus: Enabling polling', GH.Logger.Contexts.refresh);

        stopped = false;
        _.each(pollers, start);
    }

    /**
     * Stops a single poller
     * @param poller
     */
    function stop(poller) {
        if (poller.timeout != null) {
            clearTimeout(poller.timeout);
            poller.timeout = null;
        }
    }

    /**
     * Perform one 'iteration' of the polling for a single poller
     * @param {boolean} restarting - true if the poller was just restarted
     * @param poller
     */
    function doPoll(restarting, poller) {
        // If the timeout is null, and the poller isn't restarting after being stopped,
        // the poller's timeout was actually cleared, but this function still gets called in IE
        // because clearTimeout() doesn't behave as expected.
        if (!restarting && poller.timeout == null) {
            return;
        }

        GH.log('Checking updates for: ' + poller.id, GH.Logger.Contexts.refresh);

        var next = _.partial(schedule, poller);
        var promise = poller.handler.call(poller.context, restarting);
        if (promise && promise.done) {
            promise.done(next);
            return;
        }
        next();
    }

    /**
     * Schedules a single poller to start after its interval
     * @param poller
     */
    function schedule(poller) {
        poller.timeout = setTimeout(_.partial(doPoll, false, poller), poller.interval);
    }

    /**
     * Starts a single poller, immediately calling its handler.
     * @param poller
     */
    var start = _.partial(doPoll, true);

    return {
        init: function init() {
            windowActivityNotifier.on('focus', startAll).on('blur', stopAll).on('unload', stopAll);
        },
        /**
         * Adds a new poller. Handler is scheduled but NOT called immediately.
         * Adding a poller with the same id as an existing one will remove it before adding the new one.
         * @param id
         * @param [interval=30000] - in milliseconds
         * @param handler - operation to be performed.
         *    Handlers are passed a boolean indicating if it is called due to the window being refocused.
         *    Handlers can return a promise, in which case the poller will pause to wait for the promise to resolve.
         * @param [context] - context the handler should be called with
         */
        addPoller: function addPoller(id, interval, handler, context) {
            // Handle function overloading
            if (typeof interval === 'function') {
                context = handler;
                handler = interval;
                interval = DEFAULT_INTERVAL;
            }

            this.removePoller(id);

            GH.log('Adding poller: ' + id, GH.Logger.Contexts.refresh);

            var poller = {
                id: id,
                interval: interval,
                handler: handler,
                context: context
            };
            pollers[id] = poller;
            if (!stopped) {
                schedule(poller);
            }
        },
        /**
         * Removes a poller and stops it immediately.
         * @param id
         */
        removePoller: function removePoller(id) {
            var poller = pollers[id];
            if (poller) {
                GH.log('Removing poller: ' + id, GH.Logger.Contexts.refresh);

                stop(poller);
                delete pollers[id];
            }
        }
    };
}(GH.windowActivityNotifier);