define('jira-agile/rapid/configuration/card-color-config', [
    "jira/util/formatter",
    "underscore",
    "jquery",
    'jira-agile/rapid/analytics-tracker'
], function(
    Formatter
    ,
    _,
    $,
    AnalyticsTracker
) {

    /**
     * Card Colors configuration tab
     */
    const CardColorConfig = {};

    /**
     * Holds the model for the configuration.
     */
    CardColorConfig.model = undefined;

    CardColorConfig.Analytics = {};

    CardColorConfig.Analytics.base = new AnalyticsTracker("gh.config.cardcolor");
    CardColorConfig.Analytics.add = new AnalyticsTracker("gh.config.cardcolor.custom", "add");
    CardColorConfig.Analytics.edit = new AnalyticsTracker("gh.config.cardcolor.custom", "edit");
    CardColorConfig.Analytics.remove = new AnalyticsTracker("gh.config.cardcolor.custom", "delete");

    /**
     * All available card color strategies
     */
    CardColorConfig.strategies = [
        {
            id: 'issuetype',
            name: Formatter.I18n.getText('gh.rapid.config.cardcolors.strategy.issuetype'),
            hasDescriptionField: false,
            canAddRows: false,
            canReorderRows: false,
            columnOverrides: {
                value: {
                    header: Formatter.I18n.getText('gh.rapid.config.cardcolors.label.issuetype')
                }
            }
        },
        {
            id: 'priority',
            name: Formatter.I18n.getText('gh.rapid.config.cardcolors.strategy.priority'),
            hasDescriptionField: false,
            canAddRows: false,
            canReorderRows: false,
            columnOverrides: {
                value: {
                    header: Formatter.I18n.getText('gh.rapid.config.cardcolors.label.priority')
                }
            }
        },
        {
            id: 'assignee',
            name: Formatter.I18n.getText('gh.rapid.config.cardcolors.strategy.assignee'),
            hasDescriptionField: false,
            canAddRows: false,
            canReorderRows: false,
            columnOverrides: {
                value: {
                    header: Formatter.I18n.getText('gh.rapid.config.cardcolors.label.assignee')
                }
            }
        },
        {
            id: 'custom',
            name: Formatter.I18n.getText('gh.rapid.config.cardcolors.strategy.custom'),
            hasDescriptionField: true,
            canAddRows: true,
            canReorderRows: true,
            columnOverrides: {
                value: {
                    allowEdit: true,
                    allowDelete: true,
                    header: Formatter.I18n.getText('gh.rapid.config.cardcolors.label.jql'),
                    emptyText: Formatter.I18n.getText('gh.rapid.config.cardcolors.empty.jql'),
                    editView: GH.RapidBoard.Config.Util.JQLEditView.extend({ namespace: 'ghx-card-color'})
                }
            }
        },
        {
            id: 'none',
            name: Formatter.I18n.getText('gh.boards.none'),
            hasDescriptionField: false,
            canAddRows: false,
            canReorderRows: false
        }
    ];

    CardColorConfig.TableColumnsDefaults = [
        {
            id: 'color',
            header: Formatter.I18n.getText('gh.rapid.config.cardcolors.label.color'),
            fieldName: 'color',
            styleClass: 'ghx-picker',
            allowEdit: true,
            allowDelete: true,
            readView: AJS.RestfulTable.CustomReadView.extend({
                render: function (renderData) {
                    return $(GH.tpl.cardcolorconfig.renderReadOnlyViewColor(renderData));
                }
            }),
            editView: AJS.RestfulTable.CustomEditView.extend({
                render: function (renderData) {
                    var row = this;

                    if (!renderData.value) {
                        renderData.value = CardColorConfig.nextDefaultColorHex();
                    }
                    var $contents = $(GH.tpl.cardcolorconfig.renderEditableViewColor(renderData));
                    var $picker = this.$picker = $contents.filter('.ghx-picker-select');
                    $picker.ColorPicker({
                        onChange: function (hsb, hex, rgb) {
                            var hexColor = '#' + hex;
                            $picker.css('background-color', hexColor);
                            $picker.attr('data-field-value', hexColor);
                            $picker.parent().find('input').val(hexColor);
                            row.model.set({'color': hexColor});
                        },
                        onHide: function (colpkr) {
                            CardColorConfig.hidePicker(colpkr, row);
                        },
                        onBeforeShow: function () {
                            var color = $picker.parent().find('input').val().substr(1);
                            $(this).ColorPickerSetColor(color);
                        }
                    }).bind('click', _.bind(this.focus, this));
                    $contents.filter('input').bind('select', _.bind(this.focus, this));

                    return $contents;
                },
                focus: function () {
                    this.$picker.ColorPickerShow();
                }
            })
        },
        {
            id: 'value',
            header: null,
            fieldName: 'value',
            allowEdit: false,
            allowDelete: false,
            readView: AJS.RestfulTable.CustomReadView.extend({
                render: function (renderData) {
                    var value = this.model.get('displayValue');
                    return value ? document.createTextNode(value) : null;
                }
            }),
            editView: AJS.RestfulTable.CustomEditView.extend({
                render: function (renderData) {
                    var value = this.model.get('displayValue');
                    return value ? document.createTextNode(value) : null;
                }
            })
        }
    ];

    CardColorConfig.hidePicker = function (colpkr, row) {
        $(colpkr).fadeOut(500);
        var modelId = row.model.get('id');
        if (modelId) {
            // only save if we have a model id - if no model ID this is a new JQL row - color saved on add
            var rapidViewId = CardColorConfig.model.rapidViewId;
            GH.Ajax.put({
                url: '/cardcolors/' + rapidViewId + '/color/' + row.model.get('id'),
                data: row.model.toJSON()
            });
        }

        var $cancel = $('.aui-restfultable-editrow .aui-restfultable-cancel');
        $cancel.click();

        // Show the operations container again
        $cancel.parent().show();

        return false;
    };

    /**
     * Initializes the Card Color config.
     */
    CardColorConfig.init = function (model) {
        CardColorConfig.model = model.cardColorConfig;

        CardColorConfig.renderConfig();
    };

    CardColorConfig.getStrategyById = function (id) {
        return _.find(CardColorConfig.strategies, function (strategy) {
            return (strategy.id === id);
        });
    };

    CardColorConfig.getCurrentStrategy = function () {
        var model = CardColorConfig.model;
        if (model && model.cardColorStrategy) {
            return CardColorConfig.getStrategyById(model.cardColorStrategy);
        }
        return null;
    };

    CardColorConfig.renderConfig = function () {
        var model = CardColorConfig.model;
        var canEdit = model.canEdit;
        var currentStrategyId = model.cardColorStrategy;

        // render the strategies selector
        $('#ghx-config-cardColors').html(GH.tpl.cardcolorconfig.renderCardColorConfigView({
            canEdit: canEdit,
            currentStrategyId: currentStrategyId,
            strategies: CardColorConfig.strategies,
            learnMore: GH.HelpPaths.getHelpPath('board.configure.card.strategy').url
        }));

        // fetch the chosen strategy
        var strategy = CardColorConfig.getStrategyById(currentStrategyId);
        if (!strategy) {
            return;
        }

        // load the strategy configuration
        CardColorConfig.loadStrategyConfig(strategy);

        // attach listener to dropdown
        $('#ghx-cardcolorstrategy-select').change(CardColorConfig.saveStrategySelection);
    };

    /**
     * Updates the strategy
     */
    CardColorConfig.saveStrategySelection = function (event) {
        // fetch new strategy
        var newStrategyId = $('#ghx-cardcolorstrategy-select').val();

        // get the strategy
        var strategy = CardColorConfig.getStrategyById(newStrategyId);
        if (!strategy) {
            return;
        }

        // update the model
        CardColorConfig.model.cardColorStrategy = strategy.id;

        // save the config change
        var rapidViewId = CardColorConfig.model.rapidViewId;
        GH.Ajax.put({
            url: '/cardcolors/' + rapidViewId + '/strategy',
            data: {
                id: rapidViewId,
                cardColorStrategyId: strategy.id
            }
        })
            .done(function () {
                // update the ui
                CardColorConfig.loadStrategyConfig(strategy);
                CardColorConfig.Analytics.base.trigger("editStrategy", strategy.id);
            });
    };

    /**
     * Load the configuration for the given strategy
     */
    CardColorConfig.loadStrategyConfig = function (strategy) {
        // show that we are loading
        $("#ghx-cardcolorstrategy-select").after(GH.tpl.cardcolorconfig.renderCardColorLoading());

        var rapidViewId = CardColorConfig.model.rapidViewId;
        // fetch the configuration from the server
        GH.Ajax.get({
            url: '/cardcolors/' + rapidViewId + '/strategy/' + strategy.id,
            data: { preloadValues: true }
        }, "cardColorConfig")
            .done(CardColorConfig.processConfigData);
    };

    CardColorConfig.processConfigData = function (result) {
        // set the model
        CardColorConfig.model = result;

        // then render the table
        CardColorConfig.renderConfigurationTable();
    };
    /**
     * Renders the edit form using the locally stored model
     */
    CardColorConfig.renderConfigurationTable = function () {
        var model = CardColorConfig.model;
        var canEdit = model.canEdit;
        var rapidViewId = model.rapidViewId;

        var strategy = CardColorConfig.getCurrentStrategy();

        // get rid of old table and apply correct data attribute
        $("#ghx-card-color-table-form").html('<table id="ghx-card-color-table"></table>')
            .attr("data-strategy", strategy.id);

        // remove the spinner
        $("#ghx-cardcolorstrategy-select ~ .ghx-spinner").remove();

        // no rendering configuration table when choosing None strategy
        if (strategy.id === 'none') {
            return;
        }

        CardColorConfig.nextDefaultColorHex.counter = model.cardColors.length;

        // setup the restful table
        var restfulTable = new AJS.RestfulTable({
            el: $("#ghx-card-color-table"), // table to add entries to. Entries are appended to the tables <tbody> element
            resources: {
                all: function (next) {
                    next(CardColorConfig.model.cardColors);
                },
                self: GH.Ajax.buildRestUrl('/cardcolors/' + rapidViewId + '/color/')
            },
            // Allow each strategy to override the defaults
            columns: _.map(CardColorConfig.TableColumnsDefaults, function (defaults) {
                return _.extend({}, defaults, strategy.columnOverrides[defaults.id]);
            }),
            deleteConfirmation: function() {
                return GH.tpl.rapid.view.deleteForm({
                    message: Formatter.I18n.getText('gh.rapid.config.cardcolors.delete.confirm')
                });
            },
            views: {
                editRow: AJS.RestfulTable.EditRow.extend({
                    mapSubmitParams: function (params) {
                        var changed = AJS.RestfulTable.EditRow.prototype.mapSubmitParams.call(this, params);
                        // ensure we always send the selected strategy id
                        if (changed) {
                            changed.strategy = strategy.id;
                        }
                        return changed;
                    },
                    submit: function () {
                        var values = this.$el.serializeObject();
                        var oldJql = this.model.attributes.value;
                        var newJql = values && values.value ? values.value : "";
                        if (oldJql) {
                            //If we have an old query, then we have an edit event
                            CardColorConfig.Analytics.edit.trigger("new: " + newJql + ", old: " + oldJql);
                        } else {
                            // else we have a create event
                            CardColorConfig.Analytics.add.trigger(newJql);
                        }

                        return AJS.RestfulTable.EditRow.prototype.submit.apply(this, arguments);
                    }
                })
            },
            allowEdit: canEdit,
            allowDelete: canEdit,
            allowCreate: canEdit && strategy.canAddRows,
            allowReorder: canEdit && strategy.canReorderRows
        });

        if (strategy.id === 'custom') {
            // register jql auto complete
            // Allow row to get height. Required for jql autocomplete calculations
            _.delay(GH.JQLAutoComplete.initialize, 0, 'ghx-card-color-jql-create', 'ghx-card-color-jql-errormsg-create');

            // initializing the autocomplete when the field is hidden results in incorrect dropdown positioning
            $(document).bind("tabSelect", function () {
                GH.JQLAutoComplete.fixSuggestionPositioning('ghx-card-color-jql-create');
            });

            // Analytics event listeners
            $(restfulTable).bind(AJS.RestfulTable.Events.ROW_REMOVED, function (evt, data) {
                CardColorConfig.Analytics.remove.trigger(data.model.attributes.value);
            });

        }
    };

    CardColorConfig.getDefaultColorHex = function (currentCount) {
        // check the model
        var defaultColors = CardColorConfig.model.defaultColors;
        return defaultColors[currentCount % defaultColors.length];
    };

    CardColorConfig.nextDefaultColorHex = function () {
        return CardColorConfig.getDefaultColorHex(CardColorConfig.nextDefaultColorHex.counter++);
    };
    CardColorConfig.nextDefaultColorHex.counter = null;

    return CardColorConfig;
});

AJS.namespace('GH.CardColorConfig', null, require('jira-agile/rapid/configuration/card-color-config'));