/**
 * Client side validation and error display functions
 */
define('jira-agile/rapid/validation', ["jira/util/formatter", "jquery"], function (Formatter, $) {


    const Validation = {};

    /**
     * Validate that an input doesn't have an empty value. If the validation is failing, an error message
     * is displayed next to the field.
     *
     * @param field : input of type text or textarea
     * @param messageKey : the i18nKey of the message to be displayed
     * @param errorContextElement : (optional) an alternative element where the error should be shown (can be an element or a CSS selector)
     *
     * @returns true if the validation passed, false if an error was shown
     */
    Validation.notBlank = function(field, messageKey, errorContextElement) {
        var passed = Validation.applyRules(field, messageKey, errorContextElement, function() {

            return $.trim($(field).val()).length > 0;
        });

        return passed;
    };

    /**
     * Validate that an input doesn't exceed a certain length. If the validation is failing, an error message
     * is displayed next to the field.
     *
     * @param field : input of type text or textarea
     * @param messageKey : the i18nKey of the message to be displayed
     * @param errorContextElement : (optional) an alternative element where the error should be shown (can be an element or a CSS selector)
     *
     * @returns true if the validation passed, false if an error was shown
     */
    Validation.maxLength = function(field, maxLength, messageKey, errorContextElement) {
        var passed = Validation.applyRules(field, messageKey, errorContextElement, function() {

            return $.trim($(field).val()).length <= maxLength;
        });

        return passed;
    };

    /**
     * Validate that an input is a positive integer (empty or >= 0). If the validation is failing, an error message
     * is displayed next to the field.
     *
     * @param field : input of type text or textarea
     * @param messageKey : the i18nKey of the message to be displayed
     * @param errorContextElement : (optional) an alternative element where the error should be shown (can be an element or a CSS selector)
     *
     * @returns true if the validation passed, false if an error was shown
     */
    Validation.isPositiveInteger = function(field, messageKey, errorContextElement) {
        var passed = Validation.applyRules(field, messageKey, errorContextElement, function() {
            return (/^\d*$/).test($.trim($(field).val()));
        });
        return passed;
    };

    /**
     * If the given rule evaluates to false, the error message will be shown and validation fails.
     *
     * @param rule : a function that returns a boolean result
     */
    Validation.applyRules = function(field, messageKey, errorContextElement, rule) {
        field = $(field);
        var errorCtx = errorContextElement ? $(errorContextElement) : field;

        var passed = rule.call(this);
        if (!passed) {
            Validation.showContextualErrors(errorCtx, messageKey);
            return false;
        }

        return true;
    };

    Validation.showContextualErrors = function(errorCtx, messageKey, param1, param2, param3) {
        // remove previous errors
        Validation.clearContextualErrors(errorCtx);

        // show error message
        errorCtx.after(GH.tpl.rapid.notification.renderContextualErrors({
            errors : [ { message : Formatter.I18n.getText(messageKey, param1, param2, param3) } ]
        }));
    };

    /**
     * Removes all field error messages under or next to the given element
     *
     * @param element : an element or a CSS selector
     */
    Validation.clearContextualErrors = function(element) {
        element = $(element);

        // remove all children that are error messages
        element.find(".ghx-error").remove();

        // remove all siblings, in case someone passes in the field directly
        element.siblings(".ghx-error").remove();
    };

    return Validation;
});

AJS.namespace('GH.Validation', null, require('jira-agile/rapid/validation'));
