define('jira-agile/rapid/util', ["underscore", "jquery"], function (_, $) {

    var Util = {};

    /**
     * Returns the jira agile plugin key
     * @returns {string}
     */
    Util.getPluginKey = function () {
        return $('meta[name="ghx-plugin-key"]').attr('content');
    };

    /**
     * Wraps a function with a function which does prevet default and executes the wrapped with a concrete context/this
     * @param delegate - function to wrap
     * @param [context] - the context in which the wrapped function will be executed
     * @returns {*} - the result from the invocation of the delegate
     */
    Util.wrapPreventDefault = function (delegate, context) {
        return function (event) {
            event.preventDefault();
            return delegate.apply(context || this, arguments);
        };
    };

    /**
     * This is used in conjunction with andThen with the idea to have more then one result send as result of the promise.
     *
     * @returns {{_needsSplice: (Array|string|Blob)}}
     */
    Util.andThenSplice = function () {
        return {
            _needsSplice: Array.prototype.slice.call(arguments)
        };
    };

    /**
     * A clone of jquery then, but with the semantic of jquery 1.8+. This is because in the previous versions
     * then had a different semantics, we wanted to use the one of pipe, but it was deperecated.
     *
     * @param wrapDeferred - the deferred we want to wrap
     * @param fnDone - optional done function
     * @param fnFail - optional fail function
     * @param fnProgress - optional progress function
     * @returns {Promise} - the wrapped promise
     */
    Util.andThen = function (wrapDeferred /*, fnDone, fnFail, fnProgress */) {
        var deferredEvents = [{ trigger: 'resolveWith', on: 'done', filter: arguments[1] }, { trigger: 'rejectWith', on: 'fail', filter: arguments[2] }, { trigger: 'notifyWith', on: 'progress', filter: arguments[3] }];

        return $.Deferred(function (defer) {
            _.each(deferredEvents, function (ev) {
                var on = ev.on,
                    trigger = ev.trigger;
                var filter = _.isFunction(ev.filter) && ev.filter;

                wrapDeferred[on](function () {
                    var result = filter && filter.apply(this, arguments);

                    if (result && _.isFunction(result.promise)) {
                        result.promise().done(defer.resolve).fail(defer.reject).progress(defer.notify);
                    } else {
                        var actionResult;
                        if (result && _.isArray(result._needsSplice)) {
                            actionResult = result._needsSplice;
                        } else if (filter) {
                            actionResult = [result];
                        } else {
                            actionResult = arguments;
                        }
                        defer[trigger](_.isFunction(this.promise) ? defer.promise() : this, actionResult);
                    }
                });
            });
        }).promise();
    };

    /**
     * We wrap jQuery deferred to generate deferred and promise objects which have andThen as native operation.
     *
     * @returns {Deferred}
     */
    $._OriginalDeferred = $.Deferred;
    $.Deferred = function (func) {
        return $._OriginalDeferred(function (defer) {
            defer.andThen = function () {
                var args = Array.prototype.slice.call(arguments);
                args.unshift(defer);
                return Util.andThen.apply(this, args);
            };

            var originalPromise = defer.promise;
            defer.promise = function () {
                var newPromise = originalPromise.apply(this, arguments);
                newPromise.andThen = function () {
                    var args = Array.prototype.slice.call(arguments);
                    args.unshift(newPromise);
                    return Util.andThen.apply(this, args);
                };
                return newPromise;
            };

            if (func) {
                func.call(defer, defer);
            }
        });
    };

    /**
     * An indexOf with an iterator for testing equality.
     * Returns the index of the first item for which the iterator returns true.
     * The iterator is passed the current item and index.
     *
     * @param {Array} array
     * @param {Function} iterator
     * @param {boolean} [reverse] - iterate through the array in reverse order
     * @returns {number}
     */
    Util.indexOf = function (array, iterator, reverse) {
        var index = -1;
        if (reverse) {
            array = array.slice().reverse();
        }
        _.some(array, function (item, i) {
            if (iterator(item, i)) {
                index = reverse ? array.length - 1 - i : i;
                return true;
            }
            return false;
        });
        return index;
    };

    /**
     * Swap the two items in the array
     * @param {Array} array to manipulate
     * @param {number} sourceIndex
     * @param {number} targetIndex
     * @return {boolean} did a swap occur
     */
    Util.swapArrayItems = function (array, sourceIndex, targetIndex) {
        var temp;
        if (array && sourceIndex !== targetIndex && Math.min(sourceIndex, targetIndex) >= 0 && Math.max(sourceIndex, targetIndex) < array.length) {
            temp = array[targetIndex];
            array[targetIndex] = array[sourceIndex];
            array[sourceIndex] = temp;
            return true;
        }
        return false;
    };

    /**
     * Rounds a number to the specified number of decimal places.
     *
     * @param {number} number
     * @param {number} decimalPlaces
     * @returns {number}
     */
    Util.roundToDecimalPlaces = function (number, decimalPlaces) {
        // Skip rounding if:
        // - number is not a number
        // - number is an integer (note: using Math.round to check if we don't have a decimal number)
        if (typeof number !== 'number' || number === Math.round(number)) {
            return number;
        }
        // Use Math.round() if rounding to 0 decimal places
        if (decimalPlaces === 0) {
            return Math.round(number);
        }
        // Mathematical ways of doing the rounding are prone to floating point errors.
        // Instead, we convert to string and work with that.
        var str = number.toString();
        if (str.indexOf('e') >= 0) {
            // Give up
            return number;
        }
        var isNegative = number < 0;
        var parts = str.split('.');
        var nextDigit = +parts[1].charAt(decimalPlaces);
        parts[1] = parts[1].substr(0, decimalPlaces);
        // Check if rounding is necessary.
        // Note that for negative numbers, a digit 5 should still round up, but 'up' is towards 0,
        // hence nothing needs to be done.
        if (nextDigit > 5 || !isNegative && nextDigit === 5) {
            // Round up
            var leadingZeroes = /^0*/.exec(parts[1])[0];
            parts[1] = leadingZeroes + (+parts[1] + 1);
            if (parts[1].length > decimalPlaces) {
                if (leadingZeroes) {
                    // This can happen when .09 rounds to .10, but the leading 0 makes it .010
                    parts[1] = parts[1].substr(parts[1].length - decimalPlaces);
                } else {
                    // This can happen when .9 rounds to 1.0
                    parts[0] = +parts[0] + 1;
                    parts[1] = 0;
                }
            }
        }
        return parseFloat(parts.join('.'));
    };

    return Util;
});

AJS.namespace('GH.Util', null, require('jira-agile/rapid/util'));