/**
 * PromiseKeeper is a gatekeeper for promises that allows only one promise to be pending at any point in time.
 * As a promise is added to it, it will cancel the previous promise if it is still pending.
 * (Note: this file originally came from JIRA's issue-nav plugin, and was named RecurringPromise).
 *
 * Create a PromiseKeeper:
 *     var promiseKeeper = PromiseKeeper();
 *
 * Add a promise to it:
 *     promiseKeeper.add(promise1);
 *
 * Bind to the wrapper promise that is returned:
 *     promiseKeeper.add(promise1).done(function(data) {
 *         // Do something when promise1 is done,
 *         // BUT not if another promise is added while promise1 is pending
 *     });
 *
 * Or bind to a PromiseKeeper itself:
 *     promiseKeeper.done(function(data) {
 *         // Do something when any promise is done
 *     });
 *
 * Detect (or ignore) cancellations by binding to fail:
 *     promiseKeeper.fail(function(error) {
 *         if (error === 'abort') {
 *             // The promise was cancelled
 *             return;
 *         }
 *         // Other error handling
 *     });
 *
 * Cancel any pending promises by calling reset (useful for cleaning up):
 *     promiseKeeper.reset();
 */
define("jira-agile/rapid/promise-keeper", ['jquery'], function ($) {
   "use strict";
        /**
         * @param {PromiseKeeper} [parent]
         */
        const PromiseKeeper = function (parent) {
            if (!(this instanceof PromiseKeeper)) {
                return new PromiseKeeper();
            }

            this._parent = parent;
            this._doneCallbacks = $.Callbacks();
            this._failCallbacks = $.Callbacks();
            this._alwaysCallbacks = $.Callbacks();
        };

        PromiseKeeper.prototype = {
            add: function (deferred) {
                const wrapperDeferred = $.Deferred();
                wrapperDeferred.original = deferred;

                const pending = this._getPending();
                this._abortIfPending(pending);
                this._setPending(wrapperDeferred);

                var instance = this;
                deferred.done(function () {
                    if (wrapperDeferred === instance._getPending()) {
                        wrapperDeferred.resolveWith(instance, arguments);
                        instance._doneCallbacks.fireWith(instance, arguments);
                    }
                });
                deferred.fail(function () {
                    if (wrapperDeferred === instance._getPending()) {
                        wrapperDeferred.rejectWith(instance, arguments);
                        instance._failCallbacks.fireWith(instance, arguments);
                    }
                });
                deferred.always(function () {
                    if (wrapperDeferred === instance._getPending()) {
                        instance._alwaysCallbacks.fireWith(instance, arguments);
                    }
                });

                return wrapperDeferred.promise();
            },

            done: function (callback) {
                this._doneCallbacks.add(callback);
                return this;
            },

            fail: function (callback) {
                this._failCallbacks.add(callback);
                return this;
            },

            always: function (callback) {
                this._alwaysCallbacks.add(callback);
                return this;
            },

            reset: function () {
                this._abortIfPending(this._getPending());
                this._setPending(null);
            },

            sub: function () {
                return new PromiseKeeper(this);
            },

            _getPending: function () {
                return this._parent ? this._parent._getPending() : this._pending;
            },

            _setPending: function (deferred) {
                if (this._parent) {
                    this._parent._setPending(deferred);
                } else {
                    this._pending = deferred;
                }
            },

            _abortIfPending: function (wrapperDeferred) {
                if (wrapperDeferred && wrapperDeferred.state() === 'pending') {
                    wrapperDeferred.reject("abort");

                    // Abort existing pending request if an abort function is available
                    if ($.isFunction(wrapperDeferred.original.abort)) {
                        wrapperDeferred.original.abort();
                    }
                }
            }
        };

        return PromiseKeeper;
});

AJS.namespace('GH.PromiseKeeper', null, require("jira-agile/rapid/promise-keeper"));