AJS.namespace('GH.Layout');

define(
    'jira-agile/rapid/exception',
    ["jira/util/logger", "jira/util/data/meta", "jquery"],
    function(Logger, Meta, $) {


        /**
         * The idea of this library is to catch Javascript exceptions that came up when rendering the page,
         * either parse errors or runtime exceptions. The user should never be left with a blank screen, but
         * rather have information for support.
         */
        var Exception = {};

        /**
         * Only look for errors in files that match these patterns. All other errors will be logged to console but not shown to
         * the user.
         *
         * JS errors will likely happen on newer GH pages in one of these ways:
         *  - included JS from a web resource: either from Atlassian (com.atlassian) or GreenHopper (com.pyxis) or JIRA defined
         *    web resources (jira.webresources)
         *  - batched JS (batch)
         *  - keyboard shortcuts (shortcuts.js)
         *  - inline JS (RapidBoard.jspa, RapidView.jspa, ManageRapidViews.jspa)
         *
         *  Everything else will not be reported to the user, but still logged in the console. This is so problematic browser
         *  extensions don't give people a false impression of GreenHopper.
         */
        Exception.errorPattern = /(com\.atlassian|com\.pyxis|jira\.webresources|batch|shortcuts\.js|RapidBoard\.jspa|RapidView\.jspa|ManageRapidViews\.jspa)/;

        /**
         * Store whether or not the error pane is currently expanded
         */
        Exception.isMessageExpanded = false;

        /**
         * How many error messages are currently in the error pane
         */
        Exception.errorCount = 0;

        /**
         * Stores the DOM elements of the exception view
         */
        Exception.exceptionView = false;

        /**
         * This gets called through window.onError. That's a custom browser error handler, unfortunately
         * it lives outside the exception stack of the JS engines, so we don't have a stacktrace available here.
         *
         * Turn the information we have into an AUI error message.
         */
        Exception.handleJsException = function (msg, file, line, col, error) {
            var console = window.console || { log: function() {} };

            var loc = line;
            if (arguments.length > 3) {
                loc += ':' + col;
            }

            Logger.log('An exception has occurred in ' + file + ' at ' + loc + ' --- ' + msg);
            Exception.errorCount++;

            var stack = null;
            // See https://mikewest.org/2013/08/debugging-runtime-errors-with-window-onerror
            if (arguments.length > 4) {
                if (error && error.stack) {
                    stack = error.stack.split('\n');
                }
            }

            // exclude certain resources from error reporting, as they might break for reasons we don't want to show
            // to the user (e.g. browser extensions)
            if (!Exception.errorPattern.test(file)) {
                Logger.log('Ignoring file : ' + file);
                return;
            }

            var exceptionData = {
                title:'An error occurred',
                details:{
                    'Exception':msg,
                    'Resource':file,
                    'Line':line
                },
                stackTrace:stack
            };
            if (arguments.length > 3) {
                exceptionData.details.Column = col;
            }

            // see if we have ghx-errors around, our preferred error anchor.
            // if that's not the case (because the exception happened before the DOM
            // was rendered), we fall back to body.
            var ghxErrors = document.getElementById('ghx-errors');

            if (!Exception.exceptionView) {
                // construct the exception view for the first time
                Exception.exceptionView = Exception.buildExceptionView(exceptionData);
            } else {
                // just add this new error
                var $exceptionMessage = Exception.buildExceptionMessage(exceptionData.details);
                Exception.exceptionView.msgView.setAttribute('style', 'display:block;');

                Exception.exceptionView.detailList.appendChild($exceptionMessage);

                // update message count
                Exception.exceptionView.titleView.getElementsByTagName('span')[1].innerHTML = '(' + Exception.errorCount + ')';
            }

            // Add the exception content if it's not around
            // Need to do this here due to mode switching creating a new ghx-errors container div
            if (!ghxErrors || ghxErrors.innerHTML === '') {
                var msgView = Exception.exceptionView.msgView;

                // add view
                var exContext = document.createElement('div');
                exContext.setAttribute('class', 'ghx-exception');
                exContext.appendChild(msgView);

                // see if we have ghx-errors around, our preferred error anchor.
                // if that's not the case (because the exception happened before the DOM
                // was rendered), we fall back to body.
                var exTarget = ghxErrors;
                if (!exTarget) {
                    exTarget = document.createElement('div');
                    exTarget.setAttribute('class', 'ghx-exception-wrapper');
                    document.body.appendChild(exTarget);
                }
                exTarget.appendChild(exContext);
            }

            // if we have ghx-errors around, scroll to it
            if (ghxErrors) {
                ghxErrors.scrollTop = 100000;
            }

            // fire a resize event, because the error element might have changed the overall page height
            $(GH).trigger(GH.Layout.EVENT_DELAYED_RESIZE);

            // stops the browser from further handling the error (return true). Let exception pass in dev mode to ge
            // proper stack trace in console
            return !Meta.get('dev-mode');
        };

        /**
         * Construct the exception view. Since we're not sure at which point in the script execution the exception
         * happened, we must not rely on ANY library here. No jQuery please!
         */
        Exception.buildExceptionView = function (exceptionData) {

            // main container
            var $msgView = document.createElement('div');
            $msgView.setAttribute('class', 'aui-message aui-message-error error closeable');

            // title
            var $titleView = document.createElement('p');
            $titleView.setAttribute('class', 'title');
            $titleView.innerHTML = '<span class="aui-icon icon-error"></span>' + exceptionData.title;

            // read more link
            var $readMoreLink = document.createElement('a');
            $readMoreLink.setAttribute('class', 'ghx-more');
            $readMoreLink.innerHTML = 'Details&hellip;';

            $titleView.appendChild($readMoreLink);

            $msgView.appendChild($titleView);

            // container of the rest
            var $expandedContainer = document.createElement('div');
            $expandedContainer.setAttribute('style', 'display: none;');

            // onclick handler for the link to open/close the div
            $readMoreLink.onclick = function (e) {
                // figure out state
                if (Exception.isMessageExpanded) {
                    // close
                    $expandedContainer.setAttribute('style', 'display: none;');
                    $readMoreLink.innerHTML = 'Details&hellip;';
                    Exception.isMessageExpanded = false;
                } else {
                    // open
                    $expandedContainer.setAttribute('style', 'position: relative;');
                    $readMoreLink.innerHTML = 'Hide&hellip;';
                    Exception.isMessageExpanded = true;
                }
            };

            // help text
            var $helpText = document.createElement('p');
            $helpText.innerHTML = 'Please try refreshing the page, or contact your administrator / <a href="http://support.atlassian.com">Atlassian Support</a> if the problem continues.';
            $expandedContainer.appendChild($helpText);

            // details
            var $details = document.createElement('div');
            var det = exceptionData.details;
            var $detailViewTitle = document.createElement('h4');
            $detailViewTitle.innerHTML = 'Details';
            $details.appendChild($detailViewTitle);

            var $detailList = document.createElement('ol');
            $details.appendChild($detailList);

            // construct the message and append to list
            var $exceptionMessage = Exception.buildExceptionMessage(det);
            $detailList.appendChild($exceptionMessage);
            $expandedContainer.appendChild($details);

            // environment
            var $envViewTitle = document.createElement('h4');
            $envViewTitle.innerHTML = 'Environment';
            $expandedContainer.appendChild($envViewTitle);

            var $envView = document.createElement('div');
            $envView.innerHTML = navigator.userAgent;
            $expandedContainer.appendChild($envView);

            // stacktrace (currently not working, see above)
            var stack = exceptionData.stackTrace;
            if (stack) {
                var $stackViewTitle = document.createElement('h3');
                $stackViewTitle.innerHTML = 'Stack trace';
                $expandedContainer.appendChild($stackViewTitle);

                var $stackView = document.createElement('ul');
                for (var i = 0; i < stack.length; i++) {
                    var $stackItem = document.createElement('li');
                    $stackItem.innerHTML = stack[i];
                    $stackView.appendChild($stackItem);
                }
                $expandedContainer.appendChild($stackView);
            }

            var $closeLink = document.createElement('a');
            $closeLink.innerHTML = '<span class="aui-icon icon-close" title="Close"></span>';
            $titleView.appendChild($closeLink);

            $closeLink.onclick = function (e) {
                $msgView.setAttribute('style', 'display:none;');
                $detailList.innerHTML = '';
                $(GH).trigger(GH.Layout.EVENT_DELAYED_RESIZE);
            };

            $msgView.appendChild($expandedContainer);
            return {
                msgView:$msgView,
                detailList:$detailList,
                titleView:$titleView
            };
        };

        /**
         * Construct a single error message's representation. This will later be appended to the list of details.
         */
        Exception.buildExceptionMessage = function (exceptionDetails) {
            var $id = document.createElement('li');

            var $detailView = document.createElement('ul');
            $id.appendChild($detailView);

            for (var key in exceptionDetails) {
                if (exceptionDetails.hasOwnProperty(key)) {
                    var $detailItem = document.createElement('li');
                    $detailItem.innerHTML = key + ': ' + exceptionDetails[key];
                    $detailView.appendChild($detailItem);
                }
            }

            return $id;
        };

        window.onerror = Exception.handleJsException;

    }
);

AJS.namespace('GH.Exception', null, require('jira-agile/rapid/exception'));
