define('jira-agile/rapid/analytics', ['jquery', 'underscore'], function ($, _) {
    "use strict";
    /**
     * A flag so that other JavaScript resources can identify whether analytics is enabled.
     *
     * Analytics is always "enabled" in the sense that we should trigger events
     * and this is only required until we refactor the AnalyticsTracker. The Atlassian Analytics client decides whether or not
     * to forward those events beyond the instance.
     *
     * @type {boolean}
     * @const
     */
    GH.AnalyticsEnabled = true;

    /**
     * The key which we bind to for events. Other resources should trigger events with this key to capture analytics.
     *
     * @type {string}
     * @const
     */
    const ANALYTICS_EVENT_KEY = "gh.analytics.async";

    /**
     * The structure of event data that should be sent when triggering events with key ANALYTICS_EVENT_KEY.
     *
     * @typedef {Object} AnalyticsEventData
     * @property {string} category
     * @property {string} action
     * @property {string} [label]
     * @property {*} [value]
     */

    /**
     * Starts listening for gh.analytics.async events and triggers a corresponding Atlassian Analytics event with the
     * event data that was sent.
     *
     */
    function initAnalytics() {
        $(document).on(ANALYTICS_EVENT_KEY, function (e, eventData) {
            trackEvent(eventData);
        });
    }

    /**
     * Abstraction layer of Atlassian Analytics.
     *
     * @param {AnalyticsEventData} eventData
     */
    function trackEvent(eventData) {
        var data = {};
        var name = eventData.category;

        if (!_.isUndefined(eventData.action)) {
            name = name + "." + eventData.action;
        }

        if (!_.isUndefined(eventData.label)) {
            data.label = eventData.label;
        }

        if (!_.isUndefined(eventData.value)) {
            data.value = eventData.value;
        }

        AJS.trigger("analytics", {name: name, data: data});
    }

    initAnalytics();

    const analytics = {};
    analytics.isEnabled = () => GH.AnalyticsEnabled;
    analytics.setEnabled = enabled => GH.AnalyticsEnabled = enabled;

    return analytics;
});
AJS.namespace('GH.Analytics', null, require('jira-agile/rapid/analytics'));