/*jshint unused:false */

AJS.test.require(['com.pyxis.greenhopper.jira:gh-project-picker'], function () {
    var stubbedMethods = '_processAjax,_getProjectDescriptor,_findProject,_findUnmatchingProjects'.split(',');

    function commonSetup() {
        sinon.stub(GH.Ajax, 'buildRestUrl');
        this.itemDescriptorStub = sinon.stub();
        this.requestSuggestionsDeferred = AJS.$.Deferred();
        this.$select = {
            bind: sinon.stub(),
            val: sinon.stub()
        };
        this.multiselect = {
            addItem: sinon.stub(),
            requestSuggestions: sinon.stub().returns(this.requestSuggestionsDeferred.promise())
        };

        sinon.stub(AJS, '$').returns(this.$select);
        sinon.stub(AJS, 'warn');
        sinon.stub(AJS, 'MultiSelect').returns(this.multiselect);
        sinon.stub(AJS, 'ItemDescriptor').returns(this.itemDescriptorStub);

        this.context = AJS.test.mockableModuleContext();

        this.ProjectPicker = this.context.require('jira-agile/projects/project-picker');
        stubbedMethods.forEach(function (method) {
            sinon.stub(this.ProjectPicker.prototype, method);
        }.bind(this));
    }

    function commonTeardown() {
        GH.Ajax.buildRestUrl.restore();
        AJS.$.restore();
        AJS.warn.restore();
        AJS.MultiSelect.restore();
        AJS.ItemDescriptor.restore();
        stubbedMethods.forEach(function (method) {
            if (this.ProjectPicker.prototype[method].restore) {
                this.ProjectPicker.prototype[method].restore();
            }
        }.bind(this));
    }

    module('jira-agile/projects/project-picker - constructor', {
        setup: commonSetup,
        teardown: commonTeardown
    });

    test('it should maintain a copy of the jquery element for the dropdown', function () {
        var selector = '#winning',
            expected = 'yes you are';
        AJS.$.withArgs(selector).returns(expected);

        var picker = new this.ProjectPicker({ selector: selector });

        ok(AJS.$.calledWith(selector));
        equal(picker.$select, expected);
    });
    test('it should set the currentProjectId if one is passed in', function () {
        var projectId = 101010;

        var picker = new this.ProjectPicker({
            selector: true,
            currentProjectId: projectId
        });

        equal(picker.currentProjectId, projectId);
    });
    test('it should not set the currentProjectId if none is passed in', function () {
        var picker = new this.ProjectPicker({
            selector: true
        });

        ok(!picker.hasOwnProperty('currentProjectId'));
        ok(!picker.hasOwnProperty('currentProject'));
    });
    test('it should create a new AJS.MultiSelect', function () {
        var picker = new this.ProjectPicker({
            selector: true
        });

        ok(AJS.MultiSelect.calledOnce);
        equal(picker.multiselect, this.multiselect);
    });
    test('it should bind the "unselect and "change" events if a change event is passed in', function () {
        var change = function change() {};

        var picker = new this.ProjectPicker({
            selector: true,
            change: change
        });

        ok(this.$select.bind.calledWith('unselect', change));
        ok(this.$select.bind.calledWith('change', change));
    });
    test('it should not bind the "unselect and "change" events no change event is passed in', function () {
        var picker = new this.ProjectPicker({
            selector: true
        });

        ok(this.$select.bind.notCalled);
    });
    test('it should not request any suggestions if no currentProjectId was passed in', function () {
        var picker = new this.ProjectPicker({
            selector: true
        });

        ok(this.multiselect.requestSuggestions.notCalled);
    });
    test('it should force request suggestions if a currentProjectId exists', function () {
        var picker = new this.ProjectPicker({
            selector: true,
            currentProjectId: 2101
        });

        ok(this.multiselect.requestSuggestions.calledOnce);
    });

    module('jira-agile/projects/project-picker - request suggestions handler', {
        setup: commonSetup,
        teardown: commonTeardown
    });

    asyncTest('it should not add the current project to the dropdown if no project could be found with a matching id', function () {
        var picker = new this.ProjectPicker({
            selector: true,
            currentProjectId: 2101
        });
        this.requestSuggestionsDeferred.resolve();

        setTimeout(function () {
            start();
            ok(this.multiselect.addItem.notCalled);
        }.bind(this), 0);
    });
    asyncTest('it should add the current project to the dropdown if a project could be found with a matching id', function () {
        var descriptor = 'some kewl description';
        this.ProjectPicker.prototype._getProjectDescriptor.returns(descriptor);

        var picker = new this.ProjectPicker({
            selector: true,
            currentProjectId: 2101
        });
        picker.currentProject = {};
        this.requestSuggestionsDeferred.resolve();

        setTimeout(function () {
            start();
            ok(this.multiselect.addItem.calledOnce);
            ok(this.multiselect.addItem.calledWith(descriptor));
        }.bind(this));
    });

    module('jira-agile/projects/project-picker - find current project', {
        setup: function setup() {
            commonSetup.call(this);
            this.ProjectPicker.prototype._findProject.restore();
        },
        teardown: commonTeardown
    });

    test('it should return the matching project from a list of projects', function () {
        var targetId = 10101,
            match = {
            id: targetId
        },
            projects = [{
            id: 'some fake id'
        }, {
            id: null
        },
        // this project will match
        match];

        var picker = new this.ProjectPicker({ selector: true }),
            result = picker._findProject(projects, targetId);

        equal(result, match);
    });
    test('it should return the first matching project from a list of projects', function () {
        var targetId = 10101,
            match = {
            id: targetId,
            name: 'first match'
        },
            match2 = {
            id: targetId,
            name: 'second match'
        },
            projects = [{
            id: 'some fake id'
        },
        // these projects will match
        match, match2];

        var picker = new this.ProjectPicker({ selector: true }),
            result = picker._findProject(projects, targetId);

        equal(result, match);
    });
    test('it should not return the current project if a match cannot be found', function () {
        var targetId = 10101,
            projects = [{
            id: targetId + 1
        }];

        var picker = new this.ProjectPicker({ selector: true }),
            result = picker._findProject(projects, targetId);

        ok(!result);
    });
    test('it should not return the current project if no target is provided', function () {
        var targetId = 10101,
            projects = [{
            id: targetId
        }];

        var picker = new this.ProjectPicker({ selector: true }),
            result = picker._findProject(projects);

        ok(!result);
    });

    module('jira-agile/projects/project-picker - find non-current projects', {
        setup: function setup() {
            commonSetup.call(this);

            this.ProjectPicker.prototype._findUnmatchingProjects.restore();
            this.currentProjectId = 10101;
            this.unmatchingProjects = [];
            this.picker = new this.ProjectPicker({ selector: true });

            // lots of no matches
            for (var i = 1; i <= 5; i++) {
                this.unmatchingProjects.push({
                    id: this.currentProjectId + i
                });
            }

            this.projects = this.unmatchingProjects.slice(0);
        },
        teardown: function teardown() {
            commonTeardown.call(this);
        }
    });

    test('it should return all projects if no target project id is passed in', function () {
        var result = this.picker._findUnmatchingProjects(this.projects);

        deepEqual(result, this.unmatchingProjects);
    });
    test('it should return all projects if no match is found', function () {
        // there is no project with 'id'
        var result = this.picker._findUnmatchingProjects(this.projects, this.currentProjectId);

        deepEqual(result, this.unmatchingProjects);
    });
    test('it should exclude projects that match the target id', function () {
        // add a match
        this.projects.push({
            id: this.currentProjectId
        });

        var result = this.picker._findUnmatchingProjects(this.projects, this.currentProjectId);

        deepEqual(result, this.unmatchingProjects);
        notDeepEqual(result, this.projects);
    });

    module('jira-agile/projects/project-picker - get project descriptor', {
        setup: function setup() {
            commonSetup.call(this);
            this.ProjectPicker.prototype._getProjectDescriptor.restore();
        },
        teardown: commonTeardown
    });

    test('it should use an empty project when attempting to get a project descriptor if no project is provided', function () {
        var picker = new this.ProjectPicker({ selector: true });

        raises(picker._getProjectDescriptor);
    });
    test('it should return an AJS.ItemDescriptor with the project details', function () {
        var project = {
            id: 10101,
            displayName: 'Alex\'s <strong>cool</strong> project'
        },
            expectedArgs = {
            // id is converted to a string
            value: project.id + '',
            label: project.displayName,
            html: AJS.escapeHTML(String(project.displayName))
        },
            picker = new this.ProjectPicker({ selector: true });

        picker._getProjectDescriptor(project);

        deepEqual(AJS.ItemDescriptor.args[0][0], expectedArgs);
    });

    module('jira-agile/projects/project-picker - process the ajax response', {
        setup: function setup() {
            commonSetup.call(this);
            this.ProjectPicker.prototype._processAjax.restore();

            // skip the transform to project descriptor step - just return the project unchanged
            this.ProjectPicker.prototype._getProjectDescriptor.returnsArg(0);
        },
        teardown: function teardown() {
            commonTeardown.call(this);
        }
    });

    test('it should return an empty array if no parameters are passed in', function () {
        var picker = new this.ProjectPicker({ selector: true });

        var result = picker._processAjax();

        deepEqual(result, []);
    });
    test('it should return an empty array if no projects are available', function () {
        var picker = new this.ProjectPicker({ selector: true });

        var result1 = picker._processAjax({}),
            result2 = picker._processAjax({ projects: [] });

        deepEqual(result1, []);
        deepEqual(result2, []);
    });
    test('it should set the currentProject if it exists', function () {
        var currentProjectId = 100012,
            project = {
            id: currentProjectId
        },
            response = {
            projects: [project]
        },
            picker = new this.ProjectPicker({
            selector: true,
            currentProjectId: currentProjectId
        });
        this.ProjectPicker.prototype._findProject.returns(project);
        this.ProjectPicker.prototype._findUnmatchingProjects.returns([]);

        picker._processAjax(response);

        ok(picker._findProject.calledWith(response.projects, currentProjectId));
        equal(picker.currentProject, project);
    });
    test('it should return the current project and then the rest of the projects', function () {
        var currentProjectId = 100012,
            match = {
            id: currentProjectId
        },
            nonMatch = {
            id: 'some other id'
        },
            response = {
            // test should reorder these
            projects: [nonMatch, match]
        },
            picker = new this.ProjectPicker({
            selector: true,
            currentProjectId: currentProjectId
        });
        this.ProjectPicker.prototype._findProject.returns(match);
        this.ProjectPicker.prototype._findUnmatchingProjects.returns([nonMatch]);

        var result = picker._processAjax(response);

        // This would normally be the descriptors and not the project objects themselves.
        // However, for the purposes of this test the _getProjectDescriptor just returns the project
        deepEqual(result, [match, nonMatch]);
    });
    test('it should return all the project descriptors if there is no current project', function () {
        var currentProjectId = 100012,
            nonMatch = {
            id: 'some other id'
        },
            response = {
            projects: [nonMatch]
        },
            picker = new this.ProjectPicker({
            selector: true,
            currentProjectId: currentProjectId
        });
        this.ProjectPicker.prototype._findProject.returns(null);
        this.ProjectPicker.prototype._findUnmatchingProjects.returns([nonMatch]);

        var result = picker._processAjax(response);

        deepEqual(result, [nonMatch]);
    });

    module('jira-agile/projects/project-picker - legacy methods', {
        setup: function setup() {
            commonSetup.call(this);
            this.picker = new this.ProjectPicker({ selector: true });
        },
        teardown: commonTeardown
    });

    test('it should log a deprecation message when people attempt to use the "show" method', function () {
        this.picker.show();

        ok(AJS.warn.calledOnce);
    });

    test('it should return the $select element', function () {
        var result = this.picker.getElement();

        equal(result, this.picker.$select);
    });

    test('it should return whether the $select element has a value', function () {
        this.picker.$select.val.returns(document.createElement('div'));

        var result = this.picker.hasValue();

        equal(result, true);
    });

    test('it should return whether the $select element has no value', function () {
        this.picker.$select.val.returns(null);

        var result = this.picker.hasValue();

        equal(result, false);
    });

    test('it should return the $select elements value', function () {
        var value = 'superman';
        this.picker.$select.val.returns(value);

        var result = this.picker.getValue();

        equal(result, value);
    });

    test('it should know if the multiselect has errors', function () {
        var stub = sinon.stub().withArgs('.error').returns([true]);
        this.picker.multiselect.$container = {
            parent: function parent() {
                return {
                    find: stub
                };
            }
        };

        var result = this.picker.hasErrors();

        ok(result);
        ok(stub.calledWith('.error'));
    });
    test('it should know if the multiselect has no errors', function () {
        var stub = sinon.stub().withArgs('.error').returns([]);
        this.picker.multiselect.$container = {
            parent: function parent() {
                return {
                    find: stub
                };
            }
        };

        var result = this.picker.hasErrors();

        ok(!result);
        ok(stub.calledWith('.error'));
    });

    module('jira-agile/projects/project-picker - global export');

    test('it should create a global namespace for legacy purposes', function () {

        // this should exist by virtue of loading in `project-picker.js`
        ok(GH.ProjectPicker);
    });
});