define('jira-agile/projects/project-picker', [], function () {
    'use strict';

    /**
     * An auto complete field containing the existing projects
     * @param {Object} options
     * @param {string} options.selector - the css selector for the select element
     * @param {number=} options.currentProjectId - this will attempt to present the project with the matching project-id by default
     * @param {function=} options.change - function to be fired on change and unselect events
     *
     * @constructor
     */
    var ProjectPicker = function (options) {

        if (!options || !options.selector) {
            throw new Error('Project picker requires a selector');
        }

        this.$select = AJS.$(options.selector);

        // can pass in a currentProjectId
        if (options.currentProjectId) {
            this.currentProjectId = options.currentProjectId;
            this.currentProject = null;
        }

        this.multiselect = new AJS.MultiSelect({
            element: this.$select,
            width: 300,
            itemAttrDisplayed: 'label',
            ajaxOptions: {
                url: GH.Ajax.buildRestUrl('/project.json'),
                formatResponse: this._processAjax.bind(this)
            }
        });

        // change event
        if (typeof options.change === 'function') {
            this.$select.bind('unselect', options.change);
            this.$select.bind('change', options.change);
        }

        if (!this.currentProjectId) {
            return;
        }

        // force preload of the suggestions
        this.multiselect.requestSuggestions(true)
            .then(function () {
                // could not find the currentProject in the ajax response
                // just exit here
                if (!this.currentProject) {
                    return;
                }

                this.multiselect.addItem(this._getProjectDescriptor(this.currentProject));
            }.bind(this));
    };

    /**
     * @deprecated
     */
    ProjectPicker.prototype.show = function () {
        AJS.warn('ProjectPicker.show is deprecated. It is no longer needed as all initialisation occurs in the constructor');
    };

    ProjectPicker.prototype.hasValue = function () {
        return !!this.$select.val();
    };

    ProjectPicker.prototype.hasErrors = function () {
        return this.multiselect.$container.parent().find('.error').length > 0;
    };

    ProjectPicker.prototype.getValue = function () {
        return this.$select.val();
    };

    ProjectPicker.prototype.getElement = function () {
        return this.$select;
    };

    /**
     * Finds a project in an array with a matching id and returns it
     * @param {Array} projects
     * @param {number} targetProjectId
     * @returns {?project}
     * @private
     */
    ProjectPicker.prototype._findProject = function (projects, targetProjectId) {
        return (projects || [])
            .filter(function (project) {
                return targetProjectId && project.id === targetProjectId;
            }.bind(this))[0];
    };
    ProjectPicker.prototype._findUnmatchingProjects = function (projects, targetProjectId) {
        return (projects || [])
            .filter(function (project) {
                return !targetProjectId || (project.id !== targetProjectId);
            }.bind(this));
    };
    /**
     * Converts ajax response into an array of AJS.ItemDescriptor's
     * @param {object} response - what the server returns
     * @param {Array<project>} response.projects
     * @returns {Array<AJS.ItemDescriptor>}
     * @private
     */
    ProjectPicker.prototype._processAjax = function (response) {
        if (!response || !response.projects || !response.projects.length) {
            return [];
        }

        this.currentProject = this._findProject(response.projects, this.currentProjectId);

        var otherProjectsDescriptors = this._findUnmatchingProjects(response.projects, this.currentProjectId)
            .map(this._getProjectDescriptor.bind(this));


        // no currentProjectId or could not find the project
        // otherProjectsDescriptors will contain all the projects
        if (!this.currentProject) {
            return otherProjectsDescriptors;
        }


        // currentProject comes first in the list
        return [this._getProjectDescriptor(this.currentProject)].concat(otherProjectsDescriptors);

        // TODO: have two groups (current project and other projects). This is blocked by these issues:
        // 1. SW-2325 - AJS.MultiSelect does not handle group descriptors well (https://jdog.jira-dev.com/browse/SW-2325)
        // 2. SW-2326 - Update project picker so that it separates projects into 'current' and 'other' (https://jdog.jira-dev.com/browse/SW-2326) [blocked by SW-2325]
    };

    /**
     * Returns an item descriptor for a project
     * @param {!project}
     * @throws will throw an error if no project is provided
     * @returns {AJS.ItemDescriptor}
     * @private
     */
    ProjectPicker.prototype._getProjectDescriptor = function (project) {
        if (!project) {
            throw new Error('ProjectPicker.prototype._getProjectDescriptor requires a project');
        }

        return new AJS.ItemDescriptor({
            // value of item added to select
            value: project.id + '',
            // title of lozenge -- This is already HTML-escaped by MultiSelect, so no need to do it here
            label: project.displayName,
            // html used in suggestion -- This is NOT HTML-escaped by MultiSelect, so we must do so explicitly
            html: AJS.escapeHTML(String(project.displayName))
        });
    };

    return ProjectPicker;
});

AJS.namespace('GH.ProjectPicker', null, require('jira-agile/projects/project-picker'));