define('jira-agile/gh-link-issue-dialog', ["jira/ajs/ajax/smart-ajax", "jira/issue", "jira/issuenavigator/issue-navigator", "jira/dialog/dialog-register", "jquery", "jira/jquery/deferred"], function (SmartAjax, IssueApi, IssueNavigatorApi, Dialogs, $, Deferred) {

    var LinkIssueDialog = {};

    /**
     * Creates a link issue dialog
     *
     * @param {String, jQuery} trigger - jQuery selector for the element/s that trigger dialog to open
     */
    LinkIssueDialog.create = function () {

        /**
         * Adds isSelected property to linkType that should currently be selected
         *
         * @param {String} activeId
         * @param {Array} linkTypes
         * @return {Array}
         */
        function markSelectedLinkType(activeId, linkTypes) {

            var hasSelected = false;

            // If active link id equals link type set it as selected
            $.each(linkTypes, function (i, linkType) {
                if (linkType.id === activeId) {
                    linkType.isSelected = true;
                    hasSelected = true;
                    return false;
                }
            });

            // Otherwise assume it is the first one.
            if (!hasSelected) {
                linkTypes[0].isSelected = true;
            }

            return linkTypes;
        }

        /**
         * Goes to the server to get array of remote issue link types.
         *
         * @return jQuery.Promise<Array>
         */
        function getRemoteIssueLinks(issueId) {
            var deferred = Deferred();
            var wrmContextPath = require('wrm/context-path');
            SmartAjax.makeRequest({
                url: wrmContextPath() + "/rest/viewIssue/1/remoteIssueLink/linkType?issueId=" + issueId,
                complete: function complete(xhr, textStatus, smartAjaxResult) {
                    if (smartAjaxResult.successful) {
                        deferred.resolve(smartAjaxResult.data);
                    } else {
                        deferred.reject(SmartAjax.buildDialogErrorContent(smartAjaxResult));
                    }
                }
            });

            return deferred.promise();
        }

        /**
         * Loads tab content using the active trigger's (<a> clicked) href as the url to request
         *
         * @param {JIRA.Dialog} dialog
         * @return jQuery.Promise<String>
         */
        function getTabContent(dialog) {

            var deferred = Deferred(),
                ajaxOptions = Dialogs.getDefaultAjaxOptions.call(dialog);

            ajaxOptions.complete = function (xhr, textStatus, smartAjaxResult) {
                if (smartAjaxResult.successful) {
                    deferred.resolve(smartAjaxResult.data);
                } else {
                    deferred.reject(SmartAjax.buildDialogErrorContent(smartAjaxResult));
                }
            };

            SmartAjax.makeRequest(ajaxOptions);

            return deferred.promise();
        }

        /**
         * Gets id from active trigger
         *
         * @param {JIRA.Dialog} dialog
         * @return {String}
         */
        function getActiveTabId(dialog) {
            return dialog.$activeTrigger.attr("id");
        }

        return function (trigger, selectedLinkType, successCallback, currentIssueId) {

            successCallback = successCallback || Dialogs.storeCurrentIssueIdOnSucessfulSubmit;
            /*
             * This line is the difference between JIRA's LinkIssueDialog.js and our one:
             return new JIRA.FormDialog({
            */
            return new GH.FormDialog({
                id: "link-issue-dialog",
                trigger: trigger,
                onSuccessfulSubmit: successCallback,
                issueMsg: 'thanks_issue_linked',
                widthClass: 'large',
                stacked: true,
                content: function content(ready) {

                    var dialog = this,
                        issueId = currentIssueId || IssueApi.getIssueId() || IssueNavigatorApi.getSelectedIssueId();

                    getTabContent(dialog).done(function (tabContent) {

                        getRemoteIssueLinks(issueId).done(function (linkTypes) {

                            var activeLinkType = selectedLinkType || getActiveTabId(dialog);

                            linkTypes = markSelectedLinkType(activeLinkType, linkTypes);

                            ready(JIRA.Templates.LinkIssue.dialog({
                                linkTypes: linkTypes,
                                tabContent: tabContent
                            }));

                            $.each(linkTypes, function (i, linkType) {
                                var focusElementSelector;
                                if (linkType.isSelected === true) {
                                    focusElementSelector = "[name='" + linkType.focusedFieldName + "']";
                                    if ($(focusElementSelector).length === 0) {
                                        focusElementSelector = "#" + linkType.id;
                                    }
                                    dialog._focusFirstField(focusElementSelector);
                                }
                            });
                        }).fail(function (errorContent) {
                            ready(errorContent);
                        });
                    }).fail(function (errorContent) {
                        ready(errorContent);
                    });
                },
                // A work around because the submit response does not include the menu bar. So we push the submit response
                // into our current content.
                formatSubmitResponseContent: function formatSubmitResponseContent(content) {
                    this.get$popupContent().find(".dialog-pane").html(content);
                    return this.get$popupContent().html();
                },
                onContentRefresh: function onContentRefresh() {
                    // make sure that whenever we click on of our other tabs they load in the dialog.
                    this.bindAnchorsToDialog(this.get$popupContent().find(".dialog-menu-item"));
                }
            });
        };
    }();

    return LinkIssueDialog;
});

AJS.namespace('GH.LinkIssueDialog', null, require('jira-agile/gh-link-issue-dialog'));