define("jira-dashboard-items/sprint-days-remaining", ['jquery', 'underscore', "jira-dashboard-items/sprint-days-remaining-view"], function ($, _, SprintDaysRemainingView) {

    var AUTO_SPRINT = "auto";

    /**
     *
     * @param {InlineGadgetAPI} API
     * @param {Object} [options] for dashboard item
     * @constructor
     */
    var DashboardItem = function DashboardItem(API, options) {

        this.API = API;
        this.options = options || {};
        this.sprintDaysRemainingView = new SprintDaysRemainingView(this.API.getGadgetId());
        this.initialiseEventHandlers();
    };

    /**
     * Render the configured sprint-days-remaining dashboard item.
     * @param {node} element to apply dashboard item to
     * @param {Object} preferences for dashboard item that have been configured.
     */

    DashboardItem.prototype.render = function (element, preferences) {

        function getSprintIdFromData(sprintIdData) {
            var sprint = _.findWhere(sprintIdData.sprints, { state: "ACTIVE" });
            return sprint ? sprint.id : null;
        }

        preferences = preferences || {};

        var dashboardItem = this;

        dashboardItem.sprintDaysRemainingView.setElement(element);

        var rapidViewId = preferences.rapidViewId,
            sprintId = preferences.sprintId,
            isConfigured = preferences.isConfigured;

        //  If the dashboard item does not have any active sprints and is therefore not configured,
        //  show an error/warning message.
        if (!isConfigured || !rapidViewId) {
            dashboardItem.sprintDaysRemainingView.showSprintCannotBeFoundErrorMessage();
            dashboardItem.API.hideLoadingBar();
            return;
        }

        dashboardItem.API.showLoadingBar();

        // Init auto-refresh
        dashboardItem.API.initRefresh(preferences, dashboardItem.render.bind(dashboardItem, element, preferences));

        //  If the sprintId is set to 'auto' (in the initialmost phase), then request all sprints associated
        //  with the given rapidViewId, otherwise return the given sprintId in an appropriate format.
        requestSprintData(sprintId, rapidViewId).done(function (sprintIdData) {

            sprintId = getSprintIdFromData(sprintIdData);
            if (!sprintId) {
                dashboardItem.sprintDaysRemainingView.showSprintNotActiveErrorMessage();
                dashboardItem.API.hideLoadingBar();
                return;
            }

            //  Request for data pertaining to the current rapidView (board) selected, and the current
            //  sprintId selected.
            $.when(requestSprintDaysRemainingData(sprintId, rapidViewId), requestRapidViewData(rapidViewId)).done(function (sprintDaysRemainingData, rapidViewData) {

                if (sprintDaysRemainingData.length === 0 || rapidViewData.length === 0) {
                    dashboardItem.sprintDaysRemainingView.showSprintCannotBeFoundErrorMessage();
                    return;
                }

                sprintDaysRemainingData = sprintDaysRemainingData[0];
                rapidViewData = rapidViewData[0].currentViewConfig;

                //  After both AJAX requests have successfully completed, begin rendering the dashboard item
                //  (capture failure/errors below the .done sequence).
                dashboardItem._renderDashboardItem(element, preferences, sprintDaysRemainingData, rapidViewData);
            }).fail(function (response) {
                if (response.status === 403) {
                    dashboardItem.sprintDaysRemainingView.showGadgetNotAllowedToRenderErrorMessage();
                } else {
                    dashboardItem.sprintDaysRemainingView.showRapidViewCannotBeFoundErrorMessage();
                }
            }).always(function () {
                dashboardItem.API.hideLoadingBar();
            });
        }).fail(function (response) {
            if (response.status === 403) {
                dashboardItem.sprintDaysRemainingView.showGadgetNotAllowedToRenderErrorMessage();
            } else {
                dashboardItem.sprintDaysRemainingView.showRapidViewCannotBeFoundErrorMessage();
            }
            dashboardItem.API.hideLoadingBar();
        });
    };

    /**
     * Use the data acquired from AJAX requests to render the page
     * @param {Node} element to render to
     * @param {Object} preferences
     * @param {Object} sprintDaysRemainingData
     * @param {Object} rapidViewData
     */
    DashboardItem.prototype._renderDashboardItem = function (element, preferences, sprintDaysRemainingData, rapidViewData) {

        //  Render the dashboard item, passing in all params required by the Soy file.
        var renderParams = {
            rapidViewName: rapidViewData.name,
            showRapidViewName: preferences.showRapidViewName,
            sprintName: sprintDaysRemainingData.name,
            showSprintName: preferences.showSprintName,
            days: sprintDaysRemainingData.timeRemaining.days,
            preferences: preferences
        };

        this.sprintDaysRemainingView.renderDashboardItem(element, renderParams);
    };

    /**
     * Render the configuration screen for the sprint-days-remaining dashboard item
     * @param {Node} element to render into
     * @param {Object} preferences for object.
     */
    DashboardItem.prototype.renderEdit = function (element, preferences) {

        function getRapidViewData(data) {
            data = _.where(data.views, { sprintSupportEnabled: true });
            return transformData(data);
        }

        function transformData(data) {
            return _.map(data, function (obj) {
                return {
                    text: obj.name,
                    value: obj.id
                };
            });
        }

        preferences = preferences || {};

        var dashboardItem = this;

        dashboardItem.sprintDaysRemainingView.setElement(element);
        dashboardItem.API.showLoadingBar();

        //  Fire AJAX request for all current rapidViews (boards)
        $.ajax({
            method: "GET",
            url: AJS.contextPath() + "/rest/greenhopper/1.0/rapidview"
        }).done(function (data) {

            //  If no boards are currently available, show the appropriate error message
            if (data.length === 0) {
                dashboardItem.sprintDaysRemainingView.showNoBoardsAvailableErrorMessage();
                dashboardItem.API.hideLoadingBar();
                return;
            }

            // Prefix to avoid collisions with other instances of this dashboard item.
            var prefix = dashboardItem.API.getGadgetId() + "-";
            var boardTypes = getRapidViewData(data);
            var activeSprints = {};

            var configViewParams = {
                prefix: prefix,
                boardTypes: boardTypes,
                activeSprints: activeSprints,
                checkboxes: [{
                    'id': 'showRapidViewName',
                    'labelText': AJS.I18n.getText('gh.gadget.userpref.showrvname.name'),
                    'isChecked': preferences.showRapidViewName
                }, {
                    'id': 'showSprintName',
                    'labelText': AJS.I18n.getText('gh.gadget.userpref.showspname.name'),
                    'isChecked': preferences.showSprintName
                }],
                preferences: preferences
            };

            //  Render the configuration view, passing in all required params
            dashboardItem.sprintDaysRemainingView.renderConfigView(element, configViewParams);
        }).fail(function () {
            dashboardItem.sprintDaysRemainingView.showGadgetNotAllowedToRenderErrorMessage();
        }).always(function () {
            dashboardItem.API.hideLoadingBar();
        });
    };

    DashboardItem.prototype.updateSprintData = function (rapidViewId) {

        var self = this;

        requestSprintData(false, rapidViewId).done(function (data) {

            data = _.where(data.sprints, { state: "ACTIVE" });
            self.sprintDaysRemainingView.renderCurrentSprints(data);
        }).fail(function () {

            self.sprintDaysRemainingView.showSprintNotActiveErrorMessage();
            self.API.hideLoadingBar();
        });
    };

    DashboardItem.prototype.initialiseEventHandlers = function () {

        var sprintDaysRemaining = this;
        var sprintDaysRemainingView = $(this.sprintDaysRemainingView);

        sprintDaysRemainingView.on('boardChange', function (e, rapidViewId) {
            sprintDaysRemaining.updateSprintData(rapidViewId);
        });

        sprintDaysRemainingView.on('cancel', function () {
            sprintDaysRemaining.API.closeEdit();
        });

        sprintDaysRemainingView.on('submit', function (e, preferences) {
            sprintDaysRemaining.API.savePreferences(preferences);
        });

        sprintDaysRemainingView.on('refreshLayout', function () {
            sprintDaysRemaining.API.forceLayoutRefresh();
        });

        sprintDaysRemainingView.on('hideLoadingBar', function () {
            sprintDaysRemaining.API.hideLoadingBar();
        });
    };

    function requestSprintDaysRemainingData(sprintId, rapidViewId) {
        return $.ajax({
            method: "GET",
            url: AJS.contextPath() + "/rest/greenhopper/1.0/gadgets/sprints/health",
            data: {
                rapidViewId: rapidViewId,
                sprintId: sprintId
            }
        });
    }

    function requestRapidViewData(rapidViewId) {
        return $.ajax({
            method: "GET",
            url: AJS.contextPath() + "/rest/greenhopper/1.0/xboard/config?rapidViewId=" + rapidViewId
        });
    }

    function requestSprintData(sprintId, rapidViewId) {
        if (sprintId === AUTO_SPRINT || sprintId === false) {
            return $.ajax({
                method: "GET",
                url: AJS.contextPath() + "/rest/greenhopper/1.0/sprintquery/" + rapidViewId
            });
        } else {
            var deferred = $.Deferred();
            deferred.resolve({
                sprints: [{
                    id: sprintId,
                    state: "ACTIVE"
                }]
            });
            return deferred;
        }
    }

    return DashboardItem;
});