define('jira/auditing/records', ['jira/ajs/ajax/smart-ajax', 'jira/auditing/record', 'wrm/context-path', 'backbone', 'jquery', 'underscore'], function (SmartAjax, Record, wrmContextPath, Backbone, $, _) {
    'use strict';

    var Paginator = Backbone.Paginator;
    return Paginator.requestPager.extend({
        model: Record,
        filters: null, // filters collection

        // going to remember the first record we saw when loading the page so pagination doesn't change when new events are added
        // later we're going to add a pop up letting the user know it's time to refresh the page because new records were added
        topId: undefined,

        //this goes as POST params, this is a workaround to make them empty instead of "null" or "undefined"
        refreshTopId: undefined,

        // used to block turning pending state on in case of initial request (the one that is made for sync collection
        // if url contains hash)
        isInitialRequest: false,

        // used to not cause inability to navigate back after user visited non-existing page and collection was updated
        // to the latest page available; see this.ensurePageAvailable()
        _ensuringPageAvailable: false,

        _lastRefreshRange: undefined,
        _pullingInterval: undefined,
        _pullingIntervalTime: 30000, //ms

        // these two params are used to determine if there should be created new history item or current url should be replaced;
        // this fixes problem with inability to navigate back after empty result set has been refreshed
        isRefreshing: false,
        topIdBeforeRefresh: undefined,

        initialize: function initialize(models, options) {
            this.filters = options.filters;
        },

        initializePulling: function initializePulling() {
            this._enablePulling();

            // don't check for newer records if tab is inactive
            $(document).on("visibilitychange", this._togglePulling.bind(this));

            this.on("sync", this._enablePulling, this);
        },

        // asks server about highest id of records for current criteria, returns promise
        _checkForNewerRecords: function _checkForNewerRecords() {
            var deferred = new $.Deferred();
            var params = _.extend({ count: 1, page: 1 }, this.filters.getAllValues());

            SmartAjax.makeRequest({
                cache: false,
                data: $.param(params),
                dataType: "json",
                type: "GET",
                url: wrmContextPath() + "/rest/jira-auditing-plugin/1/view",
                success: _.bind(function (response) {
                    this.setRefreshTopId(response.refreshMaxId);
                    deferred.resolve();
                }, this),
                error: function error() {
                    deferred.reject();
                }
            });

            return deferred.promise();
        },

        // ask server if there are newer records, in such case stop pulling and trigger event
        _pullingFunc: function _pullingFunc() {
            this._checkForNewerRecords().done(_.bind(function () {
                if (this.isRefreshable()) {
                    this._disablePulling();
                    this.trigger("auditing:update-records-possible");
                }
            }, this));
        },

        _togglePulling: function _togglePulling() {
            if (document.hidden) {
                this._disablePulling();
            } else {
                this._enablePulling();
            }
            return this;
        },

        // helper function, enables pulling if disabled
        _enablePulling: function _enablePulling() {
            if (!this._pullingInterval && !this.isRefreshable()) {
                this._pullingInterval = setInterval(this._pullingFunc.bind(this), this._pullingIntervalTime);
            }
        },

        _disablePulling: function _disablePulling() {
            clearInterval(this._pullingInterval);
            this._pullingInterval = undefined;
        },

        getCurrentParams: function getCurrentParams() {
            return $.extend({
                topId: this.topId,
                currentPage: this.currentPage
            }, this.filters.getAllValues());
        },

        setCurrentParams: function setCurrentParams(params) {
            this.topId = params.topId || this.topId;
            this.currentPage = parseInt(params.currentPage, 10) || 1;

            this.filters.setFilters(params);

            return this;
        },

        hasExactParams: function hasExactParams(params) {
            return _.isEqual(params, this.getCurrentParams());
        },

        isRefreshable: function isRefreshable() {
            return !!(this.refreshTopId && (this.topId && +this.refreshTopId > +this.topId || typeof this.topId === "undefined"));
        },

        setRefreshTopId: function setRefreshTopId(refreshTopId) {
            this.refreshTopId = refreshTopId;
        },

        getLastRefreshRange: function getLastRefreshRange() {
            return this._lastRefreshRange;
        },

        ensurePageAvailable: function ensurePageAvailable() {
            if (this.size() === 0 && this.totalRecords > 0) {
                this._ensuringPageAvailable = true;
                this.goTo(this.info().totalPages);
            }
        },

        comparator: function descendingIdComparator(a, b) {
            return parseInt(b.get("id")) - parseInt(a.get("id"));
        },

        paginator_core: {
            type: 'GET',
            dataType: 'json',
            url: wrmContextPath() + "/rest/jira-auditing-plugin/1/view"
        },

        paginator_ui: {
            firstPage: 1,
            perPage: null,
            pagesInRange: 2
        },

        server_api: {
            'count': function count() {
                return this.perPage;
            },
            'maxId': function maxId() {
                return this.topId || '';
            },
            'page': function page() {
                return this.currentPage;
            }
        },

        search: function search() {
            this.topId = undefined;
            this.goTo(this.information.firstPage);
        },

        refresh: function refresh() {
            this.isRefreshing = true;
            this.topIdBeforeRefresh = this.topId;

            this.topId = undefined;
            this.goTo(this.information.firstPage);
        },

        parse: function parse(response) {
            this.totalRecords = response.count;
            this.empty = response.empty;
            this.setRefreshTopId(response.refreshMaxId);

            if (this.topId === undefined && response.records.length > 0) {
                this.topId = response.records[0].id;
            }

            this._lastRefreshRange = this.isRefreshing && this.topId ? {
                from: this.topIdBeforeRefresh ? +this.topIdBeforeRefresh + 1 : 0,
                to: +this.topId
            } : undefined;

            return response.records;
        },

        xhr: null,
        requestCount: 0,

        sync: function sync(method, model, options) {
            var requestNum = ++this.requestCount;
            if (this.xhr) {
                //since we are interested in results of only the latest query we can abort pending query
                this.xhr.abort();
            }
            options.success = this.getCallback(options.success, requestNum);
            options.error = this.getCallback(options.error, requestNum);
            options.data = this.getCurrentParams();

            this.xhr = Paginator.requestPager.prototype.sync.call(this, method, model, options);
        },

        //it wraps callback in if statement that checks whether it is response to latest query if not we can
        //discard this result
        getCallback: function getCallback(callback, requestNum) {
            var self = this;
            return function (model, resp, options) {
                if (self.requestCount === requestNum) {
                    self.xhr = null;
                    callback(model, resp, options);
                }
            };
        }

    });
});

AJS.namespace('JIRA.Auditing.Records', null, require('jira/auditing/records'));