define('jira/auditing/records-view', ['backbone', 'jquery', 'jira/analytics', 'jira/util/strings', 'jira/auditing/templates'], function (Backbone, $, Analytics, Strings, Templates) {
    'use strict';

    return Backbone.View.extend({
        el: ".auditing-table",
        renderEmptyTable: false,

        events: {
            'click .record-row': 'showDetails'
        },

        templates: {
            records: Templates.records,
            details: Templates.recordDetails,
            associatedItems: Templates.associatedItems
        },

        initialize: function initialize(options) {
            this.renderEmptyTable = options.emptyTable;
            this.render();
        },

        render: function render() {
            var records = this.renderEmptyTable ? {} : this.collection.toJSON();
            var range = this.collection.getLastRefreshRange();

            this.$el.html(this.templates.records({ records: records }));

            // show livestamps
            this.$('.livestamp').livestamp();

            // mark the records added by refresh
            if (!range) {
                return;
            }

            var $rows = this.$(".record-row");
            $rows.filter(function () {
                var id = +$(this).data("id");

                return id >= range.from && id <= range.to;
            }).addClass("auditing-record-added auditing-record-added-animate").delay(500).animate({ backgroundColor: "#ffffff" }, { duration: 1500, complete: function complete() {
                    $rows.removeClass("auditing-record-added-animate").removeAttr("style");
                } });
        },

        showDetails: function showDetails(event) {
            if ('A' === event.target.tagName) {
                return;
            }

            var $target = $(event.currentTarget);
            var $detailsWrapper = $target.next();
            var recordId = parseInt($target.attr("data-id"), 10);
            var record = this.collection.get(recordId).toJSON();

            if ($detailsWrapper.hasClass("hidden")) {

                if (!$detailsWrapper.is('.log-rendered')) {
                    $detailsWrapper.find(".auditing-record-details").html(this.templates.details({ record: record }));
                    $detailsWrapper.addClass('log-rendered');
                }
                $detailsWrapper.removeClass("hidden");
                $target.addClass('active');

                this._sendToggleAnalytics(record, true);
            } else {
                $detailsWrapper.addClass("hidden");
                $target.removeClass('active');

                this._sendToggleAnalytics(record, false);
            }
        },

        _sendToggleAnalytics: function _sendToggleAnalytics(record, show) {
            Analytics.send({
                name: "administration.auditing.details.toggle",
                properties: {
                    categoryId: record.categoryId,
                    categoryIdHash: Strings.hashCode(record.categoryId),
                    id: record.id,
                    show: show
                }
            });
        }
    });
});

AJS.namespace('JIRA.Auditing.RecordsView', null, require('jira/auditing/records-view'));