define('jira/auditing/records-analytics', ['underscore', 'jira/analytics'], function (_, analytics) {
    'use strict';

    return function () {

        return {
            search: false,
            collection: null,
            filters: null,
            previousCount: null,

            init: function init(collection, filters) {
                this.collection = collection;
                this.filters = filters;

                this.previousCount = this.collection.totalRecords;

                this.addEventListeners();

                return this;
            },

            addEventListeners: function addEventListeners() {
                this.collection.on('sync', this.sync, this);

                return this;
            },

            searchStarted: function searchStarted() {
                this.search = true;
            },

            sync: function sync() {
                if (this.search) {
                    var filters = this.filters.getAllValues();
                    var wordRegEx = /[\w\d]+/gi;
                    var data = {
                        recordsReturned: this.collection.totalRecords,
                        previousRecordsReturned: this.previousCount,
                        matchedFields: this.matchedFields(this.collection),
                        filtersUsed: _.keys(filters).join(',')
                    };
                    var count;

                    if (filters.filter) {
                        count = filters.filter.match(wordRegEx);
                        data.wordCount = count ? count.length : 0;
                        data.letterCount = filters.filter.length;
                    }

                    if (filters.date) {
                        data.timeSecondsTo = Math.round((new Date() - new Date(filters.date[0])) / 1000);
                        if (filters.date[0]) {
                            data.timeSecondsFrom = Math.round((new Date() - new Date(filters.date[1])) / 1000);
                        }
                    }

                    analytics.send({
                        name: 'administration.auditing.view.search',
                        data: data
                    });

                    this.previousCount = this.collection.totalRecords;
                    this.search = false;
                }
            },

            matchedFields: function matchedFields(records) {
                var words = this.filters.get('filter').get('value') || '';
                words = words.split(' ');
                var lowerCaseWords = _.map(_.filter(words, function (word) {
                    return word !== "";
                }), function (word) {
                    return word.toLowerCase();
                });
                var matchedFields = {};

                function isMatching(record, key, word, matchedFields2) {
                    if (record[key].toLowerCase().indexOf(word) !== -1) {
                        matchedFields2[key] = null;
                    }
                }

                function fix(word) {
                    if (word === undefined) {
                        return "";
                    } else {
                        return word;
                    }
                }

                records.each(function (record) {
                    //prepare view of record
                    var transformedRecord = {};
                    transformedRecord['summary'] = fix(record.get('summary'));
                    transformedRecord['category'] = fix(record.get('category'));
                    transformedRecord['remoteAddr'] = fix(record.get('remoteAddr'));
                    transformedRecord['author'] = fix(record.get('author')['fullName']) + " " + fix(record.get('author')['username']);
                    transformedRecord['object'] = fix(record.get('objectItem')['objectName']) + " " + fix(record.get('objectItem')['parentName']);
                    transformedRecord['associatedItems'] = _.reduce(record.get('associatedItems'), function (memo, item) {
                        return memo + " " + fix(item["objectName"]) + " " + fix(item["parentName"]);
                    }, "");
                    transformedRecord['changedValues'] = _.reduce(record.get('changedValues'), function (memo, item) {
                        return memo + " " + fix(item["from"]) + " " + fix(item["to"]);
                    }, "");

                    //find matched fields in view
                    (function (record) {
                        _.each(lowerCaseWords, function (word) {
                            _.each(_.keys(record), function (key) {
                                isMatching(record, key, word, matchedFields);
                            });
                        });
                    })(transformedRecord);
                });
                return _.reduce(_.keys(matchedFields), function (memo, field) {
                    return memo + field + ",";
                }, "");
            }
        };
    };
});