/**
 * Single Select Filter View
 *
 * IF you pass a custom template - generates it and extends first select.js-filter-field into SingleSelect
 * ELSE - getnerates a SingleSelect
 *
 * IF you want to have a AJAX enabled SS, specify at least one of the options
 *  restAPIpath [String] - the path to your REST, automagically prefixed with wrmContextPath
 *  restOptions [Object] - any subset of SingleSelect.ajaxOptions
 */

define('jira/auditing/filter/singleselect-view', ['underscore', 'jira/auditing/filter/view', 'jira/auditing/templates', 'jira/ajs/select/single-select', 'wrm/context-path'], function (_, View, Templates, SingleSelect, wrmContextPath) {
    'use strict';

    return View.extend({
        template: Templates.Filters.singleSelect,
        filterComponent: null,

        events: {
            'select': 'handleSelection',
            'unselect': 'handleSelection'
        },

        initialize: function initialize(options) {
            this.restAPIpath = options.restAPIpath;
            this.restOptions = options.restOptions;

            this.addEventListeners();
            return View.prototype.initialize.apply(this, arguments);
        },

        render: function render() {
            View.prototype.render.apply(this, arguments);

            this.createUserPicker();

            return this;
        },

        createUserPicker: function createUserPicker() {
            var $filter = this.$el.is(".js-filter-field") ? this.$el : this.$(".js-filter-field").first();

            if (!$filter.length || $filter.data("aui-ss")) {
                return this;
            }

            this.filterComponent = new SingleSelect({
                element: $filter,
                showDropdownButton: false,

                errorMessage: this.i18n.errorMessage,
                inputText: this.i18n.inputText,

                revertOnInvalid: true,

                content: "mixed",
                suggestionAtTop: true,
                removeDuplicates: true,

                ajaxOptions: this._getAjaxOptions()
            });

            return this;
        },

        addEventListeners: function addEventListeners() {
            this.on('aui-dropdown2-show', this.handleDropdownShow, this).on('aui-dropdown2-hide', this.handleDropdownHide, this);

            return this;
        },

        // ::[ Event Handlers ]::
        handleDropdownShow: function handleDropdownShow() {
            _.defer(function () {
                this.filterComponent.$field.focus();
            }.bind(this));
        },

        handleDropdownHide: function handleDropdownHide() {},

        handleSelection: function handleSelection() {
            var value = this.filterComponent.model.getSelectedValues();

            this.model.set({
                displayText: this._getDisplayText(),
                value: value.length ? value : null
            });
        },

        // --------[ Private ]--------8<--------
        _getAjaxOptions: function _getAjaxOptions(data) {
            var opt;

            if (this.restAPIpath || this.restOptions || data) {
                opt = _.extend({
                    url: wrmContextPath() + this.restAPIpath || '',
                    query: true, // keep going back to the sever for each keystroke
                    minQueryLength: 0
                }, this.restOptions, data);
            }

            return opt;
        },

        _getDisplayText: function _getDisplayText() {
            return _.map(this.filterComponent.model.getDisplayableSelectedDescriptors(), function (descriptor) {
                return descriptor.fieldText();
            });
        }
    });
});