/**
 * DateRange View
 * Generates Date fields extended with Date Pickers.
 */

define('jira/auditing/filter/daterange-view', ['underscore', 'jquery', 'jira/moment', 'jira/auditing/filter/view', 'jira/auditing/templates', 'jira/libs/calendar'], function (_, $, moment, View, Templates, Calendar) {
    'use strict';

    return View.extend({
        template: Templates.Filters.timeFilter,

        consts: {
            ERROR_DELAY: 2000, //ms
            DATE_CALENDAR_DEBOUNCE: 100, //ms
            DATE_MANUAL_PARSE_DEBOUNCE: 500
        },

        events: {
            'click input.date': 'handleClick',
            'input input.date': 'handleInputChange',
            'click .aui-icon': 'handleIconClick'
        },

        initialize: function initialize() {
            this._parseData = _.debounce(this._parseData.bind(this), this.consts.DATE_MANUAL_PARSE_DEBOUNCE);
            this.addEventListeners();
            return View.prototype.initialize.apply(this, arguments);
        },

        render: function render() {
            View.prototype.render.apply(this, arguments);

            this.initComponents();

            this.$dd = this.$root.closest('aui-inline-dialog');

            return this;
        },

        initComponents: function initComponents() {
            var onSelect = _.debounce(this.handleCalendarSelect.bind(this), this.consts.DATE_CALENDAR_DEBOUNCE);
            var onClose = this.handleCalendarClose.bind(this);
            var consts = this.consts;

            this.$('.js-date-picker').each(function () {
                Calendar.setup({
                    inputField: $(this),
                    singleClick: true,
                    cache: true,
                    eventName: 'openCalendar',
                    align: "Bl",
                    firstDay: consts.firstDay,
                    useISO8601WeekNumbers: consts.useISO8601,
                    showsTime: true,
                    ifFormat: consts.dateTimeFormat,
                    daFormat: consts.dateTimeFormat,
                    timeFormat: consts.timeFormat,
                    date: new Date(consts.todayDateStr),
                    todayDate: new Date(consts.todayDateStr),
                    onSelect: onSelect,
                    onClose: onClose
                });
            });

            return this;
        },

        addEventListeners: function addEventListeners() {
            this.model.on('change:dateFrom', this.handleModelChange, this).on('change:dateTo', this.handleModelChange, this).on('error', this.handleModelError, this);
        },

        handleModelError: function handleModelError(err) {
            if ('future' === err.split(':')[0]) {
                if (_.random(1000) > 42) {
                    return;
                } else {
                    err = 'future';
                }
            }
            var $err = $('<div class="error">' + this.i18n.error[err] + '</div>');
            this.$el.append($err);
            _.delay(function () {
                $err.remove();
            }, this.consts.ERROR_DELAY);
        },

        // we want to avoid accidental dropdown close
        handleClick: function handleClick(e) {
            this.openCalendar($(e.target));
        },

        handleModelChange: function handleModelChange(model) {
            _.each(model.changed, function (value, key) {
                this.$('[name=' + key + ']').val(this._formatDate(value)).trigger('input');
            }.bind(this));

            this.setTriggerDescription();
        },

        handleIconClick: function handleIconClick(e) {
            var $el = $(e.target);
            var $input = $el.prev('input.date');
            if ($el.hasClass('clear-field')) {
                this._setModelData($input.attr('name'));
            } else {
                this.openCalendar($input);
            }
        },

        // may be triggered
        handleInputChange: function handleInputChange(e) {
            var $el = $(e.target);
            var value = $el.val();
            var empty = !value;

            $el.toggleClass('with-value', !empty);
            $el.next('.aui-icon').toggleClass('noloading clear-field aui-iconfont-remove', !empty).toggleClass('aui-iconfont-calendar', empty);

            this._parseData($el.attr('name'), value);
        },

        // calendar selection will trigger just a value change on model which is then reflected in view
        handleCalendarSelect: function handleCalendarSelect(cal) {
            var p = cal.params;
            var update = cal.dateClicked || p.electric;

            if (update && p.inputField) {
                this._setModelData(p.inputField.name, cal.date.getTime());
            }
            if (update && cal.dateClicked) {
                cal.callCloseHandler();
            }
        },

        handleCalendarClose: function handleCalendarClose(cal) {
            cal.hide();
            _.defer(function () {
                this.$dd.removeAttr('persistent');
            }.bind(this));
        },

        setTriggerDescription: function setTriggerDescription() {
            var dFrom = this.model.get('dateFrom');
            var dTo = this.model.get('dateTo');

            if (dTo) {
                this.model.set('displayText', this.i18n.to(this._formatDate(dFrom), this._formatDate(dTo)));
            } else {
                this.model.unset('displayText');
            }
        },

        openCalendar: function openCalendar($el) {
            this.$dd.attr('persistent', true);
            $el.trigger('openCalendar');
        },
        // --------[ Private ]--------8<--------
        _parseData: function _parseData(name, value) {
            var d = moment.splitDate(value, this.consts.dateTimeFormat);

            // @TODO: Date.parseDate -> moment.parseDate (avoid using custom methods on proto)
            if (d.parts.length > 2 && -1 !== d.month && !isNaN(d.day)) {
                value = Date.parseDate(value, this.consts.dateTimeFormat);

                if (d.parts.length < 5) {
                    switch (name) {
                        case 'dateFrom':
                            value.setHours(0, 0, 0, 0);
                            break;
                        case 'dateTo':
                            value.setHours(23, 59, 59, 999);
                            break;
                    }
                }

                this._setModelData(name, value);
            }
        },

        _formatDate: function _formatDate(date) {
            return date ? new Date(date).print(this.consts.dateTimeFormat) : '';
        },
        _setModelData: function _setModelData(name, value) {
            this.model.setDate(name, value);
        }
    });
});