require([
    "jira/api/projects/sidebar",
    "jira/util/formatter",
    "jira/loading/loading",
    "jira/ajs/ajax/smart-ajax",
    "jira/message",
    "wrm/data",
    "underscore",
    "jquery"
], function(
    SidebarAPI,
    formatter,
    loadingIndicator,
    SmartAjax,
    Messages,
    wrmData,
    _,
    $
) {
    "use strict";

    var loading = false;
    var initializeIssueCollector = downloadResources;

    function downloadResources() {
        if (loading) {
            return;
        }

        loadingStarts();
        // once all the issue collector resources are downloaded and evaluated, there is a call to window.ATL_JQ_PAGE_PROPS.triggerFunction
        window.ATL_JQ_PAGE_PROPS =  {
            "triggerFunction": function(showIssueCollectorDialog) {
                initializeIssueCollector = showIssueCollectorDialog;
                // The downloaded issue collector resources call this method and then create a DOM node to use as the container for the dialog.
                // We need to defer this call so the container is already on the page before showing the dialog.
                _.defer(function() {
                    loadingEnds(true);
                    showIssueCollectorDialog();
                });
            }
        };

        $.ajax({
            url: "https://jira.atlassian.com/s/d41d8cd98f00b204e9800998ecf8427e/en_UK-3tj2he/6322/131/1.4.11/_/download/batch/com.atlassian.jira.collector.plugin.jira-issue-collector-plugin:issuecollector-embededjs/com.atlassian.jira.collector.plugin.jira-issue-collector-plugin:issuecollector-embededjs.js?collectorId=55320d27",
            type: "get",
            cache: true,
            timeout: 10000, // timeout before error function will be called - this is needed for errors like bad gateway where jQuery will not reject the promise
            dataType: "script"
        })
            .fail(function(xhr) {
                loadingEnds();
                Messages.showErrorMsg(SmartAjax.buildSimpleErrorContent(xhr), {
                    closeable: true
                });
            });
    }

    function loadingStarts() {
        loading = true;
        AJS.dim();
        loadingIndicator.showLoadingIndicator();
    }

    function loadingEnds(success) {
        loading = false;
        // remove transition from the dim so there is no "blink" when issue collector gets displayed
        // do not do it on failure, so the layer is disabled smoothly then
        if (success) {
            AJS.dim.$dim.css("transition", "none");
        }
        AJS.undim();

        //Some pages (I'm looking at you, IssueNav) provide a different implementation of AJS.undim()
        //that removes AJS.dim.$dim, so we have to check for that special case here.
        if (success && AJS.dim.$dim) {
            AJS.dim.$dim.css("transition", "");
        }
        loadingIndicator.hideLoadingIndicator();
    }

    $(function() {
        SidebarAPI.getSidebar().done(function(Sidebar) {
            function showTipsy(trigger) {
                var $trigger = $(trigger);
                $trigger.tipsy(tipsyOpts).tipsy('show');
                var $tip = $trigger.data('tipsy') && $trigger.data('tipsy').$tip;
                // Remove "opacity" inline style from Tipsy to allow the our own styles and transitions to be applied
                $tip.css({'opacity': ''}).addClass('tooltip-shown');
            }

            function hideTipsy(trigger, andDestroy) {
                var $trigger = $(trigger);
                var $tip = $trigger.data('tipsy') && $trigger.data('tipsy').$tip;
                if ($tip) {
                    $tip.on('transitionend', function () {
                        $trigger.tipsy('hide');
                    });
                    $tip.removeClass('tooltip-shown');
                    if (andDestroy) {
                        $tip.remove();
                        $trigger.removeData('tipsy');
                    }
                }
            }

            var feedbackCollectorData = wrmData.claim("com.atlassian.jira.jira-projects-plugin:feedback-issue-collector.data");
            if (feedbackCollectorData && feedbackCollectorData.isFeedbackButtonEnabled) {
                var AUISidebar = Sidebar.getAUISidebar();
                var feedbackGroup = Sidebar.getGroup("jira-projects-ancillary-group");

                if (feedbackGroup === undefined) {
                    return;
                }

                var feedbackItem = feedbackGroup.getItem("com.atlassian.jira.jira-projects-plugin:sidebar-feedback-link") || feedbackGroup.getItem("com.pyxis.greenhopper.jira:sidebar-feedback-link");

                var $feedbackLink = feedbackItem.ui.link;

                // Setting the tooltip value here until the NavigationItem can support custom tooltips / descriptions.
                $feedbackLink.data("custom-tooltip", formatter.I18n.getText("project.sidebar.feedback.tooltip"));

                var tipsyOpts = {
                    trigger: 'manual',
                    gravity: 'w',
                    className: 'aui-sidebar-section-tooltip',
                    title: function () {
                        return $(this).data("custom-tooltip");
                    }
                };

                AUISidebar.on('expand-end collapse-end', function () {
                    hideTipsy($feedbackLink, true);
                });

                $feedbackLink.on('mouseenter focus', function (e) {
                    showTipsy(e.currentTarget); // using currentTarget to avoid accidentally tipsy-ing child <span>s.
                });

                $feedbackLink.on('click blur mouseleave', function (e) {
                    hideTipsy(e.currentTarget); // using currentTarget to avoid accidentally tipsy-ing child <span>s.
                });

                feedbackItem.on("before:select", function(e) {
                    e.preventDefault();
                });
            }
        });
    });

    $(document).on("click", ".jira-projects-feedback-collector", function(e) {
        e.preventDefault();
        initializeIssueCollector();
    });

});
