define("jira/projects/components/pinnable-navigator/views/list-view/list-view", ["jira/projects/libs/marionette", "underscore", "jquery"], function(Marionette, _, jQuery) {
    "use strict";
    var MAXIMUM_COUNT_NUMBER = 99999;

    return Marionette.ItemView.extend({
        tooltipTimer: null,
        ui: {
            item: "li",
            link: ".js-item-link.isSelectable",
            dropdownTwoHide: "aui-dropdown2-hide",
            itemSections: ".js-items-section",
            removeButton: ".js-remove-button"
        },

        events: {
            "click @ui.link": function(e) {
                this._hideTooltips();
                var item = jQuery(e.currentTarget).data("item-id");
                this._processNavItem(item, e);
            },
            "mouseenter @ui.item": function(ev) {
                this._hideTooltips();
                this.tooltipTimer = setTimeout(function() {
                    jQuery(ev.currentTarget).tipsy("show");
                }, this.tooltipTimeout);
            },
            "mouseleave @ui.item": "_hideTooltips",
            "@ui.dropdownTwoHide": "_hideTooltips",
            "click @ui.removeButton": function(e) {
                e.preventDefault();

                var $deletingItem = this.$(e.currentTarget).siblings(".js-item-link");
                $deletingItem.addClass("js-deleting");

                var $modifiedSection = $deletingItem.closest(".js-items-section");

                this._saveModifications([$deletingItem.data("item-id")], $modifiedSection)
                    .fail(function () {
                        $deletingItem.removeClass("js-deleting");
                    });

            }
        },

        initialize: function () {
            this.tooltipTimeout = this.options.tooltipTimeout || 600;

            if (this.options.searchView) {
                this.listenTo(this.options.searchView, {
                    "doQuery": this.render
                });
            }
        },

        _hideTooltips: function () {
            clearTimeout(this.tooltipTimer);
            jQuery(".tipsy").remove();
        },

        _processNavItem: function(itemId, e){
            this.model.selectItem(itemId);

            var item = this.model.get("selectedItem");
            var event = this.triggerPreventable("itemSelected", {
                item: item
            });
            if (event.isPrevented) {
                e.preventDefault();
            }
        },

        serializeData: function () {
            var selectedItem = this.model.get("selectedItem");

            var searchValue;
            if (this.options.searchView) {
                searchValue = this.options.searchView.getQuery();
            } else {
                searchValue = "";
            }

            var sections = this.model.getAllItemsWithName(searchValue);
            _.forEach(sections, function (section) {
                _.forEach(section.items, function (item) {
                    if (item.count !== undefined && item.count !== "") {
                        if (item.count > MAXIMUM_COUNT_NUMBER) {
                            item.countDisplay = MAXIMUM_COUNT_NUMBER + "+";
                        } else if (item.count >= 0) {
                            item.countDisplay = item.count;
                        }
                    }
                });
            });

            var currentId = null;
            if(selectedItem){
                currentId = selectedItem.id;
            }

            return {
                title: this.options.title,
                sections: sections,
                selectedItem: currentId,
                isManageable: this.model.isManageable(),
                manageLocation: this.options.manageLocation,
                manageText: this.options.manageText,
                emptyText: this.options.emptyText
            };
        },

        onRender: function () {
            this._hideTooltips();
            /*
             * AUI's tooltip shows automatically by default.
             * On that mode, whenever the mouse enters the DOM element, a timeout will be scheduled to show the tooltip.
             * The problem with that is that they don't provide any way for us to cancel the timer once the mouse leaves.
             *
             * To solve this problem we are going to use AUI's tooltip in manual mode and we are going to be
             * in control of scheduling the tooltips and canceling them when the mouse enters/leaves the different options.
             */
            jQuery(this.ui.item).tooltip({gravity: "w", trigger: "manual"});

            this._makeItemsSortable();
        },

        _makeItemsSortable: function () {
            this.$(".js-sortable-items").sortable({
                axis: "y",
                placeholder: "item-holder-highlight",
                sort: function(event, ui) {
                    ui.placeholder.html(ui.helper.html());
                },
                update: function (event, ui) {
                    var $modifiedSection = ui.item.closest(".js-items-section");
                    this._saveModifications([], $modifiedSection);
                }.bind(this)
            });
        },

        _saveModifications: function (deletedItems, $modifiedSection) {
            // Get the remained items list after deleting some items
            var itemOrder = _.map($modifiedSection.find(".js-item-link:not(.js-deleting)"), function (item) {
                return jQuery(item).data("item-id");
            });

            var modifications = [
                {
                    id: $modifiedSection.data("id"),
                    deleted: deletedItems,
                    order: itemOrder,
                    inline: true
                }
            ];

            return this.model.save(modifications);
        }
    });
});
