/*
* This test file contains tests for Pinned.js and ListView.js.
* Listview.js just has lists, while Pinned.js has both lists and buttons. That's why Pinned.js will have all tests
* of Listview.js. There are some tests which need lists and buttons, so if I put the tests of lists in 2 files, there
* will be many duplicated tests
*/

AJS.test.require("com.atlassian.jira.jira-projects-plugin:pinnablenavigator-test", function() {
    "use strict";

    var Deferred = require("jira/jquery/deferred");
    var jQuery = require("jquery");

    var Items = require("jira/projects/components/pinnable-navigator/entities/items");
    var PinnedView = require("jira/projects/components/pinnable-navigator/views/navigator-view/pinned-view");
    var Search = require("jira/projects/components/pinnable-navigator/views/search/search");
    var DarkFeatureModule = require("jira/projects/components/pinnable-navigator/dark-feature");

    var ITEM_1 = { id: "item1", label: "Item 1", description: "Item 1" };
    var ITEM_2 = { id: "item2", label: "Item 2", description: "Item 2", link: "example.com"};
    var ITEM_4 = { id: "item4", label: "Item 4", description: "Item 4", count: 5 };
    var LONGLIST = [ITEM_1,ITEM_4,ITEM_1,ITEM_4,ITEM_1,ITEM_4,ITEM_1,ITEM_4,ITEM_4,ITEM_4];

    var TIMEOUT = 500;

    var ID = "id";

    module("jira/projects/components/pinnable-navigator/views/navigator-view/pinned-view", {
        setup: function() {
            jQuery('#qunit-fixture').append("<span class='tipsy'/>");

            this.$el = jQuery("<div/>");

            jQuery('#qunit-fixture').append(this.$el);

            this.clock = sinon.useFakeTimers();

            this.model = new Items({itemGroups: [{items: [ITEM_4]}]});
            this.view = new PinnedView({
                el: this.$el,
                model: this.model,
                id: ID,
                tooltipTimeout: TIMEOUT,
                title: "Title",
                manageText: "Manage",
                emptyText: "Nothing here",
                searchView: new Search({
                    searchPlaceHolder: "Search...",
                    el: ".js-subnav-search-block",
                    model: this.model
                })
            });
        },
        getViewWithDeleteOptionOnly: function(){
            var model = new Items({itemGroups: [{allowDelete: true, items: [ITEM_4]}]});
            return new PinnedView({
                el: this.$el,
                model: model,
                id: ID,
                tooltipTimeout: TIMEOUT,
                title: "Title",
                manageText: "Manage",
                emptyText: "Nothing here"
            });
        },
        getViewWithReorderOptionOnly: function(){
            var model = new Items({itemGroups: [{allowReorder: true, items: [ITEM_4]}]});
            return new PinnedView({
                el: this.$el,
                model: model,
                id: ID,
                tooltipTimeout: TIMEOUT,
                title: "Title",
                manageText: "Manage",
                emptyText: "Nothing here"
            });
        },
        getViewWithoutAnyOptions: function(){
            var model = new Items({itemGroups: [{items: [ITEM_4]}]});
            return new PinnedView({
                el: this.$el,
                model: model,
                id: ID,
                tooltipTimeout: TIMEOUT,
                title: "Title",
                manageText: "Manage",
                emptyText: "Nothing here"
            });
        },
        teardown: function() {
            if (this.clearTimeoutSpy) {
                this.clearTimeoutSpy.restore();
            }
            this.clock.restore();
        },

        assertScheduledTooltipsAreCanceled: function(timeoutId) {
            sinon.assert.calledOnce(this.clearTimeoutSpy);
            sinon.assert.calledWith(this.clearTimeoutSpy, timeoutId);
        }
    });

    test("Clicking on an item triggers the 'itemSelected' event", function() {
        this.sandbox.useFakeTimers();
        this.view.render();
        this.view.putListintoView();

        var onClick = this.spy();
        this.view.listenTo(this.view, "itemSelected", onClick);

        this.view.$(".js-item-link").click();
        sinon.assert.calledOnce(onClick);
    });

    test("Clicking on an item removes the tooltips", function() {
        this.sandbox.useFakeTimers();
        this.view.render();
        this.view.putListintoView();

        this.view.$el.find("a").click();

        equal(jQuery(".tipsy").length, 0);
    });

    test("Clicking on an item prevents scheduled tooltips to be displayed", function() {
        this.view.render();
        this.view.putListintoView();
        this.view.listItemsView.tooltipTimer = "timerId";

        this.clearTimeoutSpy = sinon.spy(window, "clearTimeout");
        this.view.$el.find("a").click();

        this.assertScheduledTooltipsAreCanceled("timerId");
    });

    test("Mouse leaving an li element removes the tooltips", function() {
        this.sandbox.useFakeTimers();
        this.view.render();
        this.view.putListintoView();

        this.view.$el.find("li").mouseleave();

        equal(jQuery(".tipsy").length, 0);
    });

    test("Mouse leaving an li element prevents scheduled tooltips to be displayed", function() {
        this.view.render();
        this.view.putListintoView();
        this.view.listItemsView.tooltipTimer = "timerId";

        this.clearTimeoutSpy = sinon.spy(window, "clearTimeout");
        this.view.$el.find("li").mouseleave();

        this.assertScheduledTooltipsAreCanceled("timerId");
    });

    test("Mouse entering a li element removes previous tooltips that were shown", function() {
        this.sandbox.useFakeTimers();
        this.view.render();
        this.view.putListintoView();

        this.view.$el.find("li").mouseenter();

        equal(jQuery(".tipsy").length, 0);
    });

    test("Mouse entering a li element cancels previous tooltips that were scheduled to be shown", function() {
        this.view.render();
        this.view.putListintoView();
        this.view.listItemsView.tooltipTimer = "timerId";

        this.clearTimeoutSpy = sinon.spy(window, "clearTimeout");
        this.view.$el.find("li").mouseenter();

        this.assertScheduledTooltipsAreCanceled("timerId");
    });

    test("Mouse entering an li element shows a tooltip", function() {
        var clock = this.sandbox.useFakeTimers();

        this.view.render();
        this.view.putListintoView();

        jQuery('#qunit-fixture').append(this.view.$el);

        this.view.$("li").mouseenter();

        clock.tick(TIMEOUT);

        equal(jQuery(".tipsy").length, 1);
        this.view.$el.remove();
    });

    test("Hiding the view removes the tooltips", function() {
        this.sandbox.useFakeTimers();
        this.view.render();
        this.view.putListintoView();

        this.view.$el.trigger("aui-dropdown2-hide");

        equal(jQuery(".tipsy").length, 0);
    });

    test("Rendering the view removes the tooltips", function() {
        this.sandbox.useFakeTimers();
        this.view.render();
        this.view.putListintoView();

        equal(jQuery(".tipsy").length, 0);
    });

    test("Rendering the view prevents scheduled tooltips to be displayed", function() {
        this.view.render();
        this.view.putListintoView();
        this.view.listItemsView.tooltipTimer = "timerId";

        this.clearTimeoutSpy = sinon.spy(window, "clearTimeout");
        this.view.render();
        this.view.putListintoView();

        this.assertScheduledTooltipsAreCanceled("timerId");
    });

    test("Uses the given id as a suffix when rendering the template", function() {
        this.sandbox.useFakeTimers();

        this.view.render();
        this.view.putListintoView();

        equal(jQuery("#pinnednav-opts-" + ID).length, 1);
    });

    test("Clicking on X icon button will fire 'navigatorUnPinned' event", function() {
        var onClick = this.spy();
        this.view.listenTo(this.view, "navigatorUnPinned", onClick);

        this.view.render();
        this.view.putListintoView();
        this.view.$(".js-unpin-button").click();

        sinon.assert.calledOnce(onClick);
    });

    test("The count number will be appeared at the right of the item text if the item has count attribute", function() {
        this.view.render();
        this.view.putListintoView();

        equal(this.view.$el.find(".navigator-item-count-number").length, 1);
        equal(this.view.$el.find(".navigator-item-count-number").text(), "5");
    });

    test("The count number won't be appeared at the right of the item text if the item hasn't count attribute", function() {
        this.model.set("itemGroups", [{items: [ITEM_1]}]);

        this.view.render();
        this.view.putListintoView();

        equal(this.view.$el.find(".navigator-item-count-number").length, 0);
    });

    test("Showing emptyText when there is nothing in model", function() {
        this.model.set("itemGroups", []);
        this.view.render();
        this.view.putListintoView();
        equal(this.view.$('.empty-group').text(), 'Nothing here');
    });

    test("Button item groups", function () {
        this.view.model.set("itemGroups", [{type: "buttons", items: [ITEM_1]}]);
        this.view.render();
        this.view.putListintoView();
        var $buttons = this.view.$(".button-list .aui-button");
        equal($buttons.text(), "Item 1");
    });

    test("Button item groups (New queue or New report) which have 'isSelectable' class still trigger itemselected", function () {
        this.model.set("itemGroups", [{type: "buttons", items: [ITEM_1]}]);
        this.view.render();
        this.view.putListintoView();
        var onClick = this.spy();
        this.view.listenTo(this.view, "itemSelected", onClick);
        this.view.$(".button-list .isSelectable").click();
        sinon.assert.calledOnce(onClick);
    });

    test("Button item groups which don't have 'isSelectable' class don't trigger itemselected", function () {
        var view = this.getViewWithDeleteOptionOnly();
        view.render();
        var onClick = this.spy();
        view.listenTo(this.view, "itemSelected", onClick);
        view.$(".button-list a:not('.isSelectable')").click();

        sinon.assert.callCount(onClick, 0);
    });

    test("Link replaces button if link attribute specified", function () {
        this.model.set("itemGroups", [{type: "buttons", items: [ITEM_1, ITEM_2]}]);
        this.view.render();
        this.view.putListintoView();
        equal(this.view.$(".button-list a.aui-button").size(), 1);
        equal(this.view.$(".button-list button.aui-button").size(), 1);
    });

    test("Prevents default on click, if itemSelected event is prevented", function () {
        this.sandbox.useFakeTimers();

        this.view.listenTo(this.view, "itemSelected", function (e) {
            e.preventDefault();
        });

        this.view.render();
        this.view.putListintoView();

        var e = jQuery.Event( "click" );
        this.view.$(".js-item-link").trigger(e);
        ok(e.isDefaultPrevented(), "expectedDefault to be prevented");

    });

    test("Does NOT prevented default on click, if itemSelected event is NOT prevented", function () {
        this.sandbox.useFakeTimers();

        this.view.render();
        this.view.putListintoView();

        var e = jQuery.Event("click");
        this.view.$(".js-item-link").trigger(e);
        ok(!e.isDefaultPrevented(), "expectedDefault to be prevented");

    });

    test("The search block will be appeared when there are more than ten items", function () {
        this.stub(DarkFeatureModule, 'isSearchEnabled').returns(true);

        this.view.render();
        this.view.renderSearch();
        this.view.putListintoView();

        ok(!this.view.$(".search-form").length, "Expect there is no search block");

        this.view.model.set("itemGroups", [{items:[ITEM_4], id: "group1"}, {items:LONGLIST, id: "group2"}]);
        this.view.renderSearch();
        this.view.renderList();

        ok(this.view.$(".search-form").length, "Expect there is search block");

        this.view.$(".js-subnav-filter-text").val("item 1").trigger("keydown");

        this.clock.tick(10);
        ok(this.view.$(".js-item-link").length === 4, "Expect there are four items (Item 1) are shown");

        ok(!this.view.$(".js-clear-query").hasClass("hidden"), "Expect there is a clear icon");

        this.view.$(".js-clear-query").click();

        this.clock.tick(10);
        ok(this.view.$(".js-subnav-filter-text").val() === "", "Expect there is empty text after clicking on clear icon");
        ok(this.view.$(".js-item-link").length === 11, "Expect there are full list");
    });

    test("Showing sort icons only for items in sections which have allowReorder: true", function () {
        var view;

        view = this.getViewWithDeleteOptionOnly();
        view.render();
        view.putListintoView();
        equal(view.$(".js-sortable-items").length, 0);

        view = this.getViewWithReorderOptionOnly();
        view.render();
        view.putListintoView();
        equal(view.$(".js-sortable-items").length, 1);

        view = this.getViewWithoutAnyOptions();
        view.render();
        view.putListintoView();
        equal(view.$(".js-sortable-items").length, 0);
    });

    test("Delete buttons are present in the DOM only when allowDelete: true", function() {
        var view;

        view = this.getViewWithDeleteOptionOnly();
        view.render();
        view.putListintoView();
        equal(view.$(".js-remove-button").length, 1);

        view = this.getViewWithReorderOptionOnly();
        view.render();
        view.putListintoView();
        equal(view.$(".js-remove-button").length, 0);

        view = this.getViewWithoutAnyOptions();
        view.render();
        view.putListintoView();
        equal(view.$(".js-remove-button").length, 0);
    });

    test("Clicking Delete button will delete item on view", function() {
        var view = this.getViewWithDeleteOptionOnly();
        view.render();
        view.putListintoView();
        // mock the save method
        view.model.save = function () {
            var deferred = new Deferred();
            deferred.resolve();
            return deferred;
        };

        var firstItem = view.$(".js-item-link").eq(0);

        firstItem.siblings(".js-remove-button").click();

        ok(firstItem.hasClass("js-deleting"));
    });
});
